// vim: set sts=2 sw=2 et:
// encoding: utf-8
//
// Copyleft 2011 RIME Developers
// License: GPLv3
//
// 2011-12-01 GONG Chen <chen.sst@gmail.com>
//
#include <boost/algorithm/string.hpp>
#include <boost/bind.hpp>
#include <boost/date_time/posix_time/posix_time_types.hpp>
#include <rime/deployer.h>

namespace rime {

int CompareVersionString(const std::string &x, const std::string &y) {
  if (x.empty() && y.empty()) return 0;
  if (x.empty()) return -1;
  if (y.empty()) return 1;
  std::vector<std::string> xx, yy;
  boost::split(xx, x, boost::is_any_of("."));
  boost::split(yy, y, boost::is_any_of("."));
  size_t i = 0;
  for (; i < xx.size() && i < yy.size(); ++i) {
    int dx = atoi(xx[i].c_str());
    int dy = atoi(yy[i].c_str());
    if (dx != dy) return dx - dy;
    int c = xx[i].compare(yy[i]);
    if (c != 0) return c;
  }
  if (i < xx.size()) return 1;
  if (i < yy.size()) return -1;
  return 0;
}

void Deployer::ScheduleTask(const shared_ptr<DeploymentTask>& task) {
  boost::lock_guard<boost::mutex> lock(mutex_);
  pending_tasks_.push(task);
}

shared_ptr<DeploymentTask> Deployer::NextTask() {
  boost::lock_guard<boost::mutex> lock(mutex_);
  shared_ptr<DeploymentTask> result;
  if (!pending_tasks_.empty()) {
    result = pending_tasks_.front();
    pending_tasks_.pop();
  }
  // there is still chance that a task is added by another thread
  // right after this call... careful.
  return result;
}

bool Deployer::Run() {
  LOG(INFO) << "running deployment tasks:";
  shared_ptr<DeploymentTask> task;
  int success = 0;
  int failure = 0;
  while ((task = NextTask())) {
    if (task->Run(this))
      ++success;
    else
      ++failure;
    boost::this_thread::interruption_point();
  }
  LOG(INFO) << success + failure << " tasks ran: "
            << success << " success, " << failure << " failure.";
  return failure == 0;
}

bool Deployer::StartMaintenance() {
  if (IsMaintenancing()) {
    LOG(WARNING) << "a maintenance thread is already running.";
    return false;
  }
  if (pending_tasks_.empty()) {
    return false;
  }
  LOG(INFO) << "starting maintenance thread for "
            << pending_tasks_.size() << " tasks.";
  boost::thread t(boost::bind(&Deployer::Run, this));
  maintenance_thread_.swap(t);
  return maintenance_thread_.joinable();
}

bool Deployer::IsMaintenancing() {
  if (!maintenance_thread_.joinable())
    return false;
  return !maintenance_thread_.timed_join(boost::posix_time::milliseconds(0));
}

void Deployer::JoinMaintenanceThread() {
  maintenance_thread_.join();
}

}  // namespace rime
