/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pxmchem-monomer.h"
#include "libpolyxmass-plugin.h"
#include "pxmchem-formula.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmMonomer *
pxmchem_monomer_new (void)
{
  PxmMonomer *monomer = g_malloc0 (sizeof (PxmMonomer));
  
  monomer->propGPA = g_ptr_array_new ();
  
  return monomer;
}


PxmMonomer *
pxmchem_monomer_new_by_code (gchar *code, GPtrArray *GPA)
{
  PxmMonomer *monomer = NULL;
  gint iter = 0;
  

  g_assert (GPA != NULL 
	    && code != NULL && strlen (code) > 0);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      monomer = g_ptr_array_index (GPA, iter);
            
      if (0 == strcmp (monomer->code, code))
	return pxmchem_monomer_dup (monomer, PXM_MONOMER_DUP_DEEP_NO);
    }
  
  return NULL;
}


PxmMonomer *
pxmchem_monomer_dup (const PxmMonomer *monomer, gint how_dup)
{
  PxmMonomer *monomer_new = NULL;
  
  PxmMonomerDup how = (PxmMonomerDup) how_dup;
  

  g_assert (monomer != NULL);
  
  /* Do not allocate prop array, we'll copy it later if 'how_dup'
     requires it. */
  monomer_new = g_malloc0 (sizeof (PxmMonomer)); 
  
  /* The template monomer is assumed to be well formed,
     which is without NULL members!
  */
  g_assert (monomer->name != NULL);
  monomer_new->name = g_strdup (monomer->name);
  
  g_assert (monomer->code != NULL);
  monomer_new->code = g_strdup (monomer->code);
  
  g_assert (monomer->formula != NULL);
  monomer_new->formula = g_strdup (monomer->formula);
  
  g_assert (monomer->propGPA != NULL);
  
  if (how == PXM_MONOMER_DUP_DEEP_YES)
    {
      monomer_new->propGPA =
	libpolyxmass_prop_GPA_dup (monomer->propGPA, how);
    }
  else
    {
      /* All we do is allocate the array as if we had used 
	 pxmchem_monomer_new ().
      */
      monomer_new->propGPA = g_ptr_array_new ();
    }

  return monomer_new;
}

PxmProp *
pxmchem_monomer_prop_dup (const PxmProp *prop, gint how_dup)
{
  PxmProp *prop_new = NULL;
  PxmMonomer *monomer_new = NULL;
  
  
  g_assert (prop != NULL);
  

  prop_new = libpolyxmass_prop_new ();
  libpolyxmass_prop_set_name (prop_new, prop->name);

  monomer_new = pxmchem_monomer_dup ((PxmMonomer *) prop->data, how_dup);
  prop_new->data = (gpointer) monomer_new;
  
  /* And now set the pointer to the prop's housekeeping functions:
   */
  g_assert (prop->custom_dup != NULL);
  prop_new->custom_dup = prop->custom_dup ;

  g_assert (prop->custom_cmp != NULL);
  prop_new->custom_cmp = prop->custom_cmp;

  g_assert (prop->custom_free != NULL);
  prop_new->custom_free = prop->custom_free;
  
  return prop_new;
}



gboolean
pxmchem_monomer_set_name (PxmMonomer *monomer, gchar *name)
{
  g_assert (monomer != NULL && name != NULL);
  
  /* The member data may be NULL, as this function can be called
     right after pxmchem_monomer_new () which leaves the members
     NULL (except the propGPA which is allocated).
  */
  if (monomer->name != NULL)
    g_free (monomer->name);
  
  monomer->name = g_strdup (name);
  
  return TRUE;
}


gboolean
pxmchem_monomer_set_code (PxmMonomer *monomer, gchar *code)
{
  g_assert (monomer != NULL && code != NULL);
  
  /* First of all check that no character is identical to the one
   * which is globally used to separate monomer codes one from
   * the other while making a delimited string reflecting the polymer
   * sequence (this character is libpolyxmass_globals_delim).
   */
  if (NULL != strchr (code, libpolyxmass_globals_delim))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to set monomer code: '%s'\n"),
	     __FILE__, __LINE__, code);
      
      return FALSE;
    }
  
  /* The member data may be NULL, as this function can be called
     right after pxmchem_monomer_new () which leaves the members
     NULL (except the propGPA which is allocated).
  */
  if  (monomer->code != NULL)
    g_free (monomer->code);
  
  monomer->code = g_strdup (code);
  
  return TRUE;
}


gboolean
pxmchem_monomer_set_formula (PxmMonomer *monomer, gchar *formula)
{
  g_assert (monomer != NULL && formula != NULL);
  
  /* The member data may be NULL, as this function can be called
     right after pxmchem_monomer_new () which leaves the members
     NULL (except the propGPA which is allocated).
  */
  if  (monomer->formula != NULL)
    g_free (monomer->formula);
  
  monomer->formula = g_strdup (formula);

  return TRUE;
}


/* COMPARISON FUNCTIONS
 */

gint
pxmchem_monomer_cmp (PxmMonomer *monomer1, PxmMonomer *monomer2,
		     gint how_cmp)
{
  gint result = 0;
  gint iter = 0;
  gint jter = 0;
  
  PxmMonomerCmp how = (PxmMonomerCmp) how_cmp;
  
  PxmProp *prop1 = NULL;
  PxmProp *prop2 = NULL;
  
  gboolean found_prop = FALSE;

  g_assert (monomer1 != NULL);
  g_assert (monomer2 != NULL);

  g_assert (how & PXM_MONOMER_CMP_NONE
	    || how & PXM_MONOMER_CMP_NAME
	    || how & PXM_MONOMER_CMP_CODE
	    || how & PXM_MONOMER_CMP_FORMULA
	    || how & PXM_MONOMER_CMP_PROP_ARRAY_SUBSET1
	    || how & PXM_MONOMER_CMP_PROP_ARRAY_SUBSET2);
  
  if (how & PXM_MONOMER_CMP_NONE)
    {
      return -1;
    }
  
  if (how & PXM_MONOMER_CMP_NAME)
    {
      result += strcmp (monomer1->name, monomer2->name);
    }
  
  if (how & PXM_MONOMER_CMP_CODE)
    {
      result += strcmp (monomer1->code, monomer2->code);
    }
  
  if (how & PXM_MONOMER_CMP_FORMULA)
    {
      result += strcmp (monomer1->formula, monomer2->formula);
    }
  
  if (how & PXM_MONOMER_CMP_PROP_ARRAY_SUBSET1)
    {
      /* It is admitted that the two monomers are not absolutely
	 identical, only if all the prop objects in monomer1->propGPA
	 are in monomer2->propGPA (ie monomer1 is a subset of
	 monomer2). Thus, a very first easy check is to ensure that
	 the number of objects in monomer1->propGPA is lower or equal to
	 monomer2->propGPA.
      */
      if (monomer1->propGPA->len > monomer2->propGPA->len)
	{
	  return ++result;
	}

      for (iter = 0; iter < monomer1->propGPA->len; iter++)
	{
	  found_prop = FALSE;
	  
	  prop1 = g_ptr_array_index (monomer1->propGPA, iter);
	  g_assert (prop1 != NULL);
	  
	  for (jter = 0; jter < monomer2->propGPA->len; jter++) 
	    {
	      prop2 = g_ptr_array_index (monomer2->propGPA, jter);
	      g_assert (prop2 != NULL);
	      
	      /*
		The comparison here is the most thoroughful it can be.
	      */
	      if (0 == 
		  libpolyxmass_prop_cmp (prop1, prop2,
				     PXM_DEFAULT_PROP_CMP_NAME
				     | PXM_DEFAULT_PROP_CMP_DATA
				     | PXM_DEFAULT_PROP_CMP_DEEP
				     | PXM_DEFAULT_PROP_CMP_DATA_SUBSET1
				     | PXM_DEFAULT_PROP_CMP_DATA_SUBSET2))
		{
		  found_prop = TRUE;
		  
		  break;
		}
	      else
		continue;
	    }
	  
	  if (found_prop == FALSE)
	    {
	      result++;
	      
	      return result;
	    }
	}
    }

   if (how_cmp & PXM_MONOMER_CMP_PROP_ARRAY_SUBSET2)
    {
      /* It is admitted that the two monomers are not absolutely
	 identical, only if all the prop objects in monomer2->propGPA
	 are in monomer1->propGPA (ie monomer2 is a subset of
	 monomer1). Thus, a very first easy check is to ensure that the
	 number of objects in monomer2->propGPA is lower or equal to
	 monomer1->propGPA.
      */
      if (monomer2->propGPA->len > monomer1->propGPA->len)
	{
	  return ++result;
	}

      for (iter = 0; iter < monomer2->propGPA->len; iter++)
	{
	  found_prop = FALSE;
	  
	  prop2 = g_ptr_array_index (monomer2->propGPA, iter);
	  g_assert (prop2 != NULL);
	  
	  for (jter = 0; jter < monomer1->propGPA->len; jter++) 
	    {
	      prop1 = g_ptr_array_index (monomer1->propGPA, jter);
	      g_assert (prop1 != NULL);
	      
	      /*
		The comparison here is the most thoroughful it can be.
	      */
	      if (0 == 
		  libpolyxmass_prop_cmp (prop1, prop2,
				     PXM_DEFAULT_PROP_CMP_NAME
				     | PXM_DEFAULT_PROP_CMP_DATA
				     | PXM_DEFAULT_PROP_CMP_DEEP
				     | PXM_DEFAULT_PROP_CMP_DATA_SUBSET1
				     | PXM_DEFAULT_PROP_CMP_DATA_SUBSET2))
		{
		  found_prop = TRUE;
		  
		  break;
		}
	      else
		continue;
	    }
	  
	  if (found_prop == FALSE)
	    {
	      return ++result;
	    }
	}
    }
  
  return result;
}


gint
pxmchem_monomer_prop_cmp (PxmProp *prop1, PxmProp *prop2, gint how_cmp)
{
  gint result = 0;

  PxmMonomer *monomer1 = NULL;
  PxmMonomer *monomer2 = NULL;
  

  g_assert (prop1 != NULL);
  g_assert (prop2 != NULL);

  monomer1 = (PxmMonomer *) prop1->data;
  monomer2 = (PxmMonomer *) prop2->data;
  
  g_assert (monomer1 != NULL);
  g_assert (monomer2 != NULL);
  
  
  /* Remember that all the prop_cmp functions can be called with a
     number of default parameters:
     
     how_cmp & PXM_DEFAULT_PROP_CMP_NONE
     
     how_cmp & PXM_DEFAULT_PROP_CMP_NAME

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA

     how_cmp & PXM_DEFAULT_PROP_CMP_DEEP

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET1

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET2

  */
  if (how_cmp & PXM_DEFAULT_PROP_CMP_NONE
      || how_cmp & PXM_MONOMER_PROP_CMP_NONE)
    {
      return -1;
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_NAME
      || how_cmp & PXM_MONOMER_PROP_CMP_NAME)
    {
      result += strcmp (prop1->name, prop2->name);
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA
      || how_cmp & PXM_MONOMER_PROP_CMP_DATA)
    {
      result += pxmchem_monomer_cmp (monomer1, monomer2, 
				     PXM_MONOMER_CMP_NAME
				     | PXM_MONOMER_CMP_CODE
				     | PXM_MONOMER_CMP_FORMULA);
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_DEEP
      || how_cmp & PXM_MONOMER_PROP_CMP_DEEP)
    {
      if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET1
	  || how_cmp & PXM_MONOMER_PROP_CMP_PROP_ARRAY_SUBSET1)
	result += pxmchem_monomer_cmp (monomer1, monomer2,
				       PXM_MONOMER_CMP_PROP_ARRAY_SUBSET1);

      if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET2
	  || how_cmp & PXM_MONOMER_PROP_CMP_PROP_ARRAY_SUBSET2)
	result += pxmchem_monomer_cmp (monomer1, monomer2,
				       PXM_MONOMER_CMP_PROP_ARRAY_SUBSET2);
    }

  return result;
}


/* INTEGRITY CHECKING FUNCTIONS
 */
gboolean
pxmchem_monomer_validate (PxmMonomer *monomer, gint codelen,
			  GPtrArray *atom_refGPA, gchar **valid)
{
  GString *gs = NULL;
  
  gchar *help = NULL;
  
  g_assert (monomer != NULL);
  g_assert (atom_refGPA != NULL);


  /* Note that for integrity reasons, *valid MUST BE NULL to ensure 
   * that it is empty.
   */
  g_assert (valid != NULL);
  g_assert (*valid == NULL);


  gs = g_string_new ("");
  
  /* The name: it must be non-NULL and be longer than 0 chars.
   */
  if (monomer->name == NULL)
    {
      g_string_append_printf (gs, 
			 _("monomer has a NULL 'name' member\n"));
    }
  else
    {
      /* Make a copy of the string, so that we can strip it of its
       * spaces and next check its "real" length.
       */
      help = g_strdup (monomer->name);
      help = g_strstrip (help);
      
      if (strlen (help) <= 0)
	{
	  g_string_append_printf (gs, 
			     _("monomer has an invalid 'name' member: '%s'\n"),
			     monomer->name);
	}
      
      g_free (help);
    }
  

  /* The code: it must be non-NULL and 
   * be a valid monomer code (length- and syntax-wise).
   */
  if (monomer->code == NULL)
    {
      g_string_append_printf (gs, 
			 _("monomer has a NULL 'code' member\n"));
    }
  else
    {
      if (FALSE == pxmchem_monomer_check_code_syntax (monomer->code, 
						      codelen))
	g_string_append_printf (gs, 
			   _("monomer has an invalid 'code' member: '%s'\n"),
			   monomer->code);
    }
  /* Further, the code must not have the globally character that is
   * used to separate monomer codes one from the other while making a
   * delimited string reflecting the polymer sequence (this character
   * is libpolyxmass_globals_delim).
   */
  if (NULL != strchr (monomer->code, (gchar) libpolyxmass_globals_delim))
    {
      g_string_append_printf (gs, 
			 _("monomer has an invalid 'code' member: '%s'\n"),
			 monomer->code);
    }
  
  
  /* The formula: it must be non-NULL and be chemically valid.
   */
  if (monomer->formula == NULL)
    {
      g_string_append_printf (gs, 
			 _("monomer has a NULL 'formula' member\n"));
    }
  else
    {
      /* We do not need to strip the string of its spaces, because
       * the formula checking function will do this itself.
       */
      if (FALSE == pxmchem_formula_check (monomer->formula, atom_refGPA))
	{
	  g_string_append_printf (gs, 
			     _("monomer has an invalid 'formula' member: '%s'\n"),
			     monomer->formula);
	}
    }
  
  /* Finally the validation is finished.
   */
  if (strlen (gs->str) > 0)
    {
      /* String is not empty, which is there were errors.
       */
      *valid = gs->str;
      
      g_string_free (gs, FALSE);
      
      return FALSE;
    }

  g_string_free (gs, TRUE);
  
  return TRUE;
}


gboolean
pxmchem_monomer_unique_by_name (PxmMonomer *monomer, 
				GPtrArray *GPA)
{
  g_assert (monomer != NULL);
  g_assert (monomer->name != NULL);
  g_assert (GPA != NULL);
  
  return (pxmchem_monomer_get_index_top_by_name (monomer->name, GPA)
	  == 
	  pxmchem_monomer_get_index_bottom_by_name (monomer->name, GPA));
}


gboolean
pxmchem_monomer_unique_by_code (PxmMonomer *monomer, 
				GPtrArray *GPA)
{
  g_assert (monomer != NULL);
  g_assert (monomer->code != NULL);
  g_assert (GPA != NULL);
  
  return (pxmchem_monomer_get_index_top_by_code (monomer->code, GPA)
	  == 
	  pxmchem_monomer_get_index_bottom_by_code (monomer->code, GPA));
}


gboolean
pxmchem_monomer_check_code_syntax (gchar *code, gint max_chars)
{
  gint iter = 0;
  gint len = 0;
  
  gsize size = 0;

  /* The code must:
   * be at least one char long.
   * start with an uppercase character
   * have a length less than or equal to max_chars
   * all chars subsequent to the first must be lowercase
   */


  g_assert (code != NULL);
  
  size = strlen (code);
  g_assert (size < G_MAXINT);
  len = (gint) size;
  
  if (len > max_chars)
    return FALSE;

  if (len <= 0)
    return FALSE;
  
  for (iter = 0; iter < max_chars; iter++)
    {
      if (iter == 0)
	{
	  if (FALSE == g_ascii_isupper (code [iter]))
	    return FALSE;
	}
      else if (TRUE == g_ascii_isupper (code [iter]))
	return FALSE;
    }

  return TRUE;
}


gchar *
pxmchem_monomer_find_code_in_delimited_codes_string (gchar *codes,
					     gchar *code,
					     gchar delim)
{
  gchar *full = NULL;
  gchar *found = NULL;
  
  g_assert (codes != NULL && code != NULL);
  
  full = g_strdup_printf ("%c%s%c", delim, code, delim);
  
  found = strstr (codes, full);
  
  g_free (full);
  
  return found;
}


/*  LOCATING FUNCTIONS
 */
gint
pxmchem_monomer_get_index_by_name (gchar *name, GPtrArray *GPA)
{
  return pxmchem_monomer_get_index_top_by_name (name, GPA);
}


gint
pxmchem_monomer_get_index_top_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = 0;

  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      monomer = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (monomer->name, name))
	return iter;
    }
  
  return -1;
}


gint
pxmchem_monomer_get_index_bottom_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = 0;

  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  
  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  monomer = g_ptr_array_index (GPA, iter);
	  
	  if (0 == strcmp (monomer->name, name))
	    return iter;
	}
    }
    
  return -1;
}


gint
pxmchem_monomer_get_index_by_code (gchar *code, GPtrArray *GPA)
{
  return pxmchem_monomer_get_index_top_by_code (code, GPA);
}


gint
pxmchem_monomer_get_index_top_by_code (gchar *code, GPtrArray *GPA)
{
  gint iter = 0;

  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      monomer = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (monomer->code, code))
	return iter;
    }
  
  return -1;
}


gint
pxmchem_monomer_get_index_bottom_by_code (gchar *code, GPtrArray *GPA)
{
  gint iter = 0;

  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  
  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  monomer = g_ptr_array_index (GPA, iter);
	  
	  if (0 == strcmp (monomer->code, code))
	    return iter;
	}
    }
    
  return -1;
}


gint
pxmchem_monomer_get_index_by_ptr (PxmMonomer *monomer, GPtrArray *GPA)
{
  gint iter = 0;
  

  g_assert (GPA != NULL && monomer != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    if ((PxmMonomer *) g_ptr_array_index (GPA, iter) == monomer)
      return iter;
      
  return -1;
}


gint
pxmchem_monomer_get_index_by_completing_code (GPtrArray *GPA, 
					      gchar *code,
					      gint *idx)
{
  gint iter = 0;

  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  g_assert (code != NULL);
  
  if (idx != NULL)
    iter = *idx;
  else
    iter = 0;

  while (iter < GPA->len)
    {
      monomer = g_ptr_array_index (GPA, iter);
      
      g_assert (monomer != NULL && monomer->code != NULL);
      
      if (NULL != strstr (monomer->code, code))
	{
	  if (idx != NULL)
	    *idx = iter;

	  return iter;
	}
      
      iter++;
    }
  
  if (idx != NULL)
    *idx = -1;
  
  return -1;
}


PxmMonomer *
pxmchem_monomer_get_ptr_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = 0;

  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      monomer = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (monomer->name, name))
	return monomer;
    }
  
  return NULL;
}


PxmMonomer *
pxmchem_monomer_get_ptr_by_code (gchar *code, GPtrArray *GPA)
{
  gint iter = 0;

  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  g_assert (code != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      monomer = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (monomer->code, code))
	return monomer;
    }
  
  return NULL;
}


gint
pxmchem_monomer_get_ptrs_by_completing_code (gchar *code,
					     GPtrArray *GPA,
					     GPtrArray *fillGPA)
{
  gint iter = 0;
  gint count = 0;
  
  PxmMonomer *monomer = NULL;
  
  
  g_assert (code != NULL && GPA != NULL);

  /* Note that we do not check if strlen (code) > 0 because it 
   * is useful also if code has no letter in it (valid empty string)
   * particularly when this function is called when the user 
   * seeks completions for the empty string. In this case all 
   * the monomer codes do match!
   */

#if 0

  /* Actually, I found that this case does even not need to be 
   * treated apart, since the function call below invokes strstr ()
   * which will match by itself all the monomer codes since any
   * valid string matches the empty string. This is because any valid
   * string ends with a \x0' character, which itself is an empty string.
   */

  /* Treat the special case of an empty code immediately:
   */
  if (strlen (code) == 0)
    {
      for (iter = 0; iter < GPA->len; iter++)
	{
	  monomer = g_ptr_array_index (GPA, iter);
	  
	  if (fillGPA != NULL)
	    g_ptr_array_add (fillGPA, monomer);

	  count++;
	}
      
      return count;
    }
#endif

  while (iter < GPA->len)
    {
      iter = 
	pxmchem_monomer_get_index_by_completing_code (GPA, code, &iter);
      
      if (iter != -1)
	{
	  monomer = g_ptr_array_index (GPA, iter);
	  
	  /* Only append the found monomer pointer if the user passed
	   * a non-NULL pointer for the GPtrArray to store pointers.
	   */
	  if (fillGPA != NULL)
	    g_ptr_array_add (fillGPA, monomer);
	  
	  /* Increment iter by one, so that we are not going to find
	   * the same index next round.
	   */
	  iter++;

	  /* Increment count by one, so that we can return the number
	   * of found items.
	   */
	  count++;
	}
      else
	break;
    }
  
  return count;
}


gchar *
pxmchem_monomer_get_modif_name (PxmMonomer *monomer)
{
  PxmProp *prop = NULL;
  
  g_assert (monomer != NULL);
  
  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				      NULL,
				      NULL,
				      "MODIF",
				      NULL,
				      PXM_UNDEF_PROP_CMP);
  if (NULL != prop)
    {
      /* The monomer effectively has a modification. Remember that a
	 modification is a two-strings PxmProp property, with name
	 "MODIF" and data "Acetylation", for example.
       */
      return g_strdup ((gchar *) prop->data);
    }
  
  return NULL;
}


PxmProp *
pxmchem_monomer_find_prop (PxmMonomer *monomer, 
			   PxmProp *prop,
			   gint how_cmp,
			   gint *idx)
{
  gint iter = 0;
  
  PxmProp *prop_found = NULL;

  
  for (iter = 0; iter < monomer->propGPA->len ; iter++)
    {
      prop_found = g_ptr_array_index (monomer->propGPA, iter);
      g_assert (prop_found != NULL);
      
      if (0 == libpolyxmass_prop_cmp (prop, prop_found,
				  how_cmp))
	{
	  if (idx != NULL)
	    *idx = iter;
	  
	  return prop_found;
	}
    }

  return NULL;
}




/* MASS/ELEMENTAL COMPOSITION ACCOUNTING FUNCTIONS
 */
gboolean
pxmchem_monomer_account_mass_by_name (gchar *name,
				      GPtrArray *monomerGPA,
				      GPtrArray *atom_refGPA,
				      gint times,
				      PxmMasspair *masspair)
{
  /* We have a monomer name, and we want to calculate its masses
   * according to its formula ( ie formula of the monomer ) - Thus we
   * need to search the monomer in the array of monomers pointed to by
   * monomerGPA, get its formula and calculate the masses from the formula.
   */
  gint iter = 0;

  PxmMonomer *monomer = NULL;

  g_assert (monomerGPA != NULL);
  g_assert (atom_refGPA != NULL);

  if (monomerGPA->len <= 0)
    return FALSE;
  
  
  g_assert (name != NULL);
  g_assert (masspair != NULL);

  for (iter = 0; iter < monomerGPA->len; iter++)
    {
      monomer = g_ptr_array_index (monomerGPA, iter);

      if (0 == strcmp (monomer->name, name))
	{
	  /* OK, we have our monomer here, just extract its formula 
	   * and account for it.
	   */
	  if (FALSE == 
	      pxmchem_formula_account_mass (monomer->formula,
					    atom_refGPA, times,
					    masspair))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to account for monomer: '%s'"
		       " with formula: '%s'\n"),
		     __FILE__, __LINE__, 
		     monomer->name, monomer->formula);

	      return FALSE;
	    }

	  /* We now have the masses for the monomer currently parsed.
	   */
	  return TRUE;
	}
    }

  /* If we are here, that means that we did not find a monomer in the 
   * monomerGPA array of monomers that had the same name as the one
   * provided in the function parameters. This an error contidition :
   */
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	 _("%s@%d: failed to find monomer in the reference array: '%s'\n"),
	 __FILE__, __LINE__, name);

  return FALSE;
}


gboolean
pxmchem_monomer_account_mass_by_code (gchar *code,
				      GPtrArray *monomerGPA,
				      GPtrArray *atom_refGPA,
				      gint times,
				      PxmMasspair *masspair)
{
  /* We have a monomer code, and we want to calculate its masses
   * according to its formula ( ie formula of the monomer ) - Thus we
   * need to search the monomer in the array of monomers pointed to by
   * monomerGPA, get its formula and calculate the masses from the formula.
   */
  gint iter = 0;

  PxmMonomer *monomer = NULL;

  g_assert (monomerGPA != NULL);
  g_assert (atom_refGPA != NULL);

  if (monomerGPA->len <= 0)
    return FALSE;
  
  
  g_assert (code != NULL);
  g_assert (masspair != NULL);

  for (iter = 0; iter < monomerGPA->len; iter++)
    {
      monomer = g_ptr_array_index (monomerGPA, iter);

      if (0 == strcmp (monomer->code, code))
	{
	  /* OK, we have our monomer here, just extract its formula 
	   * and account for it.
	   */
	  if (FALSE == 
	      pxmchem_formula_account_mass (monomer->formula,
					    atom_refGPA, times,
					    masspair))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to  account for monomer: '%s'"
		       " with formula: '%s'\n"),
		     __FILE__, __LINE__, 
		     monomer->code, monomer->formula);

	      return FALSE;
	    }

	  /* We now have the masses for the monomer currently parsed.
	   */
	  return TRUE;
	}
    }

  /* If we are here, that means that we did not find a monomer in the 
   * monomerGPA array of monomers that had the same code as the one
   * provided in the function parameters. This an error contidition :
   */
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	 _("%s@%d: failed to find monomer in the reference array: '%s'\n"),
	 __FILE__, __LINE__, code);

  return FALSE;
}


gboolean
pxmchem_monomer_account_mass_for_modif (PxmMonomer *monomer,
					GPtrArray *modifGPA,
					GPtrArray *atom_refGPA,
					gint times,
					PxmMasspair *masspair)
{
  PxmProp *prop = NULL;
  
  g_assert (monomer != NULL);
  g_assert (modifGPA != NULL);
  g_assert (atom_refGPA != NULL);
  g_assert (masspair != NULL);

  
  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				  NULL,
				  NULL,
				  "MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);
  if (NULL != prop)
    {
      /* The monomer effectively has a modification.
       */
      if (FALSE == 
	  pxmchem_modif_account_mass_by_name ((gchar *)prop->data,
					      modifGPA,
					      atom_refGPA,
					      times,
					      masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for monomer modif: '%s'\n"),
		 __FILE__, __LINE__, (gchar *)prop->data);
	  
	  return FALSE;
	}
    }
  
  return TRUE;
}

gboolean
pxmchem_monomer_account_elemcompos_for_modif (PxmMonomer *monomer,
					      GPtrArray *modifGPA,
					      GPtrArray *atom_refGPA,
					      gint times,
					      GPtrArray *acGPA)
{
  PxmProp *prop = NULL;
  
  g_assert (monomer != NULL);
  g_assert (modifGPA != NULL);
  g_assert (atom_refGPA != NULL);
  g_assert (acGPA != NULL);
  
  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				  NULL,
				  NULL,
				  "MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);
  if (NULL != prop)
    {
      /* The monomer effectively has a modification.
       */
      if (FALSE ==
	  pxmchem_modif_account_elemcompos_by_name ((gchar *)prop->data,
						    modifGPA,
						    atom_refGPA,
						    times,
						    acGPA))
	{
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		   _("%s@%d: failed to account for monomer modif: '%s'\n"),
		   __FILE__, __LINE__, (gchar *)prop->data);
	  
	  return FALSE;
	}
    }
  
  return TRUE;
}




/* CHEMICALLY-MODIFYING FUNCTIONS
 */

gboolean
pxmchem_monomer_modify (PxmMonomer *monomer, gchar *modif)
{
  PxmProp *prop = NULL;
  

  g_assert (monomer != NULL);
  g_assert (modif != NULL);
  
  /* It is not possible that the same monomer has two different
     modifications at the same time. That means that if we are to
     modify a monomer, we first must ensure that any previous modif be
     removed first.
  */

  /* Note that the function caller must have made sure that the modif
     passed as parameter is known to the polymer definition
     context. We are not taking any overhead now.
  */

  /* Check if monomer is already bearing a modification. If so remove it
     right away.
  */
  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				  NULL,
				  NULL,
				  "MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);
  if (NULL != prop)
    {
      /* The monomer that is to be modified is already modified, so first
	 remove the existing modification.
      */
      g_assert (TRUE == g_ptr_array_remove (monomer->propGPA, prop));
      
      libpolyxmass_prop_free (prop);
    }
  
  prop = libpolyxmass_prop_both_strings_new ("MODIF", modif);
      
  g_ptr_array_add (monomer->propGPA, prop);
      
  return TRUE;
}

  
/* Returns
   UN_MODIF_SUCCESS or
   UN_MODIF_NO_MATCH or
   UN_MODIF_NO_MODIF or
   UN_MODIF_FAILURE.
*/
gint
pxmchem_monomer_un_modify (PxmMonomer *monomer, gchar *modif)
{
  PxmProp *prop = NULL;
  
  g_assert (monomer != NULL);

  /* Check if monomer is already bearing a modification. If so check
     if we have to actually remove it or not.
  */
  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				  NULL,
				  NULL,
				  "MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);
  if (NULL != prop)
    {
      /* If the modif parameter is non-NULL, then we are asked to
	 remove the modification only if the current modification has
	 the same name as the parameter. If it is NULL, we remove
	 whatever modif we find in the monomer.
      */
      if (modif != NULL)
	{
	  if (0 == strcmp (prop->data, modif))
	    {
	      g_assert (TRUE == g_ptr_array_remove (monomer->propGPA, prop));
	      
	      libpolyxmass_prop_free (prop);
	      
	      return UN_MODIF_SUCCESS;
	    }
	  else
	    return UN_MODIF_NO_MATCH;
	}
      else
	{
	  g_assert (TRUE == g_ptr_array_remove (monomer->propGPA, prop));
	  
	  libpolyxmass_prop_free (prop);
	  
	  return UN_MODIF_SUCCESS;
	}
    }
      
  /* The monomer is not modified!
   */  
  return UN_MODIF_NO_MODIF;
}

  





/* UTILITY FUNCTIONS
 */
gchar *
pxmchem_monomer_make_delimited_codes_string (GPtrArray *GPA,
					 gchar delim)
{
  return pxmchem_monomer_make_delimited_codes_string_by_idces (GPA,
							       delim,
							       -1, -1);
}


gchar *
pxmchem_monomer_make_codes_string (GPtrArray *GPA)
{
  return pxmchem_monomer_make_codes_string_by_idces (GPA, -1, -1);
}


gchar *
pxmchem_monomer_make_delimited_codes_string_by_idces (GPtrArray *GPA,
						      gchar delim,
						      gint start_idx,
						      gint end_idx)
{
  GString *gs = NULL;
  
  gchar *codes = NULL;
  
  gint iter = 0;
  gint start_index = 0;
  gint end_index = 0;
    
  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  
  /* if any index parameter is <=-1, then the whole array should
   * be used to construct the string.
   */
  if (start_idx <= -1 || end_idx <= -1)
    {
      start_index = 0;
      end_index = GPA->len -1;
    }
  else
    {
      start_index = start_idx;
      end_index = end_idx;
    }
  
  /* verification that the newly cooked indices are not out of bonds.
   */
  if (end_index >= GPA->len)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING,
	    _("%s@%d: 'end_index' index is greater than last array item index\n"),
	     __FILE__, __LINE__);

      end_index = GPA->len - 1;
    }
  
  if (start_index >= GPA->len)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING,
	    _("%s@%d: 'start_index' index is greater than last array item index\n"),
	     __FILE__, __LINE__);
      
      start_index = 0;
    }
  
  gs = g_string_new ("");
  
  /* Two cases: the user provides a delimiter character or not.
   */
  if (delim != '\x0')
    {
      gs = g_string_append_c (gs, delim);
      
      for (iter = start_index; iter < (end_index + 1); iter++)
	{
	  monomer = g_ptr_array_index (GPA, iter);
	  
	  g_string_append_printf (gs, "%s%c", monomer->code, delim);
	}
    }
  else
    {
      for (iter = start_index; iter < (end_index + 1); iter++)
	{ 
	  monomer = g_ptr_array_index (GPA, iter);
	  
	  g_string_append_printf (gs, "%s", monomer->code);
	}
    }
  
  codes = gs->str;
  g_string_free (gs, FALSE);
  
  return codes;
}


gchar *
pxmchem_monomer_make_codes_string_by_idces (GPtrArray *GPA,
					    gint start_idx,
					    gint end_idx)
{
  GString *gs = NULL;
  
  gchar *codes = NULL;
  
  gint iter = 0;
  gint start_index = 0;
  gint end_index = 0;
    
  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  
  /* if any index parameter is <=-1, then the whole array should
   * be used to construct the string.
   */
  if (start_idx <= -1 || end_idx <= -1)
    {
      start_index = 0;
      end_index = GPA->len -1;
    }
  else
    {
      start_index = start_idx;
      end_index = end_idx;
    }
  
  /* verification that the newly cooked indices are not out of bonds.
   */
  if (end_index >= GPA->len)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING,
	    _("%s@%d: end index is greater than last array item index\n"),
	     __FILE__, __LINE__);

      end_index = GPA->len - 1;
    }
  
  if (start_index >= GPA->len)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING,
	    _("%s@%d: start index is greater than last array item index\n"),
	     __FILE__, __LINE__);
      
      start_index = 0;
    }
  
  gs = g_string_new ("");
  
  for (iter = start_index; iter < (end_index + 1); iter++)
    { 
      monomer = g_ptr_array_index (GPA, iter);
      
      g_string_append_printf (gs, "%s", monomer->code);
    }
  
  codes = gs->str;
  g_string_free (gs, FALSE);
  
  return codes;
}


gint
pxmchem_monomer_extract_code_from_string (gchar *code, gint codelen,
					  gchar *seq,
					  gint *idx,
					  gchar *err)
{
  gchar *code_new = NULL;
  gint is_lower = 0;
  gint len = 0;
  gint iter = 0;

  gsize size = 0;
  
  /* We get a pointer to a string that represents a polymer sequence
   * containing monomer codes. 
   *
   * A monomer code is :
   * 1. at maximum libpolyxmass_globals_max_monomer_code_len characters 
   * the code not counting the NULL terminating character.
   * 2. the first and the first only character MUST be uppercase.
   *
   * We get a pointer to a fixed length string where we will put the
   * new valid monomer code that we found while iterating in the
   * polymer sequence starting at index "idx" given as parameter.
   * This 'code' parameter MUST be of a length at least equal to 
   * libpolyxmass_globals_max_monomer_code_len and full of 'X' characters.
   *
   * The err parameter points to a 2-bytes string. It can be NULL
   * however. This parameter will be filled with the invalid chars
   * found during the parsing of seq.
   */

  g_assert (seq != NULL);

  g_assert (idx != NULL && *idx >= 0);

  g_assert (code != NULL && strlen (code) <= codelen);
  

  size = strlen (seq);
  g_assert (size < G_MAXINT);
  len = (gint) size;

  if (len <= 0)
    return -1;

  /* Now that we can start, set to '\0' all the slots in the code
   * param passed as parameter.
   */
  memset (code, '\x0', codelen + 1);

  /* Allocate the room for the globally authorized number of 
   * characters in a single code. Note that g_malloc0 set all cells
   * to NULL, so no need to memset them with '\x0'.
   */
  code_new = g_malloc0 (sizeof (gchar) * codelen + 1);

  /* First NULL'ize all the cells of the code string array.
   */
  memset (code_new, '\x0', codelen + 1);

  /* We iterate in the sequence string starting at the index 
   * given as param.
   */
  while (1)
    {
      /* First some basic checks.
       */
      if (iter >= codelen)
	{
	  iter--;
	  break;
	}

      if ((iter + *idx) >= len)
	break;

      if (seq [*idx + iter] == '\x0')
	break;

      /* Check if the currently iterated character is an alpha char,
       * and if it is not we store it in the err (if non-null)
       * variable and return error value -1.
       */
      if (0 == g_ascii_isalpha (seq [*idx + iter]))
	{
	  /* We want to inform the caller that something was wrong
	     (return -1). However, we also want that the caller may
	     take a decision in good knowledge of what goes on. This
	     is why we also provide it with a means to check what's
	     wrong by putting the "erroneous" non-ascii character in
	     'err' (if it was provided in the params). Typically this
	     is useful when parsing a string so that it is pasted in
	     the polymer sequence editor: we want to trap non-alpha
	     chars. But when the string to parse is a cleavage
	     pattern, like "Lys/Pro", we do not want that the parsing
	     be harshly stopped because of the '/' character: in fact,
	     the presence of that '/' is not an error, and it must be
	     detected by the caller function. That is why we copy this
	     char in 'err'.
	  */
	  if (err != NULL)
	    memset (err, seq [*idx + iter], 1);

	  /* Empty the code string.
	   */
	  memset (code, '\x0', codelen + 1);

	  /* Set *idx to the index at which we are, the calling
	     function will be responsible for incrementing it by one
	     unit for the following call if there is one.
	  */
	  *idx = *idx + iter;

	  g_free (code_new);
	  
	  return -1;
	}
      
      is_lower = g_ascii_islower (seq [*idx + iter]);
      
      if (iter == 0)
	{
	  /* We are at the first iteration, if current char is
	   * lowercase, this is a REAL ERROR condition.
	   */
	  if (is_lower)
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		    _("%s@%d: first char of code MUST be UPPERcase\n"),
		     __FILE__, __LINE__);

	      memset (code, '\x0', codelen + 1);
	      
	      /* If we return -1, that means that there is an error. 

	      But that could be the case where the '/' char is found
	      (less serious, and in some cases event acceptable
	      (parsing of cleavage sites), in which case the '/' char
	      is copied in the err string.

	      Here the error is more serious, it is an error in the
	      "syntax" of the string. So we empty the err string, so
	      that the caller can check this: if strlen (err) <= 0,
	      then the error is serious.
	      */

 	      if (err != NULL)
		memset (err, '\x0', 1);
	      
	      g_free (code_new);
	      
	      return -1;
	    }

	  /* We are at first char iteration and current char is UPPERcase:
	   * good.
	   */
	  memset (code_new + iter, seq [*idx + iter], 1);
	}
      else
	{
	  /* We are not at first iteration, so current char MUST
	   * be lowercase or we are initiating another code,
	   * which means we break immediately.
	   */
	  if (is_lower)
	    memset (code_new + iter, seq [*idx + iter], 1);
	  
	  else
	    {
	      /* Decrement iter because this round was for nothing,
	       * since we are overlapping to another monomer code :
	       * go back one step.
	       */
	      iter--;
	      break;
	    }
	}
      iter++;
    }
  /* End of 
     while (1)
  */

  /* We finished the libpolyxmass_globals_max_monomer_code_len (or
   * less) iterations, so we have a valid code in the code
   * variable. Or the code is empty because we have come to the end of
   * the string.
   */

  /* Set *idx to the index at which we are, the calling function will
     be responsible for incrementing it by one unit for the following
     call if there is one. (+1 because at next function call we do not
     want to re-analyze * the last position iterated here.)
  */

  *idx = *idx + iter;

  strcpy (code, code_new);

  g_free (code_new);

  return iter;
}


gint
pxmchem_monomer_fill_array_from_string (GPtrArray *fillGPA,
					gchar *seq, gint codelen,
					GPtrArray *mnm_refGPA,
					gboolean empty_first)
{
  /* We get a string corresponding to the sequence of the polymer, and
   * we have to construct a GPtrArray of monomer objects corresponding
   * to it.
   * However each time we iterate through a new code in the sequence
   * string, we have to check that this code is known. For this, we 
   * get the address of a reference monomers' GPtrArray in which
   * we check that the code corresponds to a known monomer.
   */
  gint seq_len = 0;
  gint index = 0;
  gint count = 0;
  
  gsize size = 0;
  
  gchar *code = NULL;

  PxmMonomer *monomer = NULL;
  
  g_assert (seq != NULL);

  /* The array of reference monomers cannot be NULL or empty.
  */
  g_assert (mnm_refGPA != NULL && mnm_refGPA->pdata != NULL);

  /* The array where the monomers are going to be stored cannot be
     NULL.
   */
  g_assert (fillGPA != NULL);
  
  /* Manage the empty_first parameter right now.
   */
  if (empty_first == TRUE)
    {
      /* The array should be totally emptied before filling it
       * with new-coming monomer instances.
       */
      pxmchem_monomer_GPA_empty (fillGPA);
    }

  size = strlen (seq);
  g_assert (size < G_MAXINT);
  seq_len = (gint) size;

  if (seq_len == 0)
    return fillGPA->len;
  
  /* If the seq is non-EMPTY, then the rules to extract monomers from
   * it are:
   * 
   * 1. libpolyxmass_globals_max_monomer_code_len characters non-including the
   * terminating NULL character.
   *
   * 2. First character and only first must be uppercase.
   */

  /* Allocate the room for the globally authorized number of 
   * characters in a single code:
   */
  code = g_malloc0 (sizeof (gchar) * codelen + 1);

  /* We now iteratively call a function that extracts the NEXT valid 
   * monomer code from the string passed as param.
   */
  while (1)
    {
      /* First NULL'ize all the cells of the code.
       */
      memset (code, '\x0', codelen + 1);
      
      if (-1 == pxmchem_monomer_extract_code_from_string (code, codelen,
							  seq,
							  &index,
							  NULL))
	{
	  /* Here our parsing is severe, so whatever the reason the -1
	     is returned, we consider that to be a REAL ERROR. (Note
	     that is not the case in the cleavage spec parsing
	     function, because in that case having a '/' is expected
	     and neatly dealt with. In that case, the param that is
	     NULL here is not NULL, and points to a gchar, where the
	     called function will put the invalid character.
	  */

	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to get next valid monomer code"
		 " from string: '%s' at position: '%d'\n"),
		 __FILE__, __LINE__, seq, index + 1);
	  
	  g_free (code);
      
	  return -1;
	}
      
      if (0 == strcmp ("\x0", code))
	/* We must have arrived to end of polymer string sequence
	 * because code contains no characters
	 */
	break;
      
      monomer = pxmchem_monomer_new_by_code (code, mnm_refGPA);

      if (monomer == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	_("%s@%d: failed to allocate monomer from code: '%s'\n"),
	 __FILE__, __LINE__, code);

	  g_free (code);
	  
	  return -1;
	}
      
      /* The allocated monomer seems OK, append it to the fillGPA.
       */
      g_ptr_array_add (fillGPA, monomer);
      count++;
      
      /* Increment index, so that we do not go in a wild loop!
       */
      index++;
    }

  g_free (code);
  
  return count;
}


gint
pxmchem_monomer_fill_array_from_string_check_errors (GPtrArray *fillGPA,
						     GPtrArray *errorsGPA,
						     gchar *seq, 
						     gint codelen,
						     GPtrArray *mnm_refGPA,
						     gboolean empty_first)
{
  /* We get a string corresponding to the sequence of the polymer, and
     we have to construct a GPtrArray of monomer objects corresponding
     to it.  However each time we iterate through a new code in the
     sequence string, we have to check that this code is known. For
     this, we get the address of a reference monomers' GPtrArray in
     which we check that the code corresponds to a known monomer.
   
     In case a parsed element does not correspond to a valid monomer
     code, the current position in the string is stored as an
     allocated gint in the array 'errorsGPA' that is passed as
     parameter.
   */
  gint seq_len = 0;
  gint index = 0;
  gint *errindex = NULL;
  gint count = 0;

  gsize size = 0;

  gchar errchar = '\x0';  
  gchar *code = NULL;

  PxmMonomer *monomer = NULL;
  
  g_assert (seq != NULL);

  /* The array of reference monomers cannot be NULL or empty.
  */
  g_assert (mnm_refGPA != NULL && mnm_refGPA->pdata != NULL);

  /* The array where the monomers are going to be stored cannot be
     NULL. Nor can be NULL the array where the indices of non-valid
     characters are to be stored.
   */
  g_assert (fillGPA != NULL);
  g_assert (errorsGPA != NULL);
  
  /* Manage the empty_first parameter right now.
   */
  if (empty_first == TRUE)
    {
      /* The array should be totally emptied before filling it
       * with new-coming monomer instances.
       */
      pxmchem_monomer_GPA_empty (fillGPA);
    }

  size = strlen (seq);
  g_assert (size < G_MAXINT);
  seq_len = (gint) size;
  
  if (seq_len == 0)
    return fillGPA->len;
  
  /* If the seq is non-EMPTY, then the rules to extract monomers from
   * it are:
   * 
   * 1. libpolyxmass_globals_max_monomer_code_len characters
   * non-including the terminating NULL character.
   *
   * 2. First character and only first must be uppercase.
   */

  /* Allocate the room for the globally authorized number of 
   * characters in a single code:
   */
  code = g_malloc0 (sizeof (gchar) * codelen + 1);

  /* We now iteratively call a function that extracts the NEXT valid 
   * monomer code from the string passed as param.
   */
  while (1)
    {
      /* First NULL'ize all the cells of the code.
       */
      memset (code, '\x0', codelen + 1);
      
      if (-1 == pxmchem_monomer_extract_code_from_string (code, codelen,
							  seq,
							  &index,
							  &errchar))
	{
	  /* If the returned value is -1, then that means that something
	     was wrong parsing the seq string. We have the erroneous 
	     character in errchar. And we have the index of the erroneous
	     character in index. We allocate a gint object by that value
	     and append it in the array errorsGPA passed as parameter.
	  */
	  errindex = g_malloc0 (sizeof (gint));
	  *errindex = index;
	  
	  g_ptr_array_add (errorsGPA, errindex);
	  
	  /* Increment index, so that we do not go in a wild loop!
	   */
	  index++;

	  continue;
	}
      
      if (0 == strcmp ("\x0", code))
	/* We must have arrived to end of polymer string sequence
	 * because code contains no characters
	 */
	break;
      
      monomer = pxmchem_monomer_new_by_code (code, mnm_refGPA);

      if (monomer == NULL)
	{
	  /* Apparently, the code that was returned is not valid. We
	     should try to get the index of the first character in the
	     code, as this is what we have to return in errorsGPA.

	     So, let's say that index is set at 100, and that
	     code->len is 3. What's the index of the first char of the
	     code? It should be 100 - 2, that is 98.
	   */
	  errindex = g_malloc0 (sizeof (gint));
	  *errindex = index - strlen (code) + 1;
	  
	  g_ptr_array_add (errorsGPA, errindex);
	  
	  /* Increment index, so that we do not go in a wild loop!
	   */
	  index++;

	  continue;
	}
      
      
      /* The allocated monomer seems OK, append it to the fillGPA.
       */
      g_ptr_array_add (fillGPA, monomer);
      count++;
      
      /* Increment index, so that we do not go in a wild loop!
       */
      index++;
    }
  /* End of 
     while (1)
  */

  g_free (code);
  
  return count;
}


gint
pxmchem_monomer_count_from_string (gchar* seq, gint codelen,
				   gboolean check_codes,
				   GPtrArray *mnm_refGPA)
{
  gint count = 0;
  gint index = -1;

  gchar * code = NULL;

  /* We iteratively call a function that extracts the NEXT valid 
   * monomer code from the string passed as param. Each time
   * a new valid monomer is found we increment the count variable.
   */

  g_assert (mnm_refGPA != NULL && mnm_refGPA->len > 0);

  /* Allocate the room for the globally authorized number of 
   * characters in a single code:
   */
  code = 
    g_malloc0 (sizeof (gchar) * codelen + 1);
  g_assert (code != NULL);

  while (1)
    {
      /* First NULL'ize all the cells of the code string array.
       */
      memset (code, '\x0', codelen + 1);

      if (-1 == pxmchem_monomer_extract_code_from_string (code, codelen,
							  seq,
							  &index,
							  NULL))
	{
	  /* Here our parsing is severe, so whatever the reason the -1
	     is returned, we consider that to be a REAL ERROR. (Note
	     that is not the case in the cleavage spec parsing
	     function, because in that case having a '/' is expected
	     and neatly dealt with.
	  */

	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to extract a valid code from string: '%s'\n"),
		 __FILE__, __LINE__, seq);
	  
	  g_free (code);
	  
	  return -1;
	}

      if (0 == strcmp ("\x0", code))
	/* We must have arrived to end of polymer string sequence
	   because code contains no characters
	 */
	break;
      
      /* At this point the code should be in code.  If required by the
	 function parameters, we check that the monomer code
	 corresponds to a valid code (ie is found in the array of
	 monomers passed as parameter). Finally increment the count
	 variable.
       */
      if (TRUE == check_codes 
	  && mnm_refGPA != NULL && mnm_refGPA->len > 0)
	{
	  if (-1 == pxmchem_monomer_get_index_by_code (code, mnm_refGPA))
	    return -1;
	}
      
      count++;

      /* Increment index, so that we do not go in a wild loop!
       */
      index++;
    }

  /* At this point we know how many valid monomers there are in the
   * string passed as a parameter.
   */
  g_free (code);

  return count;
}



/*  RENDERING FUNCTIONS - FORMATTER FUNCTIONS
 */
gchar *
pxmchem_monomer_format_xml_string_monomer (PxmMonomer *monomer, 
					   gchar *indent, gint offset)
{
  /* The pointer to the monomer will allow an iteration in the array
     of prop instances (propGPA) and for each prop instance will call
     the relevant pluging function which is registered to make the xml
     string to describe the property object.
   */
  gint iter = 0;
  gint new_offset = 0;
  
  gchar *xml = NULL;
  gchar *lead = NULL;
  gchar *help = NULL;
  
  GString *gs = NULL;
  
  PxmProp *prop = NULL;
  

  g_assert (monomer != NULL && indent != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We are willing to create an <monomer> node that should look like this:
   *  <monomer>
   *    <code>S</code>
   *    <prop>
   *      <name>MODIF</name>
   *      <data>Phosphorylation</data>
   *    </prop>
   *    <prop>
   *      <name>COMMENT</name>
   *      <data>Phosphorylation is only partial</data>
   *    </prop>
   *  </monomer>
   *
   */

  /* Open the <monomer> element and immediately insert the 
   * non-iterative data.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);

  g_string_append_printf (gs, "%s<monomer>\n", lead);

  g_free (lead);
  new_offset = offset + 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  g_string_append_printf (gs, "%s<code>%s</code>\n",
		     lead, monomer->code);
  
  /* The monomer may have zero, one or more properties, so we must
   * iterate in its array of property instances.
   */
  for (iter = 0; iter < monomer->propGPA->len; iter++)
    {
      /* Get the iterated PxmProp object.
       */
      prop = g_ptr_array_index (monomer->propGPA, iter);
      g_assert (prop != NULL && prop->name != NULL);

      /* Ask what function is responsible for formatting the xml
        string corresponding to the current PxmProp instance.
       
        Note that there are a number of monomer properties, like the
        ones necessary in the find/replace routines, that do not have
        any corresponding xml formatter plugin function.  In this
        case, the call below will yield a NULL pointer, as it does not
        make sense for the current prop instance to be "serialized"
        permanently to disk files.
       */
      libpolyxmass_prop_formatter_xml_prop_plugin =
	libpolyxmass_prop_choose_format_xml_prop_plugin (prop->name);

      if (libpolyxmass_prop_formatter_xml_prop_plugin != NULL)
	{
	  xml = libpolyxmass_prop_formatter_xml_prop_plugin (prop, 
							indent,
							new_offset,
							NULL);
	  
	  g_assert (xml != NULL);
	  
	  /* In some cases, we want to bypass a given prop processing
	     (like the monomer chembridge prop, for example) returning
	     a string that says not to append it to the elongating
	     gs_prop GString... this is why we check for a "non
	     applicable" "N/A" pattern. See the chembridge dealing
	     function write_seq_monomer_chembridge_prop_plugin for an
	     explanation.
	  */
	  if (0 != strcmp ("N/A", xml))
	    gs = g_string_append (gs, xml);
	  
	  g_free (xml);
	  xml = NULL;
	}
      else
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
		_("%s@%d: plugin prop: '%s' has no registered xml formatter\n"),
		 __FILE__, __LINE__, prop->name);
	  
	}
    }
  /* end of:
     for (iter = 0; iter < monomer->propGPA->len; iter++)
   */

  g_free (lead);
  
  /* Finally close the monomer element.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);

  g_string_append_printf (gs, "%s</monomer>\n", lead);
  
  g_free (lead);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}


gchar *
pxmchem_monomer_format_txt_string_monomer (PxmMonomer *monomer,
					   PxmReportOpt *reportopt)
{
  /* The pointer to the monomer will allow an iteration in the array
     of prop instances (propGPA) and for each prop instance will call
     the relevant pluging function which is registered to make the xml
     string to describe the property object.
   */
  gint iter = 0;
  
  gchar *txt = NULL;
  gchar *help = NULL;
  
  GString *gs = NULL;
  
  PxmProp *prop = NULL;
  

  g_assert (monomer != NULL && reportopt != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  
  /* The monomer may have zero, one or more properties, so we must
   * iterate in its array of property instances.
   */
  for (iter = 0; iter < monomer->propGPA->len; iter++)
    {
      /* Get the iterated PxmProp object.
       */
      prop = g_ptr_array_index (monomer->propGPA, iter);
      g_assert (prop != NULL && prop->name != NULL);

      /* Ask what function is responsible for formatting the xml
        string corresponding to the current PxmProp instance.
      */
      libpolyxmass_prop_formatter_txt_prop_plugin =
	libpolyxmass_prop_choose_format_txt_prop_plugin (prop->name);
      
      if (libpolyxmass_prop_formatter_txt_prop_plugin != NULL)
	{
	  txt = libpolyxmass_prop_formatter_txt_prop_plugin (prop, NULL);
	  
	  g_assert (txt != NULL);
	  
	  /* In some cases, we want to bypass a given prop processing
	     (like the monomer chembridge prop, for example) returning
	     a string that says not to append it to the elongating
	     gs_prop GString... this is why we check for a "non
	     applicable" "N/A" pattern. See the chembridge dealing
	     function write_seq_monomer_chembridge_prop_plugin for an
	     explanation.
	  */
	  if (0 != strcmp ("N/A", txt))
	    gs = g_string_append (gs, txt);
	  
	  g_free (txt);
	  txt = NULL;
	}
      else
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
		 _("%s@%d: plugin prop: '%s' has no registered txt formatter\n"),
		 __FILE__, __LINE__, prop->name);
	  
	}
    }
  /* end of:
     for (iter = 0; iter < monomer->propGPA->len; iter++)
   */

  g_string_append_printf (gs, 
			  _("---------------------------------------------\n"));
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}


PxmMonomer *
pxmchem_monomer_render_xml_node_monomer (xmlDocPtr xml_doc,
					 xmlNodePtr xml_node,
					 GPtrArray *mnm_refGPA,
					 gpointer user_data)
{
  /* The xml node we are in is structured this way:
   *
   * <monomer>
   *   <code>S</code>
   *   <prop>
   *     <name>MODIFICATION</name>
   *     <data>acetylation</data>
   *   </prop>
   *  ... other <prop>..</prop> elements
   * </monomer>
   *
   * And the xml_node parameter points to the 
   *
   * <monomer> element tag:
   *  ^
   *  |
   *  +----- here we are right now.
   * 
   * Which means that xml_node->name == "monomer" and that
   * we'll have to go one step down to the first child of the 
   * current node in order to get to the <code> element.
   *
   * The DTD says this : <!ELEMENT monomer (code, prop*)>
   */

  PxmMonomer *monomer = NULL;
  PxmProp *prop = NULL;

  gchar *help = NULL;
    
  xmlNodePtr xml_child_node = NULL;
  
  g_assert (mnm_refGPA != NULL && mnm_refGPA->len > 0);

  /* Make sure we have parameters pointing bona fide to the right
   * xml element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == strcmp ((gchar *) xml_node->name, "monomer"));
  
  /* Now go to the first child of current node: <code>.
   */
  xml_node = xml_node->children;

  /* From a rigorous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <code> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "code"));
  
  help = (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);

  /* Allocate the monomer instance by using the code that was read
   * from xml data, and by giving the array of reference mononomers
   * (which may be the array of monomers that are currently known
   * for a given polymer definition).
   *
   * Failure to do so will return NULL.
   */
  monomer = pxmchem_monomer_new_by_code (help, mnm_refGPA);
  
  if (monomer == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	 _("%s@%d: failed to create monomer from code: '%s'\n"),
	     __FILE__, __LINE__, help);

      g_free (help);
  
      return NULL;
    }
  
  g_free (help);
  
  /* At this point we have a monomer that is fully qualified for 
   * its name and code and formula.
   *
   * Further, we may have no, or one or more prop elements. 
   * So we should be very picky about what is happening.
   */

  /* Go to the next node, since the <prop> node is a sibling of the
   * <code> node that we just have parsed (see representation above).
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  while (xml_node != NULL)
    {
      /* There is a node there, it MUST BE a <prop> element !
       */
      g_assert (0 == strcmp ("prop", 
			     (gchar *) xml_node->name));
      
      /* Just get the prop's name so that we can get to the
       * proper xml renderer plugin function, without modifying
       * the xml_node, since the pluging function expects a node 
       * pointer corresponding to <prop> and not <name>.
       */
      xml_child_node = xml_node->children;
      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
  
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, 
			     "name"));

      help = 
	(gchar *) xmlNodeListGetString (xml_doc, xml_child_node->xmlChildrenNode, 1);

      libpolyxmass_prop_renderer_xml_prop_plugin = 
	libpolyxmass_prop_choose_render_xml_prop_plugin (help);
      
      g_free (help);
      
      if (libpolyxmass_prop_renderer_xml_prop_plugin == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: plugin prop: '%s' has no registered xml renderer\n"),
		 __FILE__, __LINE__, help);

	  pxmchem_monomer_free (monomer);

	  return NULL;
	}
      
      prop = 
	libpolyxmass_prop_renderer_xml_prop_plugin (xml_doc, xml_node, NULL);
      g_assert (prop != NULL);

      /* Append the newly allocated prop instance to the array of the 
       * monomer.
       */
      g_ptr_array_add (monomer->propGPA, prop);

      /* And now go to the next <prop>, if any.
       */
      xml_node = xml_node->next;
      /* From a rigorous XML parsing point of view, the blanks found in
       * the XML document are considered to be nodes, and we have to detect
       * these and take proper action: go next sibling (next blank) as long
       * as blanks are encountered.
       */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  /* Finished the rendering of the current <monomer> node.
   */
  return monomer;
}


gint
pxmchem_monomer_render_xml_node_codes (xmlDocPtr xml_doc,
				       xmlNodePtr xml_node,
				       GPtrArray *mnm_refGPA,
				       gint codelen,
				       GPtrArray *fillGPA,
				       gpointer user_data)
{
  /* The xml node we are in is structured this way:
   *
   * <codes>GRKDKNFLKMGRKSKKEKKEKKPVVSLYMLVG</codes>
   *
   * And the xml_node parameter points to the 
   *
   * <codes> element tag:
   *  ^
   *  |
   *  +----- here we are right now.
   * 
   * Which means that xml_node->name == "codes" and that
   * the data we want to get is in this node.
   *
   * Note that we know, by necessity, that the monomer that we will
   * render by walking in the codes data (a string made of 
   * monomer codes) are not modified. It is thus simply a matter of
   * filling an array of monomers with newly created ones by iterating
   * in the codes string.
   *
   * The fillGPA pointer passed to function is a GPtrArray pointer
   * into which we are going to append allocated PxmMonomer instances
   * as we iterate in the codes string.
   */
  gchar *codes = NULL;
  
  gint count = 0;
  

  g_assert (xml_node != NULL);
  g_assert (fillGPA != NULL);
  
  g_assert (mnm_refGPA != NULL && mnm_refGPA->len > 0);

  /* Check that we have effectively a <code> element here and that
   * its contents are not NULL.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "codes"));
  
  codes = (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);

  /* We now have to iterate in the string and extract each monomer
   * code in order to create with it a newly allocated monomer instance.
   * Each monomer instance is then appended into the user_data-pointed-by
   * GPtrArray of monomer instances.
   */
  count = pxmchem_monomer_fill_array_from_string (fillGPA, codes, codelen,
						  mnm_refGPA, FALSE);
  
  if (-1 == count)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to fill monomer array from string "
	     "of codes: '%s'\n"),
	     __FILE__, __LINE__, codes);
      
      return -1;
    }
  
  /* At this point the fillGPA array has been elongated with 
   * the monomer instances allocated using codes iteratively extracted
   * from the codes string, we can free codes.
   */
  g_free (codes);
  
  return count;
}



gchar *
pxmchem_monomer_format_xml_string_mnm (PxmMonomer *monomer, 
				   gchar *indent, gint offset)
{
  /* The pointer to the monomer will allow to create a string representing
   * its member data: name, code and formula.
   */
  gint new_offset = 0;
  
  gchar *lead = NULL;
  gchar *help = NULL;
  
  GString *gs = NULL;
  

  
  g_assert (monomer != NULL && indent != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We are willing to create an <mnm> node that should look like this:
   *
   * <mnm>
   *   <name>Glycine</name>
   *   <code>G</code>
   *   <formula>C2H3N1O1</formula>
   * </mnm>
   *
   */

  /* Open the <mnm> element and immediately insert the data.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);

  g_string_append_printf (gs, "%s<mnm>\n", lead);

  g_free (lead);

  new_offset = offset + 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  g_assert (monomer->name != NULL && strlen (monomer->name) > 0);
  g_string_append_printf (gs, "%s<name>%s</name>\n",
		     lead, monomer->name);
  
  g_assert (monomer->code != NULL && strlen (monomer->code) > 0);
  g_string_append_printf (gs, "%s<code>%s</code>\n",
		     lead, monomer->code);
  
  g_assert (monomer->formula != NULL && strlen (monomer->formula) > 0);
  g_string_append_printf (gs, "%s<formula>%s</formula>\n",
		     lead, monomer->formula);
  
  g_free (lead);

  /* Finally close the mnm element.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);
  
  g_string_append_printf (gs, "%s</mnm>\n", lead);

  g_free (lead);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}



PxmMonomer *
pxmchem_monomer_render_xml_node_mnm (xmlDocPtr xml_doc,
				 xmlNodePtr xml_node,
				 gpointer user_data)
{
  /* The xml node we are in is structured this way:
   *
   * <mnm>
   *   <name>Glycine</name>
   *   <code>G</code>
   *   <formula>C2H3N1O1</formula>
   * </mnm>
   *
   * And the xml_node parameter points to the 
   *
   * <mnm> element tag:
   *  ^
   *  |
   *  +----- here we are right now.
   * 
   * Which means that xml_node->name == "mnm" and that
   * we'll have to go one step down to the first child of the 
   * current node in order to get to the <name> element.
   *
   */
  PxmMonomer *monomer = NULL;

  /* Make sure we have parameters pointing bona fide to the right
   * xml element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == strcmp ((gchar *) xml_node->name, "mnm"));
  
  /* Now go to the first child of current node: <name>.
   */
  xml_node = xml_node->children;

  /* From a rigorous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <name> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "name"));

  monomer = pxmchem_monomer_new ();
  
  /* Since we have allocated the monomer instance at the line above,
     we know that its member data are NULL, so we can make direct
     assignements, without recoursing to the _set_xxx ().
  */
  monomer->name = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (monomer->name != NULL);
    
  /* Now go to the second child: <code> which is the next sibling of <name>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <code> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "code"));

  monomer->code = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (monomer->code != NULL);

  /* Now go to the third child: <formula> which is the next sibling of
   * <code>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <formula> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "formula"));

  monomer->formula = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (monomer->formula != NULL);
  
  /* Finished the rendering of the current <mnm> node.
   */
  return monomer;
}









/* FREE'ING FUNCTIONS
 */
gboolean
pxmchem_monomer_free (PxmMonomer *monomer)
{
  g_assert (monomer != NULL);
  
  /* Free inner material first.
   */
  if (monomer->propGPA != NULL)
    libpolyxmass_prop_GPA_free (monomer->propGPA);
  
  if (monomer->name != NULL)
    g_free (monomer->name);
  
  if (monomer->code != NULL)
    g_free (monomer->code);
  
  if (monomer->formula != NULL)
    g_free (monomer->formula);
  
  g_free (monomer);
  
  return TRUE;
}


/* GPtrArray-RELATED FUNCTIONS
 */
gint
pxmchem_monomer_GPA_empty (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmMonomer *monomer = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      monomer = g_ptr_array_remove_index (GPA, 0);
      g_assert (monomer != NULL);
      pxmchem_monomer_free (monomer);
      count++;
    }
  
  return count;
}


gint
pxmchem_monomer_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  

  g_assert (GPA != NULL);
  
  count = pxmchem_monomer_GPA_empty (GPA);
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}


gint
pxmchem_monomer_GPA_prop_free (PxmProp *prop)
{
  gint count = 0;
  
  /* We get a PxmProp object in which the data member is actually 
     a GPtrArray * of PxmMonomer instances.
  */
  g_assert (prop != NULL);
  
  if (prop->name != NULL)
    g_free (prop->name);
  
  if (prop->data != NULL)
    count = pxmchem_monomer_GPA_free ((GPtrArray *) prop->data);
  
  g_free (prop);

  return count;
}

  










