/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "libpolyxmass-polchemdefspec.h"
#include "libpolyxmass-config.h"


/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmPolchemdefSpec *
libpolyxmass_polchemdefspec_new (void)
{
  PxmPolchemdefSpec *ps = NULL;
  
  ps = g_malloc0 (sizeof (PxmPolchemdefSpec));
  
  ps->type = g_strdup ("");
  ps->file = g_strdup ("");
  ps->dir = g_strdup ("");
  ps->source = CONFIG_SYSTEM;
  
  return ps;
}


PxmPolchemdefSpec *
libpolyxmass_polchemdefspec_dup (PxmPolchemdefSpec *ps)
{
  PxmPolchemdefSpec *new_ps = NULL;
  
  g_assert (ps != NULL);
  
  new_ps = libpolyxmass_polchemdefspec_new ();
  
  new_ps->type = g_strdup (ps->type) ;
  new_ps->file = g_strdup (ps->file) ;
  new_ps->source = ps->source ;
  
  return new_ps;
}



gboolean
libpolyxmass_polchemdefspec_set_type (PxmPolchemdefSpec *ps, gchar *type)
{
  g_assert (ps != NULL && type != NULL);

  g_assert (ps->type != NULL);
  
  g_free (ps->type);
  
  ps->type = g_strdup (type);
  
  return TRUE;
}


gboolean
libpolyxmass_polchemdefspec_set_file (PxmPolchemdefSpec *ps, gchar *file)
{
  g_assert (ps != NULL && file != NULL);

  g_assert (ps->file != NULL);
  
  g_free (ps->file);
  
  ps->file = g_strdup (file);
  
  return TRUE;
}


gboolean
libpolyxmass_polchemdefspec_set_dir (PxmPolchemdefSpec *ps, gchar *dir)
{
  g_assert (ps != NULL && dir != NULL);

  g_assert (ps->dir != NULL);
  
  g_free (ps->dir);
  
  ps->dir = g_strdup (dir);
  
  return TRUE;
}

gboolean
libpolyxmass_polchemdefspec_set_source (PxmPolchemdefSpec *ps, gint source)
{
  g_assert (ps != NULL);
  g_assert (source == CONFIG_SYSTEM || source == CONFIG_USER);
  
  ps->source = source;
  
  return TRUE;
}






/*  LOCATING FUNCTIONS
 */
gint
libpolyxmass_polchemdefspec_get_index_by_type (GPtrArray *GPA, gchar *type)
{
  return libpolyxmass_polchemdefspec_get_index_top_by_type (GPA, type);
}


gint
libpolyxmass_polchemdefspec_get_index_top_by_type (GPtrArray *GPA, gchar *type)
{
  gint iter = 0;
  PxmPolchemdefSpec *polchemdefspec = NULL;
  

  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      polchemdefspec = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (polchemdefspec->type, type))
	return iter;
    }
  
  return -1;
}


gint
libpolyxmass_polchemdefspec_get_index_bottom_by_type (GPtrArray *GPA, 
						   gchar *type)
{
  gint iter = 0;
  PxmPolchemdefSpec *polchemdefspec = NULL;
  

  g_assert (GPA != NULL);
  
  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  polchemdefspec = g_ptr_array_index (GPA, iter);
	  
	  if (0 == strcmp (polchemdefspec->type, type))
	    return iter;
	}
    }
    
  return -1;
}


PxmPolchemdefSpec *
libpolyxmass_polchemdefspec_get_ptr_by_type (GPtrArray *GPA, gchar *type)
{
  return libpolyxmass_polchemdefspec_get_ptr_top_by_type (GPA, type);
}


PxmPolchemdefSpec *
libpolyxmass_polchemdefspec_get_ptr_top_by_type (GPtrArray *GPA, gchar *type)
{
  gint iter = 0;
  PxmPolchemdefSpec *polchemdefspec = NULL;
  

  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      polchemdefspec = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (polchemdefspec->type, type))
	return polchemdefspec;
    }
  
  return NULL;
}


PxmPolchemdefSpec *
libpolyxmass_polchemdefspec_get_ptr_bottom_by_type (GPtrArray *GPA, 
						    gchar *type)
{
  gint iter = 0;
  PxmPolchemdefSpec *polchemdefspec = NULL;
  

  g_assert (GPA != NULL);
  
  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  polchemdefspec = g_ptr_array_index (GPA, iter);
	  
	  if (0 == strcmp (polchemdefspec->type, type))
	    return polchemdefspec;
	}
    }
    
  return NULL;
}




/* UTILITY FUNCTIONS
 */



gboolean
libpolyxmass_polchemdefspec_reset (PxmPolchemdefSpec *ps)
{
  g_assert (ps != NULL);
  
  g_assert (ps->type != NULL);
  g_free (ps->type);
  ps->type = g_strdup ("");
  
  g_assert (ps->file != NULL);
  g_free (ps->file);
  ps->file = g_strdup ("");
  
  g_assert (ps->dir != NULL);
  g_free (ps->dir);
  ps->dir = g_strdup ("");
  
  ps->source = CONFIG_SYSTEM;
  
  return TRUE;
}


/* 
   The polchem-defs.cat catalogue files.
*/
gboolean
libpolyxmass_polchemdefspec_init_from_cat_line (gchar *line,
						PxmPolchemdefSpec *ps) 
{
  gint iter = 0;

  GString * gs = NULL;

  gboolean type_done = FALSE;
  gboolean file_done = FALSE;
  
  g_assert (ps != NULL);

  gs = g_string_new ("");

  /*
    We have a line of the kind

    "polchemdef_name=path-polchemdeffile.xml%path-polchemdefdir", i.e.

    "type=/absolute/path/to/polchemdef.xml%/absolute/path/to/dir" 

    (note that the newline character may be found if the line in
    question is not the last of the file!), we just need to
    deconstruct this into three pieces.
   */
  while (line [iter] != '\x0' && line [iter] != '\n')
    {
      if (line [iter] == '=')
	{
	  /* We should now have the polymer definition name in the gs
	   * GString.
	   */
	  if (strlen (gs->str) <= 0)
	    {
	      g_string_free (gs, TRUE);

	      return FALSE;
	    }

	  libpolyxmass_polchemdefspec_set_type (ps, gs->str);
	  
	  gs = g_string_truncate (gs, 0);

	  /*
	    g_string_free (gs, TRUE);
	    gs= g_string_new ("");
	  */
	  
	  type_done = TRUE;
	}

      else if (line [iter] == '%')
	{
	  /* We should now have the polymer definition file path+name in
	   * the gs GString.
	   */
	  if (strlen (gs->str) <= 0)
	    {
	      g_string_free (gs, TRUE);
	      
	      return FALSE;
	    }

	  /* 
	     Make sure that the file actually exists. Also be certain that
	     the gs->str string is an absolute filename.
	  */
	  g_assert (TRUE == g_path_is_absolute (gs->str));
	  g_assert (TRUE == g_file_test (gs->str, G_FILE_TEST_EXISTS));
	  
	  libpolyxmass_polchemdefspec_set_file (ps, gs->str);
	  
	  gs = g_string_truncate (gs, 0);

	  /*
	    g_string_free (gs, TRUE);
	    gs= g_string_new ("");
	  */

	  file_done = TRUE;
	}

      else
	{
	  /* We are iterating in some character, so we should simply
	   * put this into the gs GString.
	   */
	  gs = g_string_append_c (gs, line [iter]);
	}

      iter++;		    
    }

  /* And now we are at the end of the line, so the directory should
   * have completed now, but check that the type and file were done
   * already.
   */
  if (type_done == FALSE || file_done == FALSE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to parse polchemdefspec line: '%s'\n"),
	     __FILE__, __LINE__, line);
      
      g_string_free (gs, TRUE);
      
      return FALSE;
    }

  /* We should now have the directory in the gs GString.
   */
  /* 
     Make sure that the directory actually exists. Also be certain that
     the gs->str string is an absolute filename.
  */
  g_assert (TRUE == g_file_test (gs->str, G_FILE_TEST_IS_DIR));
  g_assert (TRUE == g_path_is_absolute (gs->str));
  
  libpolyxmass_polchemdefspec_set_dir (ps, gs->str);

  /* We are no longer going to use this allocated GString.
   */
  /* Free the gs GString, as the spec_set_xxx functions allocate
     a string using the param as a template.
  */
  g_string_free (gs, TRUE);
  
  return TRUE;
}



gint
libpolyxmass_polchemdefspec_parse_polchem_defs_cat_file (gchar *file,
						      GPtrArray *GPA,
						      gchar *type,
						      PxmPolchemdefSpec **pps,
						      gint pre_ap_pend,
						      gint sys_user)
{
  gint count = 0;
  
  gchar *line = NULL;
  gchar *read = NULL;

  FILE *cfgp = NULL;

  PxmPolchemdefSpec *ps = NULL;

  /* 
     We are given a file name were each line gives the name of an atom
     definition and the location of the atom definition file. All we
     have to do is parse each line and for each valid one create a
     PxmAtomSpec item that we append/prepend to the GPA file.

     Note that if 'type' parameter is non-NULL, that means that 'pps'
     should also be non-NULL. This is because this function might also
     be called to "find" the proper type from the 'file' and make a
     PxmPolchemdefSpec instance and return it. If this operation is
     successful, 1 is returned, otherwise -1 is returned.
     
  */
  g_assert (file != NULL);
  g_assert (TRUE == g_file_test (file, G_FILE_TEST_EXISTS));
  g_assert (TRUE == g_path_is_absolute (file));
  g_assert (GPA != NULL);
  g_assert (pre_ap_pend == CONFIG_APPEND || pre_ap_pend == CONFIG_PREPEND);
  g_assert (sys_user == CONFIG_SYSTEM || sys_user == CONFIG_USER);
 

  if (type != NULL)
    g_assert (pps != NULL && *pps == NULL);
  

  cfgp = fopen (file, "r");
  
  if (cfgp == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to  open file: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return -1;
    }
  
   /*
    The lines that we have are of the kind

    "polchemdef_name=path-polchemdeffile.xml%path-polchemdefdir", i.e.

    "type=/absolute/path/to/polchemdef.xml%/absolute/path/to/dir" 
   */
  
  /* Allocate the memory into which each line of the file will be
   * store for recursive parsing.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));

  read = line;

  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, cfgp);

      /*
      debug_printf (("file %s -- read line '%s'\n", 
		     file, read));
      */

      if (read != NULL && strlen (read) > 1)
	{
	  /* Apparently something interesting should be in 'read' for
	     us to parse.
	  */
	  /* We first have to check if the line we are reading is
	     actually a comment line, beginning with the '#'
	     character. The same applies if the line is empty (starts
	     with a newline...
	   */
	  if (read [0] == '#' || read [0] == '\n')
	    continue;

	  while (read [0] != '\x0')
	    {
	      if (TRUE == g_ascii_isspace (read [0]))
		read++;
	      else
		break;
	    }
	  if (strlen (read) <= 0)
	    continue;
	  
	  /* 
	     Next check if we are on a line that describes the
	     correspondence between a polymer definition name and its
	     definition file and its directory. This is most probably
	     the case if both '=' and '%' are present in the line.
	   */
	  if (NULL != strchr (read, '=') 
	      && NULL != strchr (read, '%'))
	    {
	      /* 
		 We are dealing with a
		 "polchemdef_name=path-polchemdeffile.xml%path-polchemdefdir"
		 correspondence.
	      */
	      
	      /* Allocate the PxmPolchemdefSpec instance that we'll
		 initialize using the data read from parsing of the
		 line.
	       */
	      ps = libpolyxmass_polchemdefspec_new ();
	      libpolyxmass_polchemdefspec_set_source (ps, sys_user);
	      
	      if (FALSE == 
		  libpolyxmass_polchemdefspec_init_from_cat_line (read, ps))
		{
		  g_free (line);
		  
		  libpolyxmass_polchemdefspec_free (ps);
		  
		  fclose (cfgp);
		  
		  return -1;
		}
	      else /* if return func == TRUE */
		{
		  if (type != NULL)
		    {
		      /*
			We were asked to actually just find the proper one
			and set it to 'pps'.
		      */
		      if (0 == strcmp (ps->type, type))
			{
			  g_free (line);
			  
			  fclose (cfgp);
		      
			  *pps = ps;
			  
			  return 1;
			}
		      else
			continue;
		    }
		  else
		    {
		      if (pre_ap_pend == CONFIG_APPEND)
			g_ptr_array_add (GPA, ps);
		      else /* if (pre_ap_pend == CONFIG_PREPEND) */
			g_ptr_array_insert_val (GPA, 0, ps);
		      
		      /*
			debug_printf (("added new ps: '%s--%s--%s'\n",
			ps->type, ps->file, ps->dir));
		      */
		      
		      count++;
		      
		      continue;
		    }
		}
	      /* End of 
		 else  if return func == TRUE 
	      */
	    }
	  /* End of 
	     if (NULL != strchr (read, '=') 
	     && NULL != strchr (read, '%'))
	  */
	}
      /* End of 
	 if (read != NULL && strlen (read) > 1)
      */
    }
  /* End of
     while (read != NULL)
  */
  
  /* 
     Now that we do not need this string anymore, since the file is
     finished parsing, we can free the memory for reuse of the pointer.
  */
  g_free (line);
  
  fclose (cfgp);
  
  /* 'count' will be zero if 'type' param was non-NULL, because it was 
     incremented only when PxmPolchemdefSpec instances were added to GPA, 
     which does not happen when 'type' is non-NULL.
  */
  return count;
}


gint
libpolyxmass_polchemdefspec_parse_polchem_defs_cat_files (GPtrArray *GPA,
						       gint pre_ap_pend,
						       gint sys_user)
{
  const gchar *parsed_file = NULL;
  gchar *parsed_file_utf8 = NULL;
  gchar *catalogue_dir = NULL;
  gchar *absolute_filename = NULL;

  GDir*  cat_dir = NULL;

  GError *g_error = NULL;

  gint flags = 0;
  gint count = 0;
  
  /*
    We are asked to fill the GPA array with allocated PxmAtomSpec
    objects corresponding to data found in the different polchem-def
    catalogue files that are parsed according to function parameter
    requirements.
  */

  g_assert (GPA != NULL);
  g_assert (pre_ap_pend == CONFIG_APPEND || pre_ap_pend == CONFIG_PREPEND);
  g_assert (sys_user == CONFIG_USER 
	    || sys_user == CONFIG_SYSTEM 
	    || sys_user == CONFIG_BOTH);
  

  if (sys_user == CONFIG_USER || sys_user == CONFIG_BOTH)
    {
      catalogue_dir = 
	libpolyxmass_config_get_user_polchem_defs_cat_dir ();

      /* 
	 We only parse the files in the directory if the directory
	 exists...
      */
      if (catalogue_dir != NULL 
	  && TRUE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
	{
	  /* Let's check if there are catalogue files in the directory.
	   */
	  
	  cat_dir = g_dir_open (catalogue_dir,
				flags,
				&g_error);

	  /* And now get the files in this directory and check if 
	     they are actually catalogue files.
	  */
	  parsed_file = g_dir_read_name (cat_dir);
	  
	  while (parsed_file != NULL)
	    {
	      /* Translate the name to utf_8.
	       */
	      parsed_file_utf8 = g_filename_to_utf8 (parsed_file,
						     -1,
						     NULL,
						     NULL,
						     &g_error);
	      if (NULL == parsed_file_utf8)
		{
		  parsed_file = g_dir_read_name (cat_dir);
		  continue;
		}

	      /* Make sure the filename actually ends with the proper
		 suffix ("polchem-defs-cat").
	      */
	      if (FALSE == g_str_has_suffix (parsed_file_utf8,
                                             "polchem-defs-cat"))
		{
		  g_free (parsed_file_utf8);
		  parsed_file = g_dir_read_name (cat_dir);
		  continue;
		}
	      
	      /* We will parse the file, but to do that we have to 
		 construct an absolute path to that file.
	      */
	      absolute_filename = g_strdup_printf ("%s/%s",
						   catalogue_dir,
						   parsed_file_utf8);
	      
	      /*
		Do the parsing of the file into the GPA array.
	      */
	      count += 
		libpolyxmass_polchemdefspec_parse_polchem_defs_cat_file 
		(absolute_filename,
		 GPA,
		 NULL,
		 NULL,
		 pre_ap_pend,
		 CONFIG_USER);
	      
	      if (count == -1)
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
			 _("%s@%d: failed to parse file: '%s'\n"),
			 __FILE__, __LINE__, absolute_filename);
		}
	      
	      g_free (parsed_file_utf8);
	      g_free (absolute_filename);

	      parsed_file = g_dir_read_name (cat_dir);

	      continue;
	    }
	  
	  /* We can free the catalogue_dir string, we do not need it 
	     anymore.
	  */
	  g_free (catalogue_dir);
	  
	  g_dir_close (cat_dir);
	}
      /* End of
	 if (TRUE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
      */
    }
  /* End of 
     if (sys_user == CONFIG_USER || sys_user == CONFIG_BOTH)
  */
  if (sys_user == CONFIG_SYSTEM || sys_user == CONFIG_BOTH)
    {
      catalogue_dir = libpolyxmass_config_get_system_polchem_defs_cat_dir ();

      /* 
	 We only parse the files in the directory if the directory
	 exists...
      */
      if (catalogue_dir != NULL 
	  && TRUE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
	{
	  /* Let's check if there are catalogue files in the directory.
	   */
	  
	  cat_dir = g_dir_open (catalogue_dir,
				flags,
				&g_error);

	  /* And now get the files in this directory and check if 
	     they are actually catalogue files.
	  */
	  parsed_file = g_dir_read_name (cat_dir);
	  
	  while (parsed_file != NULL)
	    {
	      /* Translate the name to utf_8.
	       */
	      parsed_file_utf8 = g_filename_to_utf8 (parsed_file,
						     -1,
						     NULL,
						     NULL,
						     &g_error);
	      if (NULL == parsed_file_utf8)
		{
		  parsed_file = g_dir_read_name (cat_dir);
		  continue;
		}

	      /* Make sure the filename actually ends with the proper
		 suffix ("polchem-defs-cat").
	      */
	      if (FALSE == g_str_has_suffix (parsed_file_utf8,
                                             "polchem-defs-cat"))
		{
		  g_free (parsed_file_utf8);
		  parsed_file = g_dir_read_name (cat_dir);
		  continue;
		}
	      
	      /* We will parse the file, but to do that we have to 
		 construct an absolute path to that file.
	      */
	      absolute_filename = g_strdup_printf ("%s/%s",
						   catalogue_dir,
						   parsed_file_utf8);
	      
	      /*
		Do the parsing of the file into the GPA array.
	      */
	      count += 
		libpolyxmass_polchemdefspec_parse_polchem_defs_cat_file 
		(absolute_filename,
		 GPA,
		 NULL,
		 NULL,
		 pre_ap_pend,
		 CONFIG_USER);
	      
	      if (count == -1)
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
			 _("%s@%d: failed to parse file: '%s'\n"),
			 __FILE__, __LINE__, absolute_filename);
		}
	      
	      g_free (parsed_file_utf8);
	      g_free (absolute_filename);

	      parsed_file = g_dir_read_name (cat_dir);

	      continue;
	    }
	  
	  /* We can free the catalogue_dir string, we do not need it 
	     anymore.
	  */
	  g_free (catalogue_dir);
	  
	  g_dir_close (cat_dir);
	}
      /* End of
	 if (TRUE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
      */
    }
  /* End of 
     if (sys_user == CONFIG_SYSTEM || sys_user == CONFIG_BOTH)
  */

  return count;  
}




/* FREE'ING FUNCTIONS
 */
gint
libpolyxmass_polchemdefspec_free (PxmPolchemdefSpec *ps)
{
  g_assert (ps != NULL);
  
  if (ps->type != NULL)
    g_free (ps->type);
  
  if (ps->file != NULL)
    g_free (ps->file);
  
  if (ps->dir != NULL)
    g_free (ps->dir);
  
  g_free (ps);
  
  return 1;
}




/* GPtrArray-RELATED FUNCTIONS
 */
gint
libpolyxmass_polchemdefspec_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmPolchemdefSpec *ps = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      ps = g_ptr_array_remove_index (GPA, 0);
      g_assert (ps != NULL);
      libpolyxmass_polchemdefspec_free (ps);
      count++;
    }
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}

