/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.

   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include "libpolyxmass-globals.h"

/* fully copied from glib file garray.c in order to implement
 * function below g_ptr_array_insert_val
 */
/** glib function to calculate the size of the GPtrArray in order
 * to allow insertion of a new item to it
 */
gint 
g_nearest_pow (gint num)
{
  gint n = 1;

  while (n < num)
    n <<= 1;

  return n;
}

/* glib function to determine if it is necessary to increase the
 * size of a GPtrArray when insertion of element(s) is required
 */
void 
g_ptr_array_maybe_expand (GRealPtrArray *array,
			       gint len)
{
  guint old_alloc;

  if ((array->len + len) > array->alloc)
    {
      old_alloc = array->alloc;

      array->alloc = g_nearest_pow (array->len + len);
      array->alloc = MAX (array->alloc, MIN_ARRAY_SIZE);
      if (array->pdata)
	array->pdata = g_realloc (array->pdata, 
				  sizeof(gpointer) * array->alloc);
      else
	array->pdata = g_new0 (gpointer, array->alloc);

      memset (array->pdata + old_alloc, 0, 
	      sizeof (gpointer) * (array->alloc - old_alloc));
    }
}

/**
 * g_ptr_array_insert_val:
 * @farray: pointer to the array in which the @data is to be inserted.
 * @index: index at which insertion must take place.
 * @data: data pointer that must be inserted in @farray.
 * 
 * Inserts a new data pointer in @farray. Each data pointer already
 * present in the @farray and at index higher than @index are
 * moved to the higher indexes.
 * 
 * Returns: a pointer to the data inserted at @index.
 */
gpointer 
g_ptr_array_insert_val (GPtrArray * farray, guint index,
			gpointer data)
{
  GRealPtrArray* array = (GRealPtrArray*) farray;
  g_return_val_if_fail (array, NULL);

  g_return_val_if_fail (index <= array->len, NULL);

  g_ptr_array_maybe_expand (array, 1);

  g_memmove (array->pdata + index + 1, array->pdata + index, 
	     sizeof (gpointer) * (array->len - index - 1 + 1));

  array->pdata [index] = data;

  array->len += 1;

  return array->pdata [index];
}


/* These global variables were declared extern in the libpolyxmass-globals.h.
 */
GPtrArray* pxmchem_atom_GPA;
gchar* libpolyxmass_globals_atom_num_format;
gchar* libpolyxmass_globals_monomer_num_format;
gchar* libpolyxmass_globals_oligomer_num_format;
gchar* libpolyxmass_globals_polymer_num_format;

gchar *etc_polyxmass_data_cfg_file;
gchar *etc_polchem_defs_cat;
gchar *etc_atom_defs_cat;
gchar *etc_polchem_defs_atom_defs_dic;




  /* NUMERAL-FORMATTING FUNCTIONS.
   */
/* Numerical format setting functions for the different weighing species:
 */
gchar*
libpolyxmass_globals_set_atom_num_format (gchar* fmt)
{
  g_assert (fmt != NULL);
  
  if (libpolyxmass_globals_atom_num_format != NULL)
    g_free (libpolyxmass_globals_atom_num_format);
  
  libpolyxmass_globals_atom_num_format = g_strdup (fmt);
  
  return libpolyxmass_globals_atom_num_format;
}

      
gchar*
libpolyxmass_globals_set_monomer_num_format (gchar* fmt)
{
  g_assert (fmt != NULL);
  
  if (libpolyxmass_globals_monomer_num_format != NULL)
    g_free (libpolyxmass_globals_monomer_num_format);
  
  libpolyxmass_globals_monomer_num_format = g_strdup (fmt);
  
  return libpolyxmass_globals_monomer_num_format;
}

      
gchar*
libpolyxmass_globals_set_oligomer_num_format (gchar* fmt)
{
  g_assert (fmt != NULL);
  
  if (libpolyxmass_globals_oligomer_num_format != NULL)
    g_free (libpolyxmass_globals_oligomer_num_format);
  
  libpolyxmass_globals_oligomer_num_format = g_strdup (fmt);
  
  return libpolyxmass_globals_oligomer_num_format;
}

     
gchar*
libpolyxmass_globals_set_polymer_num_format (gchar* fmt)
{
  g_assert (fmt != NULL);
  
  if (libpolyxmass_globals_polymer_num_format != NULL)
    g_free (libpolyxmass_globals_polymer_num_format);
  
  libpolyxmass_globals_polymer_num_format = g_strdup (fmt);
  
  return libpolyxmass_globals_polymer_num_format;
}





/* STRING MANIPULATION FUNCTIONS
 */
/* Attention the *str gets freed in the process! This removes all
   standard isspace ()-matched characters (tab, space, newline... see
   man isspace ()).
 */
gchar *
libpolyxmass_globals_unspacify_string (gchar **str)
{
  gint iter = 0;

  GString * gs = NULL;

  g_assert (str != NULL);
  g_assert (*str != NULL);
  
  /* Make a local comfortable copy to work on.
   */
  gs = g_string_new (*str);

  /* Free the original which we do not need any more
   */
  g_free (*str);

  /* Now iterate in the local copy to see if there are
   * spaces, if so just eliminate them.
   */
  for (iter = 0; iter < strlen (gs->str); iter++)
    {
      if ( 0 != g_ascii_isspace (gs->str[iter]))
	{
	  gs = g_string_erase (gs, iter, 1);
	  iter--;
	}
    }
	   
  *str = gs->str;

  g_string_free (gs, FALSE);

  return *str;
}

gint
libpolyxmass_globals_parse_masses_in_text (gchar *text,
				      GPtrArray *GPA)
{
  gint iter = 0;
  gint count = 0;
  
  gchar *text_local = NULL;
  
  gdouble mass_test = 0;
  gdouble *mass = NULL;
  
  gchar **masses = NULL;
  gchar *temp = NULL;
  gchar *unspacified_text = NULL;
  
  
  g_assert (text != NULL);
  g_assert (GPA != NULL);
  
  if (strlen (text) <= 0)
    return 0;
  
  /* Make a local copy of the text.
   */
  text_local = g_strdup (text);
  
  /* The text that we get may contain tabs, newlines, ... We actually
     need only the following characters: "+-0123456789., ". Each time
     a character is not found in the string above, it is replaced using
     a space character.
  */

#if 0
  /* We first want to replace each newline with a space so that we can
     later split the whole text using ' ' as a delimiter.
   */
  while (text_local [iter] != '\x0')
    {
      if (text_local [iter] == '\n')
	text_local [iter] = ' ';
      
      iter++;
    }
#endif

  text_local = g_strcanon (text_local, "+- 0123456789.", ' ');
  g_assert (text_local != NULL);
  
  /* Split the text_local using the ' ' as the delimiter.
   */
  masses = g_strsplit (text_local, " ", 0);
  
  /* Iterate in the array of strings generated and for each try
     to make a gdouble out of it.
  */
  iter= 0;

  while (masses [iter] != NULL)
    {
      /* We first have to make sure we remove all the spaces from the
	 string that we are analyzing.
      */
      temp = g_strdup (masses [iter]);
      
      /* 
	 temp gets freed in the function below. Note that
	 unspacified_text must be freed also once done, because it is
	 allocated in the function below.
      */
      unspacified_text = libpolyxmass_globals_unspacify_string (&temp);
      
      if (strlen (unspacified_text) > 0)
	{
	  /* There is still something in the string after
	     unspacification.
	   */
	  if (FALSE == libpolyxmass_globals_strtod (unspacified_text, 
						    &mass_test))
	    {
	      /* Failed converting the string to a gdouble.
	       */
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
		    _("%s@%d: failed to convert from '%s' to gdouble\n"),
		     __FILE__, __LINE__, unspacified_text);
	      
	      /* Go on anyway, maybe there are some strings that are worth
		 continuing the conversion work.
	      */
	      iter++;
	      
	      g_free (unspacified_text);
	      
	      continue;
	    }
	  
	  /* It's OK, we could convert the unspacified string into a gdouble,
	     so we can faithfully go on with the work.
	  */
	  g_free (unspacified_text);
	  
	  mass = g_malloc0 (sizeof (gdouble));
	  *mass = mass_test;
	  
	  g_ptr_array_add (GPA, mass);
	  
	  count++;
	  iter++;
	}
      /* End of 
	 if (strlen (unspacified_text) > 0)
      */
      else
	{
	  iter++;
	  
	  g_free (unspacified_text);
	  
	  continue;
	}
    }
  /* End of 
     while (masses [iter] != NULL)
  */

  /* We have finished parsing all the mass strings.
   */
  g_strfreev (masses);
  
  return count;
}






/* XML-format TRANSACTIONS
 */
gchar*
libpolyxmass_globals_format_string_lead (gchar *indent, gint offset)
{
  GString *gs = NULL;
  gchar *help = NULL;
  
  gint iter = 0;
  
  g_assert (indent != NULL);

  gs = g_string_new ("");

  if (strlen (indent) <= 0 || offset <= 0)
    {
      help = gs->str;
      g_string_free (gs, FALSE);
      
      return help;
    }
  
  for (iter = 0; iter < offset; iter++)
    gs = g_string_append (gs, indent);
  
  help = gs->str;
  g_string_free (gs, FALSE);
    
  return help;
}


gboolean
libpolyxmass_globals_check_xml_file (gchar *file)
{
  /* we just want to check that the file starts with the following
   * line : <?xml version="1.0"?>
   */
  FILE *stream = NULL;

  gchar *line = NULL;
  gchar *read = NULL;

  gchar xml_decl [] = "<?xml version=\"1.0\"";

  g_assert (file != NULL);

  stream = fopen (file, "r");

  if (stream == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	_("%s@%d: failed to open file: '%s'\n"),
	     __FILE__, __LINE__, file);

      return FALSE;
    }

  /* Allocate a long memory chunk where the read line is to be
   * stored.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));
  g_assert (line != NULL);

  read = line;

  /* The XML specifications stipulates that the xml declaration has to be
     the first line in the xml document.
  */

  read = fgets (read, MAX_LINE_LENGTH, stream);

  if (read != NULL && strlen (read) > 1)
    {
      if (strlen (read) > MAX_LINE_LENGTH)
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
	       _("%s@%d: the line read from file cannot be larger"
		 " than the authorized length! This is dangerous.\n"),
	       __FILE__, __LINE__);
      
      /* Something was read from file, test if the line contains 
	 an xml declaration.
      */
      if (strstr (read, xml_decl))
	{
	  g_free (line);
	  
	  fclose (stream);
	  
	  return TRUE;
	}
    }
  
  g_free (line);
  
  fclose (stream);
  
  return FALSE;
}




/* FILE COPYING UTILITIES
 */
gboolean 
libpolyxmass_globals_copy_files (gchar *dest, gchar *src)
{
  FILE *destp = NULL;
  FILE *srcp = NULL;

  gchar *line = NULL;
  gchar *read = NULL;

  gint result = EOF;
  

  g_assert (dest != NULL && src != NULL);

  if (strlen (src) == 0 || strlen (dest) == 0)
    return FALSE;

  srcp = fopen (src, "r");

  if (srcp == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	 _("%s@%d: failed to  open source file: '%s'\n"),
	 __FILE__, __LINE__, src);

      if (srcp != NULL)
	fclose (srcp);
      
      return FALSE;
    }

  destp = fopen (dest, "w");

  if (destp == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	 _("%s@%d: failed to  open destination file: '%s'\n"),
	 __FILE__, __LINE__, dest);

      if (destp != NULL)
	fclose (destp);
      
      return FALSE;
    }

  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));
  g_assert (line != NULL);

  read = line;

  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, srcp);
      
      if (read != NULL)
	{
	  if (strlen (read) > MAX_LINE_LENGTH)
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
		   _("%s@%d: the line read from file cannot be larger"
		     " than the authorized length! This is dangerous.\n"),
		   __FILE__, __LINE__);
	  
	  result = fputs (read, destp);
	  
	  if (result == EOF || result < 0)
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to write to file: '%s'\n"),
		     __FILE__, __LINE__, dest);

	      g_free (line);
	      
	      return FALSE;
	    }
	}
      else
	break;
    }

  g_free (line);

  fclose (destp);
  fclose (srcp);

  return TRUE;
}






/* NUMERICAL <--> STRING conversions.
 */
gboolean
libpolyxmass_globals_strtod (gchar* str, gdouble* val)
{
  gchar *endptr = NULL;
  gdouble doubleval;
  
  g_assert (str != NULL);
  g_assert (val != NULL);


  if (strlen (str) <= 0)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING,
	_("%s@%d: string to convert is empty\n"),
	 __FILE__, __LINE__);

      return FALSE;
    }

  /* Check that ALL characters of the string to be converted
   * are ciphers or one dot or one - or one +:
   */
  if (strlen (str) > strspn (str, "0123456789.+-"))
    return FALSE;
  
  /* next check that there is only one of the '.', '+' and '-'
   * characters
   */
  if (strchr (str, '.') != strrchr (str, '.'))
    return FALSE;
  
  if (strchr (str, '+') != strrchr (str, '+'))
    return FALSE;
  
  if (strchr (str, '-') != strrchr (str, '-'))
    return FALSE;


  /* We first apply a conversion to a long, and then, using the returned
   * value we can test if it is higher or lower than the gint can
   * tolerate.
   */
  errno = 0;
  
  doubleval = g_ascii_strtod (str, &endptr);

  if (doubleval == 0.0)
    {
      if (errno == ERANGE)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to convert from '%s' to gdouble, value too small\n"),
		 __FILE__, __LINE__, str);
	  
	  return FALSE;
	}
    }
  else if (doubleval == +HUGE_VAL)
    {
      if (errno == ERANGE)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to convert from '%s' to gdouble, overflow occurred: value too positively big\n"),
		 __FILE__, __LINE__, str);

	  return FALSE;
	}
    }
  else if (doubleval == -HUGE_VAL)
    {
      if (errno == ERANGE)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to convert from '%s' to gdouble, underflow occurred, value too negatively big\n"),
		 __FILE__, __LINE__, str);

	  return FALSE;
	}
    }

  if (strlen (endptr) > 0)
    return FALSE;
  
  *val = doubleval;
  
  return TRUE;
}




gboolean
libpolyxmass_globals_strtoi (gchar *str, gint* val, gint base)
{
  glong long_val = 0;

  gchar *endptr = NULL;
  
  g_assert (str != NULL);
  g_assert (val != NULL);


  if (strlen (str) <= 0)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING,
	_("%s@%d: string to convert is empty\n"),
	 __FILE__, __LINE__);

      return FALSE;
    }

  /* Check that ALL characters of the string to be converted
   * are ciphers or one dot or one - or one +:
   */
  if (strlen (str) > strspn (str, "0123456789.+-"))
    return FALSE;
  
  /* next check that there is only one of the '.', '+' and '-'
   * characters
   */
  if (strchr (str, '.') != strrchr (str, '.'))
    return FALSE;
  
  if (strchr (str, '+') != strrchr (str, '+'))
    return FALSE;
  
  if (strchr (str, '-') != strrchr (str, '-'))
    return FALSE;

  /* We first apply a conversion to a long, and then, using the returned
   * value we can test if it is higher or lower than the gint can
   * tolerate.
   */
  errno = 0;
  
  long_val = strtol (str, &endptr, base);

  if (errno != 0) 
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: failed to convert from '%s' to gint, overflow\n"),
	     __FILE__, __LINE__, str);
      
      return FALSE;
    }

  if (endptr != NULL && strlen (endptr) > 0)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to convert from '%s' to gint\n"
	     "with endptr='%s' (endptr[0]=%c) of length: '%zu'"
	     " and long_val='%ld'.\n"),
	     __FILE__, __LINE__, str, endptr, endptr[0], 
	     strlen (endptr), long_val);
      
      return FALSE;
    }

  /* Now make sure that the glong value is not an error for the gint
     type !
   */
  if (long_val >= G_MAXINT || long_val <= G_MININT)
    {
      /* Not in the gint range.
       */
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: failed to convert from '%s' to gint, not in the gint range\n"),
	     __FILE__, __LINE__, str);
      
      return FALSE;
    }
  
  *val = long_val;
  
  return TRUE;
}


gchar *
libpolyxmass_globals_dtoa (gdouble val, gint prec)
{
  gchar *format = NULL;
  gchar *convert = NULL;

  /* Construct a format string like the ones used with printf()
   */
  format = g_strdup_printf ("%%.%df", prec);

  convert = g_strdup_printf (format, val);

  g_free (format);

  return convert;
}



/* HELPER FUNCTIONS
 */
gint
libpolyxmass_globals_locate_string_in_array (gchar *string,
					     GPtrArray *GPA)
{
  gint iter = 0;
  
  g_assert (string != NULL);
  g_assert (GPA != NULL);
  
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      if (0 == strcmp (g_ptr_array_index (GPA, iter), string))
	return iter;
    }

  return -1;
}



/* Function that will put in the GArray (of gint values) passed as
   parameter all the gint values that correspond to a string of 
   text (exp). The max parameter is the gint value that cannot be
   surpassed by found gint values.
*/
gint 
libpolyxmass_globals_fill_int_array_match_exp (gchar *exp, GArray *GA,
					   gint max)
{
  /* We get a string in the form:
     "3;57;1002-1017;[1312-1715,odd];[755-855];[1724-1937,even]" or
     simpler. But can also be: "even" or "odd" or "all". From this
     string and a maximum gint value, we'll have to make an array
     containing all the gint values that are described by the expression.
  */
  gchar **sub_exp = NULL;
  gint iter = 0;

  gint res = 0;
  gint count = 0;

  g_assert (exp != NULL);
  g_assert (GA != NULL);

  if (strlen (exp) <= 0)
    return 0;

  /* the very first step is to divide the string into a number
   * of string delimited with ';'.
   */

  sub_exp = g_strsplit (exp, ";", 0);

  /* following the example above, we now have an array of strings
   * like this:
   *
   * [0] = "3"
   * [1] = "57"
   * [2] = "1002-1017"
   * [3] = "[1312-1715,odd]"
   * [4] = "[755-855]"
   * [5] = "[1724-1937,even]"
   *
   * We now have to parse each single string.
   */

  while (sub_exp [iter] != NULL)
    {
      res = libpolyxmass_globals_match_int_subexp (sub_exp [iter],
					      GA, max);
      if (res == -1)
	return -1;
      
      count += res;
      iter++;
    }

  return count;
}


gint
libpolyxmass_globals_match_int_subexp (gchar *exp, GArray *GA, gint max)
{
  gchar * pos_dash = NULL;
  gchar * pos_open_bracket = NULL;
  gchar * pos_close_bracket = NULL;

  gchar * local = NULL;

  g_assert (exp != NULL);
  g_assert (GA != NULL);

  /* below are some examples of valid substrings:
   * [0] = "3"
   * [1] = "57"
   * [2] = "1002-1017"
   * [3] = "[1312-1715,odd]"
   * [4] = "[755-855]"
   * [5] = "[1724-1937,even]"
   */

  if (strlen (exp) <= 0)
    return 0;

  /* Start with some easy checks:
   */

  /* First of all if no '[' nor '-' are found, that means that the
     string is a simple cipher/number: "3" or "57". Thus, we make the
     conversion, append to the array, and return 1.
   */
  pos_open_bracket = strchr (exp, '[');
  pos_close_bracket = strchr (exp, ']');
  pos_dash= strchr (exp, '-');

  if (NULL == pos_open_bracket)
    {
      /* If no '[' was found, it is not correct that a ']' be found.
       */
      if (NULL != pos_close_bracket)
	return -1;

      /* OK, none of the '[' and ']' chars are there. Coherent.
       */
      /* We now should be careful because there are still a number of
	 possibilities :
	 
	 We might have either a single cipher/number, like 1 or 1245,
	 or we might have a dashed cipher/number pair, like 1-12, or
	 we might have non-numerical strings (see below).
       */
      if (NULL == pos_dash)
	{
	  /* No '-' char, which means we have a simple cipher/number
	     or a special string (all | ALL | even | EVEN | odd | ODD)
	     -or whatever uppercase/lowercase variant- here.
	   */
	  local = g_ascii_strup (exp, -1); /* local is allocated. */

	  if (strstr (local, "ALL") == local)
	    {
	      /* The string just starts with "ALL".
	       */
	      g_free (local);
	      
	      return libpolyxmass_globals_match_int_all_subexp (GA, max);
	    }

	  else if (strstr (local, "EVEN") == local)
	    {
	      /* The string just starts with "EVEN".
	       */
	      g_free (local);
	      
	      return libpolyxmass_globals_match_int_even_subexp (GA, max);
	    }

	  else if (strstr (local, "ODD") == local)
	    {
	      /* The string just starts with "ODD"
	       */
	      g_free (local);
	      
	      return libpolyxmass_globals_match_int_odd_subexp (GA, max);
	    }

	  /* If none of the authorized strings, and no '-', then
	     should be a number/cipher like 1 or 12 or 1458 whatever.
	   */
	  else
	    {
	      g_free (local);
	      
	      return libpolyxmass_globals_match_int_single_value_subexp (exp, 
								     GA, 
								     max);
	    }
	}
      /* end case for which no '-' was found.
       */
      else
	{
	  /* There is a '-' character, which means that we are in the
	     presence of an unbracketted dashed value, like 1002-1017,
	     and there is no EVEN nor ODD nor ALL single criterion.
	   */
	  return  
	    libpolyxmass_globals_match_int_bracketted_dashed_values_subexp 
	    (exp, 
	     GA, 
	     max);
	}
    }

  /* SO, we have finished dealing with the case where no
     '[' nor '-' nor ']' were encountered.
   */

  /* If we are here, that means that a '[' was found. Check that
     there is a ']' and a '-'. If not, that's an error.
   */
  if (NULL == pos_close_bracket || NULL == pos_dash)
    return -1;

  /* Since we know we have '[' and '-' and ']', we already can assume
     that '[' should be the very first character and ']' the last
     one. This assumption is true because the string passed to this
     function should have been unspacified already (unspacify_string
     ()). Check this.
   */
  if (exp [0] != '[')
    return -1;
  if (exp [strlen (exp) -1] != ']')
    return -1;

  /* OK, so now, we know for sure that we have a string of this form:
     "[xxxxxxx]". What we do not know is what xxxxxxx is but this is
     going to be the work of the function we call now:
   */
  return libpolyxmass_globals_match_int_bracketted_dashed_values_subexp 
    (exp, 
     GA, 
     max);
  
}


gint
libpolyxmass_globals_match_int_single_value_subexp (gchar * exp, 
						GArray *GA, gint max)
{
  gint converted = 0;

  g_assert (exp != NULL && GA != NULL);
  
  if (TRUE != libpolyxmass_globals_strtoi (exp, &converted, 10))
    return -1;

  /* since the user speaks of positions and we internally do work
   * on indices, we just decrement the converted value by one
   * before going on with checks and appending to the array:
   */
  converted--;
  /* we should make sure that the converted value is not
   * greater than the max accepted index value:
   */
  if (converted > max)
    return -1;

  /* apparently the conversion went ok, just append
   * the new value to the array and return 1, since we
   * append only one value to the array.
   */
  GA = g_array_append_val (GA, converted);

  return 1;
}


gint
libpolyxmass_globals_match_int_bracketted_dashed_values_subexp (gchar * exp, 
								GArray *GA, 
								gint max)
{
  gint iter = 0;

  GString * gs_curval = NULL;

  gint leftval = 0;
  gint rightval = 0;
  gint momentval = 0;

  /* used to guess if number is even or odd
   */
  div_t divres ;

  gboolean found_dash = FALSE;
  gboolean found_left_bracket = FALSE;
  gboolean found_right_bracket = FALSE;
  gboolean found_comma = FALSE;

  gint even_odd_all = ALL;
  gboolean even_odd_all_set = FALSE;

  gboolean leftvalDone = FALSE;
  gboolean rightvalDone = FALSE;

  /* we get something in the form "1002-1017".
   */
  // 1; 53; 104; 200-220; [ 250-300,even]; [700-800,odd]; [ 900-1000 ]

  g_assert (exp != NULL && GA != NULL);

  gs_curval = g_string_new ("");

  for (iter = 0; iter < strlen (exp); iter++)
    {
      /* first test if the character parsed is not a 0-9 digit.
       */
      if (0 == isdigit (exp[iter]))
	{
	  /* if it is not a digit, it can be only a '-' ,
	   * a '[' or a ']' because the string passed here should 
	   * have been previously unspacified (unspacify_string ());
	   * otherwise there is a syntax error.
	   */
	  if ('[' == exp[iter])
	    {
	      if (found_left_bracket == TRUE)
		/* not possible that 2 '[' be found here
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	     
	      else
		{
		  found_left_bracket = TRUE;
		  continue;
		}
	    }
	  /* end '[' was found
	   */

	  else if ('-' == exp[iter])
	    {
	      if (found_dash == TRUE)
		/* not possible that a '-' be found more than one time
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      
	      if (strlen (gs_curval->str) <= 0)
		/* not possible that no single digit was found left
		 * of this '-'. That would mean that the string passed
		 * as parameter would start with '-'. Error condition.
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      /* we have already finished parsing the left value string.
	       * Convert it to a number, and truncate the GString for
	       * use for the rightval parsing...
	       */
	      if (TRUE != 
		  libpolyxmass_globals_strtoi (gs_curval->str, &leftval, 10))
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      
	      found_dash = TRUE;
	      
	      gs_curval = g_string_truncate (gs_curval, 0);
	      leftvalDone = TRUE;
	    }
	  /* end '-' was found
	   */

	  else if (',' == exp[iter])
	    {
	      /* we manifestly have finished now parsing the rightval
	       * string. Make some sanity checks:
	       */
	      if (leftvalDone == FALSE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      if (rightvalDone == TRUE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}

	      found_comma = TRUE;

	      /* so for the moment we do not deal with the numerical
	       * representation of the rightval string that was prepared
	       * until now, because we want to know if the next
	       * character is a (all) or e (even) or o (odd) so that
	       * we can continue the work out of this loop.
	       */
	      continue;
	    }

	  else if ('a' == exp[iter] || 'A' == exp[iter])
	    {
	      /* sanity checks, the comma should have been found
	       * already and, more important, we should not have
	       * already set the mode ! 
	       */
	      if (found_comma == FALSE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      if (even_odd_all_set == TRUE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}

	      /* evidently the operation is required for both
	       * the odd and even numerical values.
	       */
	      even_odd_all = ALL;
	      even_odd_all_set = TRUE;

	      /* at this stage, we know that we have all the info
	       * required to terminate the work outside of the
	       * loop.
	       */
	      break;
	    }

	  else if ('e' == exp[iter] || 'E' == exp[iter])
	    {
	      /* sanity checks, the comma should have been found
	       * already and, more important, we should not have
	       * already set the mode ! 
	       */
	      if (found_comma == FALSE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      if (even_odd_all_set == TRUE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      
	      /* evidently the operation is required for both
	       * the odd and even numerical values.
	       */
	      even_odd_all = EVEN;
	      even_odd_all_set = TRUE;

	      /* at this stage, we know that we have all the info
	       * required to terminate the work outside of the
	       * loop.
	       */
	      break;
	    }

	  else if ('o' == exp[iter] || 'O' == exp[iter])
	    {
	      /* sanity checks, the comma should have been found
	       * already and, more important, we should not have
	       * already set the mode ! 
	       */
	      if (found_comma == FALSE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      if (even_odd_all_set == TRUE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      
	      /* evidently the operation is required for both
	       * the odd and even numerical values.
	       */
	      even_odd_all = ODD;
	      even_odd_all_set = TRUE;

	      /* at this stage, we know that we have all the info
	       * required to terminate the work outside of the
	       * loop.
	       */
	      break;
	    }

	  /* we reach this point only if the ",even" or ",odd" or
	   * ",all" element in [xxxx-yyyy,zzz] is not existent,
	   * which means that we just have to make some sanity
	   * checks and to break the loop after having set the
	   * even_odd_all to the default ALL value.
	   */
	  else if (']' == exp[iter])
	    {
	      if (found_right_bracket == TRUE)
		/* not possible that 2 ']' be found here
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      
	      found_right_bracket = TRUE;

	      /* we are evidently closing the [xxxxxxxxx]
	       * string. Depending on what is in between the
	       * two brackets we might have finished already
	       * with the left and right values. We need to 
	       * make some checks.
	       */
	      if (found_comma == TRUE)
		/* it is not possible that we still are in this
		 * loop if the comma had been already found.
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	    
	      if (even_odd_all_set == TRUE)
		{
		  g_string_free (gs_curval, TRUE);
		  return -1;
		}
	      
	      even_odd_all = ALL;
	      even_odd_all_set = TRUE;
	      
	      break;
	    }
	  /* end ']' was found
	   */

	  else
	    /* there is a syntax error
	     */
	    {
	      g_string_free (gs_curval, TRUE);
	      return -1;
	    }
	}
      /* end of case in which the character is not a digit.
       */

      else
	{
	  /* the parsed character is a digit.
	   */
	  gs_curval = 
	    g_string_append_c (gs_curval, exp[iter]);
	      
	  continue;
	}
    }
  /* end of iteration in the string that was passed as parameter,
   * which means that gs_curval should contain the second value
   * string, the string that was after the '-' character. We have
   * to deal with this !
   */

  if (found_dash == FALSE)
    {
      /* not possible that we did not find a '-' char after parsing
       * the whole string !
       */
      if (gs_curval != NULL)
	g_string_free (gs_curval, TRUE);
      return -1;
    }

  if (leftvalDone == FALSE)
    {
      /* not possible that we did not finish parsing the left
       * value !
       */
      if (gs_curval != NULL)
	g_string_free (gs_curval, TRUE);
      return -1;
    }
  
  if (strlen (gs_curval->str) <= 0)
    {
      /* error if the second val is of zero length. We should
       * have an interval string, like "1002-1017".
       */
      if (gs_curval != NULL)
	g_string_free (gs_curval, TRUE);
      return -1;
    }
  else
    {
      /* we should convert the second string into the second
       * numerica value.
       */
      if (TRUE != libpolyxmass_globals_strtoi (gs_curval->str, &rightval, 10))
	{
	  g_string_free (gs_curval, TRUE);
	  return -1;
	}
    }
  
  /* OK, we now have the two values, leftval and rightval, we can
   * free gs_curval, since we do not need it anymore.
   */
  if (gs_curval != NULL)
    g_string_free (gs_curval, TRUE);

  /* check that both are lower than the max index authorized.
   */

  /* DO NO T FORGET THAT THE PASSED VALUES ARE POSITIONS BUT
   * THAT WE SHOULD WORK ON INDICES...
   */

  if (leftval - 1 > max || rightval - 1 > max)
    return -1;
  
  if (leftval -1 < 0 || rightval - 1 < 0)
    return -1;
  

  /* put the value in "order" so that leftval is lower than 
   * rightval.
   */
  if (leftval > rightval)
    {
      momentval = leftval;
      leftval = rightval;
      rightval = momentval;
    }

  /* change from position to index.
   */
  leftval--; /* from 1002 to 1001 */
  rightval--; /* from 1017 to 1016 */

  /* recycle momentval as a counter !
   */
  momentval = 0;

  for (iter = leftval; iter < rightval + 1; iter++)
    {
      /* in our example this would be iteration from
       * 1001 to 1016.
       */
      /* for each value, append its value to the array, but remember
       * we have to know if the EVEN/ODD criterion was asked.
       */
      if (even_odd_all == ALL)
	{
	  GA = g_array_append_val (GA, iter);
	  momentval++;
	}

      else 
	{
	  divres = div (iter, 2);

	  if (even_odd_all == ODD && divres.rem == 0)
	    {
	      /* only insert if iter is an EVEN value (do not forget
	       * the user asks for ODD positions, which turn out to be
	       * EVEN indices, and we are now working on indices,
	       * remember the leftval-- and rightval-- 
	       * instructions above? If the div.rem(ainder) is 0
	       * that means that the numerator (iter) was EVEN !
	       */
	      GA = g_array_append_val (GA, iter);
	      momentval++;
	    }
	  else if (even_odd_all == EVEN && divres.rem != 0)
	    {
	      /* only insert if iter is an ODD value (do not forget
	       * the user asks for EVEN positions, which turn out to be
	       * ODD indices, and we are now working on indices,
	       * remember the leftval-- and rightval-- 
	       * instructions above? If the div.rem(ainder) is not 0
	       * that means that the numerator (iter) was ODD !
	       */
	      GA = g_array_append_val (GA, iter);
	      momentval++;
	    }
	  else
	    continue;
	}
    }

  /* we finished doing the job, momentval now stores the number
   * of numerical values that we appended to the array.
   */
  return momentval;
}


gint
libpolyxmass_globals_match_int_all_subexp (GArray *GA, 
					   gint max)
{
  gint iter = 0;

  g_assert (GA != NULL);

  for (iter = 0; iter < max + 1; iter++)
    GA = g_array_append_val (GA, iter);
  
  return iter;
}


gint
libpolyxmass_globals_match_int_even_subexp (GArray *GA, 
					    gint max)
{
  /* remembet that if the user says he wants all the even
   * positions, that means that he wants all the odd indices,
   * and since we work on indices internally we need to 
   * be careful with this.
   */
  gint iter = 0;

  g_assert (GA != NULL);

  for (iter = 1; iter < max + 1; iter = iter + 2)
    GA = g_array_append_val (GA, iter);
  
  return (iter/2);

}

gint
libpolyxmass_globals_match_int_odd_subexp (GArray *GA, 
					   gint max)
{
  /* remembet that if the user says he wants all the odd
   * positions, that means that he wants all the even indices,
   * and since we work on indices internally we need to 
   * be careful with this.
   */
  gint iter = 0;

  g_assert (GA != NULL);

  for (iter = 0; iter < max + 1; iter = iter + 2)
    GA = g_array_append_val (GA, iter);
  
  return (iter/2);

}

gboolean
libpolyxmass_globals_parse_range_coordinates (gchar *coords,
					      gint *start, gint *end)
{
  gint iter = 0;
  
  GString * gs_curval = NULL;

  gboolean found_dash = FALSE;
  gboolean found_left_bracket = FALSE;
  gboolean found_right_bracket = FALSE;

  gboolean leftvalDone = FALSE;


  g_assert (coords != NULL);
  g_assert (start != NULL);
  g_assert (end != NULL);
  
  
  /* The string we get is of the form [12-125], so we have to
     find the first numerical datum (12) and the second one (125).
  */

  gs_curval = g_string_new ("");

  for (iter = 0; iter < strlen (coords); iter++)
    {
      /* first test if the character parsed is not a 0-9 digit.
       */
      if (0 == isdigit (coords [iter]))
	{
	  /* if it is not a digit, it can be only a '-' , a '[' or a
	     ']' because the string passed here should have spaces in
	     it; otherwise there is a syntax error.
	   */
	  if ('[' == coords [iter])
	    {
	      if (found_left_bracket == TRUE)
		/* not possible that 2 '[' be found here
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return FALSE;
		}
	     
	      else
		{
		  found_left_bracket = TRUE;
		  continue;
		}
	    }
	  /* end '[' was found
	   */

	  else if ('-' == coords [iter])
	    {
	      if (found_dash == TRUE)
		/* not possible that a '-' be found more than one time
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return FALSE;
		}
	      
	      if (strlen (gs_curval->str) <= 0)
		/* Not possible that no single digit was found left of
		   this '-'. That would mean that the string passed as
		   parameter would start with '-'. Error condition.
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return FALSE;
		}

	      /* We have already finished parsing the left value
		 string.  Convert it to a number, and truncate the
		 GString for use for the rightval parsing...
	       */
	      if (TRUE != 
		  libpolyxmass_globals_strtoi (gs_curval->str, start, 10))
		{
		  g_string_free (gs_curval, TRUE);
		  return FALSE;
		}
	      
	      found_dash = TRUE;
	      
	      gs_curval = g_string_truncate (gs_curval, 0);
	      leftvalDone = TRUE;
	    }
	  /* end '-' was found
	   */

	  else if (']' == coords [iter])
	    {
	      if (found_right_bracket == TRUE)
		/* not possible that 2 ']' be found here
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return FALSE;
		}

	      found_right_bracket = TRUE;

	      /* we are evidently closing the [xxxx-xxxxx]
		 string. We should have already a leftvalue and also have
		 something in gs_curval, that should be the right value.
	       */
	      if (FALSE == leftvalDone)
		{
		  g_string_free (gs_curval, TRUE);
		  return FALSE;
		}
	      
	      if (strlen (gs_curval->str) <= 0)
		/* Not possible that no single digit was found left of
		   this ']'. That would mean that the string passed as
		   parameter would not contain numerical digits left
		   of ']'. Error condition.
		 */
		{
		  g_string_free (gs_curval, TRUE);
		  return FALSE;
		}

	      /* We have already finished parsing the right value
		 string.  Convert it to a number, and free the GString
		 for we do not need it anymore.
	      */
	      if (TRUE != 
		  libpolyxmass_globals_strtoi (gs_curval->str, end, 10))
		{
		  g_string_free (gs_curval, TRUE);
		  return FALSE;
		}
	      
	      /* At this point we really can return, as we have
		 successfully parsed both the start and end integer
		 values.
	       */
	      g_string_free (gs_curval, TRUE);
	      return TRUE;
	    }
	  /* end ']' was found
	   */

	  else
	    /* there is a syntax error
	     */
	    {
	      g_string_free (gs_curval, TRUE);
	      return FALSE;
	    }
	  
	}
      /* end of case in which the character is not a digit.
       */

      else
	{
	  /* the parsed character is a digit.
	   */
	  gs_curval = 
	    g_string_append_c (gs_curval, coords [iter]);
	      
	  continue;
	}
    }
  
  /* OK, we now have the two values, leftval and rightval, we can
   * free gs_curval, since we do not need it anymore.
   */
  g_assert (gs_curval != NULL);
  g_string_free (gs_curval, TRUE);

  return TRUE;
}
