#include "osl/container/moveVector.h"
#include "osl/hash/hashKey.h"
#include "osl/state/numEffectState.h"
#include "osl/record/ki2.h"
#include "osl/record/kisen.h"
#include "osl/record/kakinoki.h"
#include "osl/record/csaRecord.h"
#include "osl/record/checkDuplicate.h"

#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/program_options.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/foreach.hpp>
#include <boost/format.hpp>
#include <deque>
#include <exception>
#include <iostream>
#include <fstream>
#include <tr1/unordered_map>

std::vector<std::string> good_tournaments;
bool accept_tournament(const std::string& name) 
{
  if (good_tournaments.empty())
    return true;
  for (size_t i=0; i<good_tournaments.size(); ++i)
    if (good_tournaments[i].find(name) == 0)
      return true;
  return false;
}

static void convert(const std::string &kisen_filename,
		    const std::vector<std::string> &files,
		    bool output_ipx,
                    osl::record::CheckDuplicate& check_duplicates,
		    int default_rating)
{
  std::ofstream ofs(kisen_filename.c_str());
  osl::record::OKisenStream ks(ofs);

  boost::scoped_ptr<osl::record::KisenIpxWriter> ipx_writer;
  boost::scoped_ptr<std::ofstream> ipx_ofs;
  if (output_ipx)
  {
    const boost::filesystem::path ipx_path =
      boost::filesystem::change_extension(boost::filesystem::path(kisen_filename), ".ipx");
    const std::string ipx = ipx_path.file_string();
    ipx_ofs.reset(new std::ofstream(ipx.c_str()));
    ipx_writer.reset(new osl::record::KisenIpxWriter(*ipx_ofs));
  }

  for (size_t i = 0; i < files.size(); ++i)
  {
    try
    {
      osl::record::Record record;
      const std::string& filename = files[i];
      if (boost::algorithm::iends_with(filename, ".csa"))
      {
        const osl::record::csa::CsaFile csa(filename);
        record = csa.getRecord();
      }
      else if (boost::algorithm::iends_with(filename, ".ki2"))
      {
        const osl::Ki2File ki2(filename);
        record = ki2.getRecord();
	if (! accept_tournament(record.tounamentName()))
	  continue;
      }
      else if (boost::algorithm::iends_with(filename, ".kif"))
      {
        const osl::KakinokiFile kif(filename);
        record = kif.getRecord();
      }
      else
      {
        std::cerr << "Unknown file type: " << filename << "\n";
        continue;
      }

      // 重複チェック 
      const osl::vector<osl::Move>& moves = record.getMoves();
      if (check_duplicates.regist(moves))
	continue;

      ks.save(&record);
      if (output_ipx)
      {
        ipx_writer->save(record, default_rating, default_rating, "", "");
      }
    }
    catch(std::exception& e)
    {
      std::cerr << "ERROR: reading " <<  files[i] << "; " << 
        e.what() << std::endl;
      continue;
    }
  }
}

int main(int argc, char **argv)
{
  bool output_ipx;
  std::string kisen_filename, tournament_filename;
  int default_rating;
  boost::program_options::options_description command_line_options;
  command_line_options.add_options()
    ("output-ipx",
     boost::program_options::value<bool>(&output_ipx)->default_value(true),
     "Whether output IPX file in addition to KIF file")
    ("tournament-file", boost::program_options::value<std::string>(&tournament_filename)
     ->default_value(""),
     "ignore records unless the name of their tournament is listed in the file in EUC-JP")
    ("kisen-filename,o",
     boost::program_options::value<std::string>(&kisen_filename)->
     default_value("test.kif"),
     "Output filename of Kisen file")
    ("input-file", boost::program_options::value< std::vector<std::string> >(),
     "input files in kisen format")
    ("default-rating", boost::program_options::value<int>(&default_rating)->
     default_value(0),
     "default rating")
    ("help", "Show help message");
  boost::program_options::variables_map vm;
  boost::program_options::positional_options_description p;
  p.add("input-file", -1);

  try
  {
    boost::program_options::store(
      boost::program_options::command_line_parser(
	argc, argv).options(command_line_options).positional(p).run(), vm);
    boost::program_options::notify(vm);
    if (vm.count("help"))
    {
      std::cerr << "Usage: " << argv[0] << " [options] csa-files | ki2-files \n";
      std::cerr << "       " << argv[0] << " [options]\n";
      std::cout << command_line_options << std::endl;
      return 0;
    }
  }
  catch (std::exception &e)
  {
    std::cerr << "error in parsing options" << std::endl
	      << e.what() << std::endl;
    std::cerr << "Usage: " << argv[0] << " [options] csa-files | ki2-files\n";
    std::cerr << "       " << argv[0] << " [options]\n";
    std::cerr << command_line_options << std::endl;
    return 1;
  }

  if (tournament_filename != "")
  {
    std::ifstream is(tournament_filename.c_str());
    std::string name;
    while(std::getline(is, name))
    {
      boost::algorithm::trim(name);
      good_tournaments.push_back(name);
    }
  }

  std::vector<std::string> files;
  if (vm.count("input-file"))
  {
    const std::vector<std::string> temp = vm["input-file"].as<std::vector<std::string> >();
    files.insert(files.end(), temp.begin(), temp.end());
  }
  else
  {
    std::string line;
    while(std::getline(std::cin , line))
    {
      boost::algorithm::trim(line);
      files.push_back(line);
    }
  }

  osl::record::CheckDuplicate check_duplicate;
  convert(kisen_filename, files, output_ipx, check_duplicate, default_rating);

  std::locale::global(std::locale(""));
  check_duplicate.print(std::cout);

  return 0;
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; coding:utf-8
// ;;; End:
