/* atomicCounter.h
 */
#ifndef OSL_ATOMICCOUNTER_H
#define OSL_ATOMICCOUNTER_H

#include "osl/config.h"
#ifdef USE_TBB_ATOMIC
#  include <tbb/atomic.h>
#else
#  include "osl/misc/lightMutex.h"
#endif
#include <algorithm>

namespace osl
{
  namespace misc
  {
    template <class Counter>
    struct IncrementLock
    {
      Counter& counter;
      explicit IncrementLock(Counter& c) : counter(c) 
      {
	counter.inc();
      }
      ~IncrementLock() 
      {
	counter.dec();
      }
    };
#ifdef USE_TBB_ATOMIC
    class AtomicCounter
    {
      tbb::atomic<int> count;
    public:
      explicit AtomicCounter(int count_=0) {
	this->count=count_;
      }
      void inc(){
	count.fetch_and_increment();
      }
      int valueAndinc(){
	return count.fetch_and_increment();
      }
      void dec(){
	count.fetch_and_decrement();
      }
      void max(int val){
	int x=count;
	if(x<val){
	  int oldx;
	  while((oldx=count.compare_and_swap(val,x))!=x){
	    x=oldx;
	    if(x>=val) break;
	  }
	}
      }
      int value() const{ 
	return count; 
      }
      void setValue(int value) { 
	count = value; 
      }
      typedef IncrementLock<AtomicCounter> IncLock;
    };
#else
    class AtomicCounter
    {
      typedef LightMutex Mutex;
      mutable Mutex m;
      int count;
    public:
      explicit AtomicCounter(int count=0) :count(count){}
      void inc(){
	Mutex::scoped_lock lk(m);
	count++;
      }
      int valueAndinc(){
	Mutex::scoped_lock lk(m);
	return count++;
      }
      void dec(){
	Mutex::scoped_lock lk(m);
	count--;
      }
      void max(int val){
	Mutex::scoped_lock lk(m);
	count=std::max(count,val);
      }
      int value() const{ 
	Mutex::scoped_lock lk(m);
	return count; 
      }
      void setValue(int value) { 
	Mutex::scoped_lock lk(m);
	count = value; 
      }
      typedef IncrementLock<AtomicCounter> IncLock;
    };
#endif
  }
  using misc::AtomicCounter;
}

#endif /* OSL_ATOMICCOUNTER_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:

