%{
/*============================================================================
 * Define the scanner for the mathematical expression
 *============================================================================*/

/*
  This file is part of the "Mathematical Expression Interpreter" library.

  Copyright (C) 2008-2009  EDF

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*----------------------------------------------------------------------------
 * Standard C library headers
 *----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*----------------------------------------------------------------------------
 * BFT library headers
 *----------------------------------------------------------------------------*/

#include <bft_mem.h>
#include <bft_error.h>

/*----------------------------------------------------------------------------
 * Local headers
 *----------------------------------------------------------------------------*/

#include "mei_hash_table.h"
#include "mei_evaluate.h"
#include "mei_parser.h"
#include "mei_parser_glob.h"

/* Redefine macro to redirect scanner input from string instead of standard input. */

#undef YY_INPUT
#define YY_INPUT(b, r, ms) ( r = my_yyinput(b, ms) )

int my_yyinput(char *buffer, int max_size);

char *buf = NULL;
size_t len;

%}

%%
" "             {column++;         } /* skip whitespace */
"\t"            {column+=8;        } /* skip whitespace */
"\n"            {line++; column=0; } /* skip whitespace */
"#".*\n         {line++; column=0; } /* discard commentary (other possibility: "#"[^\n]*"\n" ) */


([0-9]+|[0-9]+\.[0-9]*|[0-9]*\.[0-9]+)([Ee][+\-]?[0-9]+)? {
    yylval.iValue = strtod(yytext, NULL);
    column += yyleng; 
    return NUMBER;
}


"e"|"pi" {
    strncpy(yylval.sIndex, yytext, yyleng+1);
    column += yyleng;
    return VAR;
}


"min"|"max"|"atan2"|"mod" {
    strncpy(yylval.sIndex, yytext, yyleng+1);
    column += yyleng;
    return FUN2;
}

"exp"|"log"|"sqrt"|"sin"|"cos"|"tan"|"asin"|"acos"|"atan"|"sinh"|"cosh"|"tanh"|"abs"|"int" {
    strncpy(yylval.sIndex, yytext, yyleng+1);
    column += yyleng;
    return FUN1;
}

"while"         {column+=5; return WHILE; }
"if"            {column+=2; return IF;    }
"else"          {column+=4; return ELSE;  }
"print"         {column+=5; return PRINT; }


[_a-zA-Z][\-._a-zA-Z0-9]* {
    strncpy(yylval.sIndex, yytext, yyleng+1);
    if (yyleng+1 > 200)
        bft_error(__FILE__, __LINE__, 0, "Error: identifiant is to long\n");
    column += yyleng;
    return VAR;
}


">="             {column+=2; return GE;  }
"<="             {column+=2; return LE;  }
"=="             {column+=2; return EQ;  }
"!="             {column+=2; return NE;  }
"||"             {column+=2; return OR;  }
"&&"             {column+=2; return AND; }
"!"              {column++;  return *yytext;}


[-()<>=+^*/;{},]  {column++; return *yytext;}


. {
    column++;
    len = strlen("unknown character: ") +1;
    BFT_MALLOC(buf, len, char);
    strncpy(buf, "unknown character: ", len);
    BFT_REALLOC(buf, len+yyleng, char);
    strncat(buf, yytext, yyleng);
    yyerror(buf);
    BFT_FREE(buf);
}

%%

int yywrap(void)
{
    return 1;
}


/* The calling sequence for 'YY_INPUT()' is 'YY_INPUT(buf,result,max_size)'.
   Its action is to place up to 'max_size' characters in the character
   array 'buf' and return in the integer variable 'buffer_length' either
   the number of characters read or the constant 'YY_NULL' to indicate 'EOF'.
   The default 'YY_INPUT' reads from the global file-pointer 'yyin'. */

/* max_sise = 8192 bytes. One can change this default
   value with YY_BUF_SIZE macro */


int
my_yyinput(char *buffer, int max_size)
{
    int buffer_length;

    if (max_size < string_end - string_begin)
    {
        buffer_length = max_size;
    }
    else
    {
        buffer_length = string_end - string_begin;
    }

    if (buffer_length > 0)
    {
        memcpy(buffer, string_begin, buffer_length);
        string_begin +=buffer_length;
    }

    return buffer_length;
}
