/*  This file is part of LingoTeach, the Language Teaching program 
 *  Copyright (C) 2001-2003 The LingoTeach Team
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by 
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version.  
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *  GNU General Public License for more details.  
 *
 *  You should have received a copy of the GNU General Public License 
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#ifndef LING_LESSON_H
#define LING_LESSON_H 1

#include <libxml/xpath.h>

#include "lingdefs.h" 

/*
 * lesson_data, lessonData:
 * 
 * stores the available lesson files
 *
 * xmlDocPtr lesson_file is created by xmlParseFile,
 * number is automatically increased
 * meanings is automatically fetched from the file 
 * next points to the next struct containing the next lesson
 * prev points to the previous struct containing the next lesson
 *
 */
typedef struct lesson_data lessonData;
struct lesson_data
{
  lingConfig         *settings; /* settings for the lesson */
  xmlDocPtr           lesson;   /* the xmlDocPtr for parsing, etc. */
  xmlXPathContextPtr  x_path;   /* the xmlXPathContextPtr for the lesson */ 
  int                 number;   /* internal number, starting by 0 */
  lingbool            used;     /* used file */
  int		      meanings; /* the count of meanings of the lesson */
  char               *path;     /* path to the lesson file */
};

/*
 * allocates memory for a new lesson 
 */
lingLesson*
lesson_alloc_new (void);

/*
 * returns a xmlXPathContextPtr of the given xmlDocPtr
 */
xmlXPathContextPtr
lesson_get_xpath (xmlDocPtr doc);

/* 
 * returns the last lesson from the list of used ones
 */
lingLesson*
lesson_get_last (lingLesson *lesson);

/*
 * returns the maximum word id, not the 'real' count of sets.
 * if there are missing numbers within the series, this function
 * only returns the number of the last set. Do not use it for 
 * counting the sets. It will not work.
 */
int
lesson_get_max_word_id (lessonData *data);

/*
 * creates and returns the lesson data for a file
 */
void*
lesson_create_lesson_data (char *filename, lingConfig *settings);

/*
 * frees all memory hold by the passed lessonData, including the lessonData
 * itself
 */
void
lesson_free_lesson_data (lessonData *data);

/*
 * returns a random lesson from the lesson list, which was passed
 */
lingLesson*
lesson_return_rand_lesson (lingLesson *lesson);

/*
 * saves the passed lesson into the passed file
 */
lingbool
lesson_save_lesson (lingLesson *lesson, char *filename);

/*
 * gets a specific description of a meaning node
 */
lingchar*
lesson_node_get_description (xmlNodePtr node, lingchar *language);

#endif /* LING_LESSON_H */
