package Kolab::Conf;

##
##  Copyright (c) 2004  Klaralvdalens Datakonsult AB
##  Copyright (c) 2003  Code Fusion cc
##
##    Writen by Stuart Bing?<s.binge@codefusion.co.za>
##    Portions based on work by the following people:
##
##      (c) 2003  Tassilo Erlewein  <tassilo.erlewein@erfrakon.de>
##      (c) 2003  Martin Konold     <martin.konold@erfrakon.de>
##      (c) 2003  Achim Frank       <achim.frank@erfrakon.de>
##
##
##  This  program is free  software; you can redistribute  it and/or
##  modify it  under the terms of the GNU  General Public License as
##  published by the  Free Software Foundation; either version 2, or
##  (at your option) any later version.
##
##  This program is  distributed in the hope that it will be useful,
##  but WITHOUT  ANY WARRANTY; without even the  implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
##  General Public License for more details.
##
##  You can view the  GNU General Public License, online, at the GNU
##  Project's homepage; see <http://www.gnu.org/licenses/gpl.html>.
##

use 5.008;
use strict;
use warnings;
use IO::File;
use File::Copy;
use File::Temp;
use File::stat;
use Kolab;
use Kolab::Util;
use Kolab::LDAP;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = (
    'all' => [ qw(
        &buildPostfixTransportMap
        &buildCyrusConfig
        &buildCyrusGroups
        &buildLDAPReplicas
        &rebuildTemplates
	&checkPermissions
        %haschanged
    ) ]
);

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(

);

our $VERSION = '0.9';

sub fixup {
   my $file = shift;
   my $ownership = shift;
   my $perm = shift;

   (my $owner, my $group) = split(/:/, $ownership, 2);
   my $uid = (getpwnam($owner))[2];
   my $gid = (getgrnam($group))[2];
   Kolab::log('T', sprintf("Changing permissions of %s to 0%o", $file, $perm ), KOLAB_DEBUG );
   if( chmod($perm, $file) != 1 ) {
     Kolab::log('T', "Unable to change permissions of `$file' to ".sprintf("0%o",$perm), KOLAB_ERROR);
     exit(1);
   }
   Kolab::log('T', "Changing owner of $file to $owner:$group ($uid:$gid)", KOLAB_DEBUG );
   if( chown($uid,$gid,$file) != 1 ) {
     Kolab::log('T', "Unable to change ownership of `$file' to $uid:$gid", KOLAB_ERROR);
     exit(1);
   }
}

sub build {
    my $tmpl = shift;
    my $cfg = shift;
    my $owner = shift;
    my $perm = shift;
    my $oldcfg = $cfg . '.old';
    my $prefix = $Kolab::config{'prefix'};
    
    my $oldmask = umask 077;
    copy($cfg, $oldcfg);
    chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $oldcfg);
    umask $oldmask;
    #chmod(0600, $oldcfg) if ($oldcfg =~ /openldap/);

    Kolab::log('T', "Creating new configuration file `$cfg' from template `$tmpl'", KOLAB_DEBUG );
    #print STDERR "Creating new configuration file `$cfg' from template `$tmpl'\n";

    my $template;
    if (!($template = IO::File->new($tmpl, 'r'))) {
        Kolab::log('T', "Unable to open template file `$tmpl'", KOLAB_ERROR);
	# Error, fail gracefully
	return;
    }
    my $config;
    if (!($config = new File::Temp( TEMPLATE => 'tmpXXXXX',
				    DIR => $prefix.'/etc/kolab/',
				    SUFFIX => '.kolabtmp',
				    UNLINK => 0 ))) {
        Kolab::log('T', "Unable to open configuration file `$cfg'", KOLAB_ERROR);
        exit(1);
    }

    #Kolab::log('T', "Using temporary file '".$config->filename."'", KOLAB_DEBUG );

    my $skip = 0;
    my $keep = 0;
    while (<$template>) {
    	#Eat the meta data sections
	if (/^KOLAB_META_START$/) {
		my $found_end;
		while (!$found_end) {	
			$_ = <$template>;
			$found_end = /^KOLAB_META_END$/;
		}
		$_ = <$template>;
	}
        if (/\@{3}if (\S+?)\@{3}/) {
            if ($Kolab::config{$1} && lc($Kolab::config{$1}) ne "false" ) {
                # Keep text
		$keep = 1;
            } else {
                # Skip text
                $skip++;
		$keep = 0;
            }
            s/\@{3}if (\S+?)\@{3}\n?//;
	} elsif (/\@{3}else\@{3}/) {
	    if( $keep == 0 ) {
		# Now keep
		$keep = 1;
		$skip--;
	    } else {
		# Now skip
		$keep = 0;
                $skip++;
	    }
            s/\@{3}else\@{3}\n?//;
        } elsif (/\@{3}endif\@{3}/) {
            ($skip > 0) && $skip--;
            s/\@{3}endif\@{3}\n?//;
        } else {
            while (/\@{3}(\S+?)(\|.+?)?\@{3}/) {
                my $attr = $1;
                my $opt  = $2;
                #print STDERR "attr=$attr, opt=$opt\n";
                if ($Kolab::config{$attr}) {
                    my $val = "";
                    if (ref $Kolab::config{$attr} eq "ARRAY") {
                        if ($opt && $opt =~ /\|join(.*)/) {
                            $val = join $1, @{$Kolab::config{$attr}};
                        } else {
                            $val = $Kolab::config{$attr}->[0];
                        }
                    } else {
                        $val = $Kolab::config{$attr};
                    }
                    s/\@{3}(\S+?)(\|.+?)?\@{3}/$val/;
		    last if ( $val eq "\@\@\@$attr\@\@\@" ); # prevent endless loop
                } else {
                    Kolab::log('T', "No configuration variable corresponding to `$1' exists", KOLAB_WARN);
                    s/\@{3}(\S+?)\@{3}//;
                }
            }
        }
        ($skip == 0) && print $config $_;
    }

    $template->close;
    $config->close;

    move($config->filename, $cfg) || Kolab::log('T', "Error moving configfile into place: $!", KOLAB_ERROR );
    fixup( $cfg, $owner, $perm );
    #chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $cfg);
    #chmod(0600, $cfg) if ($cfg =~ /openldap/);

    if (-f $oldcfg) {
        my $rc = `diff -q $cfg $oldcfg`;
        chomp($rc);
        if ($rc) {
            if ($cfg =~ /postfix/) {
                $Kolab::haschanged{'postfix'} = 1;
            } elsif ($cfg =~ /saslauthd/) {
                $Kolab::haschanged{'saslauthd'} = 1;
            } elsif ($cfg =~ /apache/) {
                $Kolab::haschanged{'apache'} = 1;
            } elsif ($cfg =~ /proftpd/) {
                $Kolab::haschanged{'proftpd'} = 1;
            } elsif ($cfg =~ /openldap/) {
                $Kolab::haschanged{'slapd'} = 1;
            } elsif ($cfg =~ /imapd/) {
                $Kolab::haschanged{'imapd'} = 1;
	    } elsif ($cfg =~ /amavisd/) {
                $Kolab::haschanged{'amavisd'} = 1;
	    } elsif ($cfg =~ /clamav/) {
                $Kolab::haschanged{'clamav'} = 1;
	    }

            Kolab::log('T', "`$cfg' change detected: $rc", KOLAB_DEBUG );
        }
    }

    Kolab::log('T', "Finished creating configuration file `$cfg'", KOLAB_DEBUG );
}

sub buildPostfixTransportMap
{
    buildPostfixMap( 'transport' );
}

sub buildPostfixVirtualMap
{
    buildPostfixMap( 'virtual' );
}

sub buildPostfixMap
{
    my $map = shift;
    Kolab::log('T', "Building Postfix $map map", KOLAB_DEBUG);

    my $prefix = $Kolab::config{'prefix'};
    my $cfg = "$prefix/etc/postfix/$map";
    my $oldcfg = $cfg . '.old';

    my $oldmask = umask 077;
    copy($cfg, $oldcfg);
    chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $oldcfg);
    umask $oldmask;
    copy("$prefix/etc/kolab/templates/$map.template", $cfg);

    my $transport;
    if (!($transport = IO::File->new($cfg, 'a'))) {
        Kolab::log('T', "Unable to create Postfix $map map", KOLAB_ERROR);
        exit(1);
    }

    my $ldap = Kolab::LDAP::create(
        $Kolab::config{'ldap_ip'},
        $Kolab::config{'ldap_port'},
        $Kolab::config{'bind_dn'},
        $Kolab::config{'bind_pw'}
    );

    my $mesg = $ldap->search(
        base    => $Kolab::config{'base_dn'},
        scope   => 'sub',
        filter  => '(objectclass=*)'
    );
    if ($mesg->code) {
        Kolab::log('T', "Unable to locate Postfix $map map entries in LDAP", KOLAB_ERROR);
        exit(1);
    }

    my $ldapobject;
    if ($mesg->code <= 0) {
        foreach $ldapobject ($mesg->entries) {
            my $routes = $ldapobject->get_value("postfix-$map", asref => 1);
            foreach (@$routes) {
                $_ = trim($_);
                Kolab::log('T', "Adding entry `$_' to $map");
                print $transport $_ . "\n";
            }
        }
    } else {
        Kolab::log('T', "No Postfix $map map entries found");
    }

    Kolab::LDAP::destroy($ldap);
    $transport->close;

    # FIXME: bad way of doing things...
    #system("chown root:root $prefix/etc/postfix/*");
    system("$prefix/sbin/postmap $prefix/etc/postfix/$map");

    if (-f $oldcfg) {
        my $rc = `diff -q $cfg $oldcfg`;
        chomp($rc);
        if ($rc) {
           Kolab::log('T', "`$cfg' change detected: $rc", KOLAB_DEBUG);
           $Kolab::haschanged{'postfix'} = 1;
        }
    } else {
        $Kolab::haschanged{'postfix'} = 1;
    }

    Kolab::log('T', "Finished building Postfix $map map", KOLAB_DEBUG);
}

sub buildCyrusConfig
{
    Kolab::log('T', 'Building Cyrus config', KOLAB_DEBUG);

    my $prefix = $Kolab::config{'prefix'};
    my $cyrustemplate;
    if (!($cyrustemplate = IO::File->new("$prefix/etc/kolab/templates/cyrus.conf.template", 'r')))  {
        Kolab::log('T', "Unable to open template file `cyrus.conf.template'", KOLAB_ERROR);
        exit(1);
    }

    my $cfg = "$prefix/etc/imapd/cyrus.conf";
    my $oldcfg = $cfg . '.old';
    my $oldmask = umask 077;
    copy($cfg, $oldcfg);
    chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $oldcfg);
    umask $oldmask;

    my $cyrusconf;
    if (!($cyrusconf = IO::File->new($cfg, 'w'))) {
        Kolab::log('T', "Unable to open configuration file `$cfg'", KOLAB_ERROR);
        exit(1);
    }

    while (<$cyrustemplate>) {
        if (/\@{3}cyrus-imap\@{3}/ && ($Kolab::config{"cyrus-imap"} =~ /true/i)) {
            $_ = "imap cmd=\"imapd -C $prefix/etc/imapd/imapd.conf\" listen=\"143\" prefork=0\n";
        }
        elsif (/\@{3}cyrus-imap\@{3}/ && ($Kolab::config{"cyrus-imap"} =~ /false/i)) {
	    # Even if imapd is disabled, enable it for localhost -- resource handlers depend on it
            $_ = "imap cmd=\"imapd -C $prefix/etc/imapd/imapd.conf\" listen=\"127.0.0.1:143\" prefork=0\n";
        }
        elsif (/\@{3}cyrus-pop3\@{3}/ && ($Kolab::config{"cyrus-pop3"} =~ /true/i)) {
            $_ = "pop3 cmd=\"pop3d -C $prefix/etc/imapd/imapd.conf\" listen=\"110\" prefork=0\n";
        }
        elsif (/\@{3}cyrus-imaps\@{3}/ && ($Kolab::config{"cyrus-imaps"} =~ /true/i)) {
            $_ = "imaps cmd=\"imapd -s -C $prefix/etc/imapd/imapd.conf\" listen=\"993\" prefork=0\n";
        }
        elsif (/\@{3}cyrus-pop3s\@{3}/ && ($Kolab::config{"cyrus-pop3s"} =~ /true/i)) {
            $_ = "pop3s cmd=\"pop3d -s -C $prefix/etc/imapd/imapd.conf\" listen=\"995\" prefork=0\n";
        }
        elsif (/\@{3}cyrus-sieve\@{3}/ && ($Kolab::config{"cyrus-sieve"} =~ /true/i)) {
            $_ = "sieve cmd=\"timsieved -C $prefix/etc/imapd/imapd.conf\" listen=\"2000\" prefork=0\n";
        }
        $_ =~ s/\@{3}.*\@{3}//;
        print $cyrusconf $_;
    }

    $cyrustemplate->close;
    $cyrusconf->close;

    chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $cfg);

    if (-f $oldcfg) {
        my $rc = `diff -q $cfg $oldcfg`;
        chomp($rc);
        if ($rc) {
           Kolab::log('T', "`$cfg' change detected: $rc", KOLAB_DEBUG);
           $Kolab::haschanged{'imapd'} = 1;
        }
    } else {
        $Kolab::haschanged{'imapd'} = 1;
    }

    Kolab::log('T', 'Finished building Cyrus config', KOLAB_DEBUG);
}

sub buildCyrusGroups
{
    Kolab::log('T', 'Building Cyrus groups', KOLAB_DEBUG);

    my $prefix = $Kolab::config{'prefix'};
    my $cfg = "$prefix/etc/imapd/imapd.group";
    my $oldcfg = $cfg . '.old';

    my $oldmask = umask 077;
    copy($cfg, $oldcfg);
    chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $oldcfg);
    umask $oldmask;
    copy("$prefix/etc/kolab/templates/imapd.group.template", $cfg);
    my $groupconf;
    if (!($groupconf = IO::File->new($cfg, 'a'))) {
        Kolab::log('T', "Unable to open configuration file `$cfg'", KOLAB_ERROR);
        exit(1);
    }

    my $ldap = Kolab::LDAP::create(
        $Kolab::config{'ldap_ip'},
        $Kolab::config{'ldap_port'},
        $Kolab::config{'bind_dn'},
        $Kolab::config{'bind_pw'}
    );

    my $mesg = $ldap->search(
        base    => $Kolab::config{'base_dn'},
        scope   => 'sub',
        filter  => '(objectclass=kolabgroupofnames)'
    );
    if ($mesg->code) {
        Kolab::log('T', 'Unable to locate Cyrus groups in LDAP', KOLAB_ERROR);
        exit(1);
    }

    my $ldapobject;
    my $count = 60000;
    if ($mesg->code <= 0) {
        foreach $ldapobject ($mesg->entries) {
	    my @dcs = split(/,/,$ldapobject->dn());
	    my @dn;
	    while( pop( @dcs ) =~ /dc=(.*)/ ) {
	      push(@dn, $1);
	    }
            my $group = $ldapobject->get_value('cn') . '@'.join('.',reverse(@dn)) . ":*:$count:";
            my $userlist = $ldapobject->get_value('member', asref => 1);
            foreach (@$userlist) {
	      my $uid = $_;
	      my $umesg = $ldap->search( base => $uid,
					 scope => 'exact', 
					 filter => '(objectClass=*)' );
	      if ( $umesg && $umesg->code() <= 0 && $umesg->count() == 1 ) {
		my $mail;
		($mail = $umesg->entry(0)->get_value('mail')) or
		($mail = $umesg->entry(0)->get_value('uid'));
		$group .= $mail.',';		
	      }
	    }
            $group =~ s/,$//;
            print $groupconf $group . "\n";
            Kolab::log('T', "Adding cyrus group `$group'");
            $count++;
        }
    } else {
        Kolab::log('T', 'No Cyrus groups found');
    }

    $groupconf->close;
    Kolab::LDAP::destroy($ldap);

    chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $cfg);

    Kolab::log('T', 'Finished building Cyrus groups', KOLAB_DEBUG );
}

sub buildLDAPReplicas
{
    Kolab::log('T', 'Building LDAP replicas', KOLAB_DEBUG);

    my $prefix = $Kolab::config{'prefix'};
    my $cfg = "$prefix/etc/openldap/slapd.replicas";
    my $oldcfg = $cfg . '.old';

    my $oldmask = umask 077;
    copy($cfg, $oldcfg);
    chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $oldcfg);
    copy("$prefix/etc/kolab/templates/slapd.replicas.template", $cfg);
    my $repl;
    if (!($repl = IO::File->new($cfg, 'a'))) {
        Kolab::log('T', "Unable to open configuration file `$cfg'", KOLAB_ERROR);
        exit(1);
    }

    if( $Kolab::config{'is_master'} eq "true" ) {
      # Master setup
      my @kh;
      if( ref $Kolab::config{'kolabhost'} eq 'ARRAY' ) {
	@kh = @{$Kolab::config{'kolabhost'}};
      } else {
	@kh = ( $Kolab::config{'kolabhost'} );
      }
      for my $h ( @kh ) {
	next if lc($h) eq lc($Kolab::config{'fqdnhostname'});
	print $repl "replica uri=ldaps://$h\n"
	  ."  binddn=\"".$Kolab::config{'bind_dn'}."\"\n"
	    ."  bindmethod=simple credentials=".$Kolab::config{'bind_pw'}."\n\n";
      }
    } else {
      # Slave setup
      # Output an update dn statement instead
      print $repl "updatedn ".$Kolab::config{'bind_dn'}."\n";
      print $repl "updateref ".$Kolab::config{'ldap_master_uri'}."\n";
    }

    $repl->close;

    chown($Kolab::config{'kolab_uid'}, $Kolab::config{'kolab_gid'}, $cfg);

    if (-f $oldcfg) {
        my $rc = `diff -q $cfg $oldcfg`;
        chomp($rc);
        if ($rc) {
           Kolab::log('T', "`$cfg' change detected: $rc", KOLAB_DEBUG);
           $Kolab::haschanged{'slapd'} = 1;
        }
    } else {
        $Kolab::haschanged{'slapd'} = 1;
    }
    umask $oldmask;

    Kolab::log('T', 'Finished building LDAP replicas', KOLAB_DEBUG);
}

sub loadMetaTemplates
{	
	my $templatedir = shift;
	my ($tref, $pref, $oref) = @_;

	Kolab::log('T', 'Collecting template files', KOLAB_DEBUG );
	opendir(DIR, $templatedir);
	my @metatemplates = grep { /\.template$/ } readdir (DIR);
	closedir(DIR);

	foreach my $template (@metatemplates) {
		#Open each file and check for the META
		if (open (TEMPLATE, "$templatedir/$template" )) {
			my $line = <TEMPLATE>;
			if ($line =~ /^KOLAB_META_START$/) {
				Kolab::log('T', 'Processing META template :'.$template, KOLAB_DEBUG );
				my ($found_end, $target, $permissions, $ownership);
				while (<TEMPLATE>) {
				$line = $_;
				if (!$found_end) {
					$found_end = $line =~ /^KOLAB_META_END$/;
					if (!$found_end && $line) {
						my ($key,$value) = split(/=/,$line);
						chomp($value);
						Kolab::log('T', 'META Key: '.$key.' Value: '.$value, KOLAB_DEBUG );
						if ($key =~ /^TARGET$/) {
							$target = $value;
							Kolab::log('T', 'META Target '.$target, KOLAB_DEBUG );
						}
						if ($key =~ /^PERMISSIONS$/) {
							$permissions = $value;
							Kolab::log('T', 'META Permissions '.$permissions, KOLAB_DEBUG );
						}
						if ($key =~ /^OWNERSHIP$/) {
							$ownership = $value;
							Kolab::log('T', 'META Ownership '.$ownership, KOLAB_DEBUG );
						}
					}
				}
				}
				if ($found_end && $target && $permissions && $ownership) {
				   Kolab::log('T', 'All mandatory fields populated in '.$template, KOLAB_DEBUG );
				   $$tref{$template} = $target;
				   $$oref{$target} = $ownership;
				   $permissions = oct($permissions);
				   $$pref{$target} = $permissions;
				}
				
			}
		} else {
		  Kolab::log('T', 'Could not open template file: '. $template, KOLAB_WARN);
		}
	}

}

my $prefix = $Kolab::config{'prefix'};
my $templatedir = "$prefix/etc/kolab/templates";

my %templates = (
		 "$templatedir/rc.conf.template" => "$prefix/etc/rc.conf",
		 "$templatedir/session_vars.php.template" => "$prefix/var/kolab/php/admin/include/session_vars.php",
		 "$templatedir/clamd.conf.template" => "$prefix/etc/clamav/clamd.conf",
		 "$templatedir/freshclam.conf.template" => "$prefix/etc/clamav/freshclam.conf",
		 "$templatedir/amavisd.conf.template" => "$prefix/etc/amavisd/amavisd.conf",
		 "$templatedir/main.cf.template" => "$prefix/etc/postfix/main.cf",
		 "$templatedir/master.cf.template" => "$prefix/etc/postfix/master.cf",
		 "$templatedir/saslauthd.conf.template" => "$prefix/etc/sasl/saslauthd.conf",
		 "$templatedir/imapd.conf.template" => "$prefix/etc/imapd/imapd.conf",
		 "$templatedir/httpd.conf.template" => "$prefix/etc/apache/apache.conf",
		 "$templatedir/httpd.local.template" => "$prefix/etc/apache/apache.local",
		 "$templatedir/php.ini.template" => "$prefix/etc/apache/php.ini",
		 "$templatedir/proftpd.conf.template" => "$prefix/etc/proftpd/proftpd.conf",
		 "$templatedir/ldap.conf.template" => "$prefix/etc/openldap/ldap.conf",
		 "$templatedir/slapd.conf.template" => "$prefix/etc/openldap/slapd.conf",
		 "$templatedir/DB_CONFIG.slapd.template" => "$prefix/var/openldap/openldap-data/DB_CONFIG",
		 "$templatedir/freebusy.conf.template" => "$prefix/etc/resmgr/freebusy.conf",
		 "$templatedir/fbview.conf.template" => "$prefix/etc/resmgr/fbview.conf",
		 "$templatedir/resmgr.conf.template" => "$prefix/etc/resmgr/resmgr.conf"
		);

my %permissions = (
		   "$prefix/etc/rc.conf"                                  => 0644,
		   "$prefix/var/kolab/php/admin/include/session_vars.php" => 0640,
		   "$prefix/etc/resmgr/freebusy.conf"                     => 0640,
		   "$prefix/etc/resmgr/fbview.conf"                       => 0640,
		   "$prefix/etc/resmgr/resmgr.conf"                       => 0640,
		   "$prefix/etc/postfix/main.cf"                          => 0640,
		   "$prefix/etc/postfix/master.cf"                        => 0640,
		   "$prefix/etc/clamav/clamd.conf"                        => 0640,
		   "$prefix/etc/clamav/freshclam.conf"                    => 0640,
		   "$prefix/etc/amavisd/amavisd.conf"                     => 0640,
		   "$prefix/etc/sasl/saslauthd.conf"                      => 0600,
		   "$prefix/etc/imapd/imapd.conf"                         => 0640,
		   "$prefix/etc/apache/apache.conf"                       => 0640,
		   "$prefix/etc/apache/apache.local"                      => 0640,
		   "$prefix/etc/apache/php.ini"                           => 0640,
		   "$prefix/etc/proftpd/proftpd.conf"                     => 0640,
		   "$prefix/etc/openldap/slapd.conf"                      => 0640,
		   "$prefix/var/openldap/openldap-data/DB_CONFIG"         => 0640,
		   "$prefix/etc/openldap/ldap.conf"                       => 0644,
		   "$prefix/etc/postfix/transport"                        => 0640,
		   "$prefix/etc/imapd/cyrus.conf"                         => 0640,
		   "$prefix/etc/imapd/imapd.group"                        => 0640);

my %ownership = (
		 "$prefix/etc/rc.conf"                                  => "kolab:kolab",
		 "$prefix/var/kolab/php/admin/include/session_vars.php" => "kolab:kolab-n",
		 "$prefix/etc/resmgr/freebusy.conf"                     => "kolab:kolab-n",
		 "$prefix/etc/resmgr/fbview.conf"                       => "kolab:kolab-n",
		 "$prefix/etc/resmgr/resmgr.conf"                       => "kolab:kolab-n",
		 "$prefix/etc/postfix/main.cf"                          => "kolab:kolab-r",
		 "$prefix/etc/postfix/master.cf"                        => "kolab:kolab-r",
		 "$prefix/etc/clamav/clamd.conf"                        => "kolab:kolab-r",
		 "$prefix/etc/clamav/freshclam.conf"                    => "kolab:kolab-r",
		 "$prefix/etc/amavisd/amavisd.conf"                     => "kolab:kolab-r",
		 "$prefix/etc/sasl/saslauthd.conf"                      => "kolab:kolab",
		 "$prefix/etc/imapd/imapd.conf"                         => "kolab:kolab-r",
		 "$prefix/etc/apache/apache.conf"                       => "kolab:kolab-n",
		 "$prefix/etc/apache/apache.local"                      => "kolab:kolab-n",
		 "$prefix/etc/apache/php.ini"                           => "kolab:kolab-n",
		 "$prefix/etc/proftpd/proftpd.conf"                     => "kolab:kolab-n",
		 "$prefix/etc/openldap/ldap.conf"                       => "kolab:kolab",
		 "$prefix/etc/openldap/slapd.conf"                      => "kolab:kolab",
		 "$prefix/var/openldap/openldap-data/DB_CONFIG"         => "kolab:kolab",
		 "$prefix/etc/postfix/transport"                        => "root:kolab",
		 "$prefix/etc/imapd/cyrus.conf"                         => "kolab:kolab",
		 "$prefix/etc/imapd/imapd.group"                        => "kolab:kolab-r");


sub rebuildTemplates
{
    my $key;
    my $value;
    my $section="";


    Kolab::log('T', 'Regenerating configuration files', KOLAB_DEBUG );

    loadMetaTemplates( $templatedir, \%templates, \%permissions, \%ownership );

    foreach $key (keys %templates) {
      my $tpl = $templates{$key};
      build($key, $tpl, $ownership{$tpl}, $permissions{$tpl} );
    }

    buildPostfixTransportMap;
    buildPostfixVirtualMap;
    buildLDAPReplicas;
    buildCyrusConfig;
    buildCyrusGroups;

    Kolab::log('T', 'Finished regenerating configuration files', KOLAB_DEBUG );
}

sub checkPermissions {
    my $key;
    my $value;

    Kolab::log('T', 'Checking generated config file permissions and ownership', KOLAB_DEBUG );

    loadMetaTemplates( $templatedir, \%templates, \%permissions, \%ownership );

    my $ok = 1;

    foreach $key (keys %templates) {
      my $tpl = $templates{$key};
      my $st = stat($tpl);
      my $owner = getpwuid($st->uid).':'.getgrgid($st->gid);
      if( ( ($st->mode & 07777) != $permissions{$tpl}) ||
	  ($owner ne $ownership{$tpl}) ) {
	  my $str = 'File '.$tpl.' has the wrong persmissions/owner. Found '
		     .sprintf("%lo", $st->mode&07777).' '.$owner.', expected '
		     .sprintf("%lo",$permissions{$tpl}).' '.$ownership{$tpl};
	  print( "$str\n" );
	  Kolab::log('T', $str, KOLAB_ERROR );
	  $ok = 0;
      }
  }
  return $ok;
}

1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

Kolab::Conf - Perl extension for Kolab template generation

=head1 ABSTRACT

  Kolab::Conf handles the generation of template files, used by
  kolabconf.

=head1 AUTHOR

Stuart Bing묠E<lt>s.binge@codefusion.co.zaE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (c) 2003  Code Fusion cc

Portions based on work by the following people:

  (c) 2003  Tassilo Erlewein  <tassilo.erlewein@erfrakon.de>
  (c) 2003  Martin Konold     <martin.konold@erfrakon.de>
  (c) 2003  Achim Frank       <achim.frank@erfrakon.de>

This  program is free  software; you can redistribute  it and/or
modify it  under the terms of the GNU  General Public License as
published by the  Free Software Foundation; either version 2, or
(at your option) any later version.

This program is  distributed in the hope that it will be useful,
but WITHOUT  ANY WARRANTY; without even the  implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You can view the  GNU General Public License, online, at the GNU
Project's homepage; see <http://www.gnu.org/licenses/gpl.html>.

=cut
