#ifndef _ASYNCHTTPREQUEST_H_
#define _ASYNCHTTPREQUEST_H_

#include "talk/base/httpclient.h"
#include "talk/base/logging.h"
#include "talk/base/proxyinfo.h"
#include "talk/base/socketserver.h"
#include "talk/base/thread.h"
#include "talk/base/signalthread.h"

namespace cricket {
  class FirewallManager;
  class MemoryStream;

// Performs an HTTP request on a background thread.  Notifies on the foreground
// thread once the request is done (successfully or unsuccessfully).
class AsyncHttpRequest:
  public SignalThread,
  public sigslot::has_slots<> {
public:
    AsyncHttpRequest(const std::string &user_agent);

  void set_proxy(const cricket::ProxyInfo& proxy) { proxy_ = proxy; }

  // The DNS name of the host to connect to.
  const std::string& host() { return host_; }
  void set_host(const std::string& host) { host_ = host; }

  // The port to connect to on the target host.
  int port() { return port_; }
  void set_port(int port) { port_ = port; }
       
   // Whether the request should use SSL.
  bool secure() { return secure_; }
  void set_secure(bool secure) { secure_ = secure; }

  // Returns the redirect when redirection occurs
  const std::string& response_redirect() { return response_redirect_; }

  // Time to wait on the download, in ms.  Default is 5000 (5s)
  int timeout() { return timeout_; }
  void set_timeout(int timeout) { timeout_ = timeout; }

  // Fail redirects to allow analysis of redirect urls, etc.
  bool fail_redirect() const { return fail_redirect_; }
  void set_fail_redirect(bool fail_redirect) { fail_redirect_ = fail_redirect; }

  HttpRequestData& request() { return client_.request(); }
  HttpResponseData& response() { return client_.response(); }
   
  virtual void DoWork(bool *halt);

private:
  cricket::ProxyInfo proxy_;
  std::string response_redirect_;
  int complete_;
  int err_;
  int timeout_;
  bool fail_redirect_;
  bool secure_;
  bool success_;
  bool error_;
  std::string host_;
  int port_;
  unsigned status_code_;
  std::string agent_;
  HttpClient client_;
  
  // Performs the HTTP request synchronously.
  void DoRequest();
};

class HttpMonitor : public sigslot::has_slots<> {
public:
  HttpMonitor(SocketServer *ss);

  void reset() { complete_ = false; }

  bool done() const { return complete_; }
  int error() const { return err_; }

  void Connect(cricket::HttpClient* http);  
  void OnHttpClientComplete(cricket::HttpClient * http, int err);

private:
  bool complete_;
  int err_;
  SocketServer *ss_;
};

class SslSocketFactory : public cricket::SocketFactory {
 public:
  SslSocketFactory(cricket::SocketFactory * factory, const std::string &user_agent)
    : factory_(factory), logging_level_(LS_VERBOSE), 
      binary_mode_(false), agent_(user_agent) { }

  void UseSSL(const char * hostname) { hostname_ = hostname; }
  void DisableSSL() { hostname_.clear(); }

  void SetProxy(const cricket::ProxyInfo& proxy) { proxy_ = proxy; }
  const cricket::ProxyInfo& proxy() const { return proxy_; }
  bool ignore_bad_cert() {return ignore_bad_cert_;}
  void SetIgnoreBadCert(bool ignore) { ignore_bad_cert_ = ignore; }

  void SetLogging(LoggingSeverity level, const std::string& label, 
      bool binary_mode = false) {
    logging_level_ = level;
    logging_label_ = label;
    binary_mode_ = binary_mode;
  }

  virtual cricket::Socket * CreateSocket(int type);
  virtual cricket::AsyncSocket * CreateAsyncSocket(int type);

private:
  cricket::SocketFactory * factory_;
  cricket::ProxyInfo proxy_;
  std::string hostname_, logging_label_;
  LoggingSeverity logging_level_;
  bool binary_mode_;
  std::string agent_;
  bool ignore_bad_cert_;
};

} // namespace cricket_

#endif // _ASYNCHTTPREQUEST_H_
