/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.tools.buildmagic.task.config;

import java.io.File;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;

import org.apache.tools.ant.Task;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.types.PatternSet;

import org.jboss.tools.buildmagic.task.MissingAttributeException;
import org.jboss.tools.buildmagic.task.MissingElementException;
import org.jboss.tools.buildmagic.task.util.TaskLogger;

/**
 * A helper task to setup a path from a set of library elements.
 *
 * <p>Usage:
 * <xmp>
 *
 *   <libraryset name="library" root="${project.thirdparty}">
 *
 *    <!-- Java API for XML Processing (JAXP) -->
 *    <library vendor="sun" name="jaxp"/>
 *
 *    <!-- IBM Bean Scripting Framework (BSF) -->
 *    <library vendor="ibm" name="bsf"/>
 *
 *    <!-- JUnit -->
 *    <library vendor="junit" name="junit"/>
 *
 *    <!-- Log4j -->
 *    <library vendor="apache" name="log4j"/>
 *
 *  </libraryset>
 *
 * </xmp>
 *
 * @version <tt>$Revision: 21939 $</tt>
 * @author  <a href="mailto:jason@planet57.com">Jason Dillon</a>
 */
public class LibrarySet
   extends Task 
{
   /** Instance logger. */
   protected final TaskLogger log = new TaskLogger(this);

   protected String pathID;
   protected File root;
   protected List libraries;

   protected Path classpath;

   public void setPathID(final String pathID)
   {
      this.pathID = pathID;
   }

   public void setRoot(final File root)
   {
      this.root = root;
   }

   public Library createLibrary() 
   {
      Library lib = (Library)getProject().createTask("library");

      if (libraries == null) {
         libraries = new ArrayList();
      }
      libraries.add(lib);

      return lib;
   }

   public Library createModuleLibrary()
   {
      Library lib = (Library)getProject().createTask("modulelibrary");

      if (libraries == null) {
         libraries = new ArrayList();
      }
      libraries.add(lib);

      return lib;
   }

   /**
    * Validate the attributes for this task.
    *
    * @throws BuildException    Attributes are not valid.
    */
   protected void validate() throws BuildException 
   {
      if (pathID == null)
         throw new MissingAttributeException("pathID", this);
   }

   /**
    * Execute the task.
    *
    * @throws BuildException    Failed to execute.
    */
   public void execute() throws BuildException 
   {
      // short circuit if we are done
      if (classpath != null) return;

      validate();

      log.debug("pathID: " + pathID);
      log.debug("root: " + root);

      // setup the classpath
      classpath = new Path(getProject());

      if (libraries != null) {
         Iterator iter = libraries.iterator();
         while (iter.hasNext()) {
            Library lib = (Library)iter.next();

            // set the root if we have one, else use the libs
            if (root != null) {
               lib.setRoot(root);
            }

            // execute
            lib.execute();
         
            // append the classpath
            classpath.append(lib.getClasspath());
         }

         log.debug("classpath: " + classpath);
      }

      // set classpath ref
      Map refs = getProject().getReferences();
      refs.put(pathID, classpath);

      // help gc
      pathID = null;
      root = null;
      libraries = null;
   }

   /**
    * For helper tasks.
    */
   public Path getClasspath()
   {
      return classpath;
   }
}
