/*
 *   Greebo: An Ant task to manage dependency files.
 *
 *   Copyright (C) 2002 Ozben Evren
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

//
// Heavily modified for use with JBoss
//

package oevren.ant.greebo;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.BuildException;

import org.apache.tools.ant.taskdefs.Get;
import org.apache.tools.ant.taskdefs.Touch;
import org.apache.tools.ant.taskdefs.UpToDate;

import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.FileSet;

import org.apache.xerces.parsers.DOMParser;
import org.w3c.dom.*;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.Collection;
import java.util.Stack;
import java.io.File;
import java.net.URL;
import java.net.MalformedURLException;

import org.jboss.tools.buildmagic.task.util.TaskLogger;

/**
 *
 * <p>
 * <pre>
 * $Log: FetchDependencyTask.java,v $
 * Revision 1.1.2.2  2003/03/08 19:30:38  user57
 *  o make sure we read the right file
 *
 * Revision 1.1.2.1  2003/03/08 18:07:43  user57
 *  o import of heavily modified Greebo dependency manager task
 *  o modified buildmagic/tasks module to use the new fluff
 *
 * Revision 1.1  2002/11/22 02:41:47  ozbene
 * Initial check-in to CVS
 *
 * </pre>
 *
 * @author   Ozben Evren
 * @version  $Id: FetchDependencyTask.java 21628 2008-08-19 22:06:02Z pgier $
 *
 */
public class FetchDependencyTask 
    extends Task 
{
    public static final String DEFAULT_REPOSITORY = "http://jboss.sf.net/buildmagic/thirdparty/";
    
    public static class Context
    {
        private FetchDependencyTask task;
        private TaskLogger logger;
        
        public Context(final FetchDependencyTask task)
        {
            this.task = task;
            this.logger = new TaskLogger(task);
        }
        
        public Task getTask()
        {
            return task;
        }
        
        public TaskLogger getLogger()
        {
            return logger;
        }
        
        public boolean isOffline()
        {
            return task.isOffline();
        }
    }
    
    private Context context;
    private TaskLogger log;
    
    private boolean offline = false;
    
    /**
     * An external dependencies file, with the same dependency structure of the
     * project.xml file of Maven.
     */
    private File dependencyFile;
    
    /**
     * Optional state file for dependency metadata
     */
    private File metadataFile;
    
    /**
    * The optional path id where the generated dependecy path will be installed.
    */
    private String pathID;
    
    /**
     * The destination directory for this dependency task.
     */
    private File dir;

    private LinkedList dependencies;
    private LinkedList repositories;
    private Get get;

    public FetchDependencyTask() 
    {
        dependencies = new LinkedList();
        repositories = new LinkedList();
        get = new Get() {
            public void log (String msg)
            {
                if (msg.startsWith("Error opening connection "))
                {
                    return;
                }
                if (msg.startsWith("Can't get "))
                {
                    return;
                }
                if (msg.startsWith("Not modified - so not downloaded"))
                {
                    return;
                }
                if (msg.startsWith("Getting:"))
                {
                    return;
                }
                super.log(msg);
            }
        };
        get.setUseTimestamp(true);
        get.setVerbose(false);
    }

    public void init()
    {
        log = new TaskLogger(this);
        context = new Context(this);
    }
    
    private void validate() throws BuildException 
    {
        if (dependencyFile == null && dependencies.size() == 0)
            throw new BuildException("You have to have either a dependencyFile, or at least one JarDependency");

        if (dir == null)
            throw new BuildException("target directory <dir> is not specified.");

        if (!dir.exists()) {
            dir.mkdirs();
            log.info("Created dir: " + dir);
        }

        if (!dir.isDirectory())
            throw new BuildException("target directory <dir> is not a directory.");
    }

    public void execute() throws BuildException 
    {
        get.setTaskName(getTaskName());
        get.setProject(getProject());
        validate();

        // Setup repository list, add default if none specified
        if (repositories.size() == 0) {
            log("No Repository Url is specified, assuming: " + DEFAULT_REPOSITORY);
            Repository repository = new Repository(context);
            
            try {
                repository.setURL(new URL(DEFAULT_REPOSITORY));
            }
            catch (MalformedURLException e) {
                throw new org.jboss.util.NestedError(e); // Should never happen
            }
            
            repositories.add(repository);
        }
        
        // Generate a list of depends from an optional dependency file
        if (dependencyFile != null) {
            log.verbose("Reading dependency file: " + dependencyFile);

            try {
                Collection temp = getDependenciesFromFile(dependencyFile);
                dependencies.addAll(temp);
            }
            catch (Exception e) {
                e.printStackTrace();
                throw new BuildException("Failed to read dependencies from file: " + dependencyFile);
            }
        }
        
        boolean skipVerify = false;
        
        // If a metadata file was given use it to check if we need update depenencies
        if (metadataFile != null && metadataFile.exists()) {
            UpToDate u2d = (UpToDate)getProject().createTask("uptodate");
            u2d.setTaskName(getTaskName());
            u2d.setTargetFile(metadataFile);
            
            File antFile = new File(getProject().getProperty("ant.file"));
            u2d.setSrcfile(antFile);
            log.verbose("Checking if " + antFile + " is newer than our metadata");
            skipVerify = u2d.eval();
            
            if (skipVerify) {
                if (dependencyFile != null) {
                    log.verbose("Checking if " + dependencyFile + " is newer than our metadata");
                    u2d.setSrcfile(dependencyFile);
                    skipVerify = u2d.eval();
                }
            }
            
            // now make sure that all of the dependencies still exist
            if (skipVerify) {
                Iterator iter=dependencies.iterator();
                while (iter.hasNext()) {
                    JarDependency dep = (JarDependency)iter.next();
                    File file = new File(dir, dep.getFileNameFragment());
                    if (!file.exists()) {
                        // missing one, force verify
                        log.verbose("Forcing verification; Missing cached dependency: " + file);
                        skipVerify = false;
                        break;
                    }
                }
            }
        }
        
        if (!skipVerify) {
            // Verify depends, downloads to cache as needed
            int size = dependencies.size();
            log.info("Verifying "+size+" dependenc"+(size>1?"ies":"y")+"...");
            grabDependencies(dependencies);
        }
        else {
            log.info("Dependencies are up-to-date (metadata check).");
        }
        
        // Generate an Ant FileSet containing all of the depends
        FileSet dependFileSet = new FileSet();
        dependFileSet.setDir(dir);
        
        if (pathID != null) {
            log("Generate path with ID: " + pathID, Project.MSG_DEBUG);
            
            Iterator iter=dependencies.iterator();
            while (iter.hasNext()) {
                JarDependency dep = (JarDependency)iter.next();
                dependFileSet.createInclude().setName(dep.getFileNameFragment());
            }
        
            Path path = new Path(getProject());
            path.addFileset(dependFileSet);
            
            log("Path: " + path, Project.MSG_DEBUG);
            getProject().getReferences().put(pathID, path);
        }
        
        // If we have a metadata file touch it to keep from checking over and over
        if (metadataFile != null) {
            metadataFile.getParentFile().mkdirs();
            Touch touch = (Touch)getProject().createTask("touch");
            touch.setTaskName(getTaskName());
            touch.setFile(metadataFile);
            touch.execute();
            log.verbose("Touched metadata file: " + metadataFile);
        }
    }

    public void grabDependencies(Collection dependencies) throws BuildException 
    {
        if (offline) {
            log.warning("Remote repositories offline; Can not verify cached dependencies");
        }
        
        Iterator i = dependencies.iterator();
        Iterator iRepos;
        Stack synchroStack = new Stack(); //stack of repositories that need to be synchronized
        File downloadedFile = null;

        int count=0;
        while (i.hasNext()) {

            JarDependency dependency = (JarDependency) i.next();

            iRepos = repositories.iterator();
            while (iRepos.hasNext()) {
                Repository repository = (Repository) iRepos.next();
                
                try {
                    File destinationFile = new File(dir, dependency.getFileNameFragment());
                    long lastModified = destinationFile.lastModified();
                    
                    downloadedFile = repository.getDependency(dependency, dir, get);
                    
                    while (!synchroStack.isEmpty()) {
                        Repository synchroRepository = (Repository) synchroStack.pop();
                        synchroRepository.synchronizeFile(downloadedFile, dependency);
                    }
                    
                    count++;
                    if (lastModified != downloadedFile.lastModified()) {
                        log(" " + count + ") Downloaded " + dependency + " from: " + repository);
                    }
                    else if (offline) {
                        log(" " + count + ") " + dependency + " exists; may be stale");
                    }
                    else {
                        log(" " + count + ") " + dependency + " is up-to-date");
                    }
                    break;
                } 
                catch (Exception e) {
                    if (!iRepos.hasNext())
                        throw new BuildException("Could not get <" + dependency.getName() + "> from any repository");
                    else {
                        if (repository.isSynchronize()) {
                            synchroStack.push(repository);
                        }
                        log("Couldn't get <" + dependency.getName() + ">, trying next repository...",Project.MSG_DEBUG);
                    }
                }
            }
        }
    }

    /**
     * Utility function to load a Xerces DOM Document with the given filename.
     *
     * @param filename the name of the file to load
     * @return the DOM Document object that is created
     * @throws org.apache.tools.ant.BuildException
     */
    private Document getDocument(File file) throws Exception 
    {
        DOMParser parser = new DOMParser();
        parser.parse(file.getAbsolutePath());
        return parser.getDocument();
    }

    /**
     * Loads the dependencies from the file with the given name.
     *
     * @param fileName of the file to be loaded
     */
    private LinkedList getDependenciesFromFile(File file) throws Exception 
    {
        LinkedList list = new LinkedList();
        Document doc = getDocument(file);
        NodeList dependencies = doc.getElementsByTagName("dependency");

        for (int c = 0; c < dependencies.getLength(); ++c) {
            Element element = (Element) dependencies.item(c);

            JarDependency dep = new JarDependency();
            dep.setName(getTextOfChildElement(element, "id", 0));
            dep.setVersion(getTextOfChildElement(element, "version", 0));
            dep.setJar(getTextOfChildElement(element, "jar", 0));
            dep.setGroupName(getTextOfChildElement(element, "group-id", 0));
            
            list.add(dep);
        }

        return list;
    }
    
    /**
     * Utility method to grab the text data of the child of the given parent object.
     */
    private String getTextOfChildElement(Element parent, String childName, int childNo) 
    {
        NodeList childList = parent.getElementsByTagName(childName);

        if (childList.getLength() <= childNo)
            return null;

        Element child = (Element) childList.item(childNo);

        NodeList grandChildren = child.getChildNodes();

        for (int c = 0; c < grandChildren.getLength(); ++c) {
            Node grandChild = grandChildren.item(c);
            if (grandChild.getNodeType() == Node.TEXT_NODE)
                return ((Text) grandChild).getData();
        }

        return null;
    }

    /**
     * Creates a JarDependency based on the <code>jarDependency</code> tag
     * embedded inside in this task.
     *
     * @return the newly created JarDependency
     */
    public Object createJarDependency() {
        JarDependency ret = new JarDependency();
        dependencies.add(ret);
        return ret;
    }

    public Object createRepository() {
        Repository ret = new Repository(context);
        repositories.add(ret);
        return ret;
    }

    public File getDependencyFile() {
        return dependencyFile;
    }

    public void setDependencyFile(File dependencyFile) {
        this.dependencyFile = dependencyFile;
    }

    public File getMetadataFile()
    {
        return metadataFile;
    }
    
    public void setMetadataFile(File file)
    {
        this.metadataFile = file;
    }
    
    public File getCacheDir() {
        return dir;
    }

    public void setCacheDir(File dir) {
        this.dir = dir;
    }

    public String getPathId() {
        return pathID;
    }
    
    public void setPathId(String id) {
        this.pathID = id;
    }
    
    public void setOffline(boolean flag)
    {
        offline = flag;
    }
    
    public boolean isOffline()
    {
        return offline;
    }
}
