/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>

#include <zlib.h>

#include <libinstrudeo/isddatafile.h>
#include <libinstrudeo/isdprogresscallback.h>
#include <libinstrudeo/isdrectangle.h>
#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdvnclogimporter.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while importing isdViewer file: \n"
#include <libinstrudeo/isdloggermacros.h>

//-----CONSTRUCTORS-----
ISDVncLogImporter::ISDVncLogImporter(const string vncLogFileName)
    : ISDImporter(vncLogFileName),
      needsNormalization(true)
{
}

//-----DESTRUCTORS-----
ISDVncLogImporter::~ISDVncLogImporter()
{
}

//-----PUBLIC FUNCTIONS-----
ISDObject::ISDErrorCode ISDVncLogImporter::convertTo(string& outputFileName,
						     ISDProgressCallback* callbackClass,
						     bool createNew)
{
    ISDObject::ISDErrorCode retVal;
    
    //init the output file name
    if (createNew){
	if ((retVal = ISDUtils::getInstance()->createTempFile(this->outputFileName)) != ISD_SUCCESS) {
	    cleanup();
	    RETURN_ERROR(ISD_FILE_ERROR);
	}
    }
    else
	this->outputFileName = outputFileName;

    //init streams
    inputStream.open(this->inputFileName.c_str(), ios::binary);
    if (!inputStream.good()) {
	cleanup();
	RETURN_ERROR(ISD_FILE_ERROR);
    }
    outputStream.open(this->outputFileName.c_str(), ios::binary);
    if (!outputStream.good()) {
	cleanup();
	RETURN_ERROR(ISD_FILE_ERROR);
    }

    //read magic codes and check if compatible
    string magic = ISDVIEWER_MAGIC_TEMPLATE;
    if ((retVal = readMagic(magic)) != ISD_SUCCESS) {
	cleanup();
	RETURN_ERROR(retVal);
    }
	
    //note: version numbers are encoded in ascii and at most 1 char long each
    int majorVersion = static_cast<int>(magic[9])-48;
    int minorVersion = static_cast<int>(magic[11])-48;
    if (magic.substr(0, 9)!=string(ISDVIEWER_MAGIC_TEMPLATE).substr(0, 9) || 
	majorVersion>MAX_MAJOR_SUPPORTED_VERSION || 
	minorVersion>MAX_MINOR_SUPPORTED_VERSION)
    {
	cleanup();
	RETURN_ERROR(ISD_VERSION_ERROR);
    }

    //skip the RFB version and security entries; we don't need them
    inputStream.seekg(20, ios::cur);
    
    //read the RFB init data
    if ((retVal = readRfbInit()) != ISD_SUCCESS) {
	cleanup();
	RETURN_ERROR(retVal);
    }

    //skip the VNC server name
    inputStream.seekg(isdViewerHeader.nameLength, ios::cur);
    /*
    char buf[isdViewerHeader.nameLength];
    inputStream.read(buf, isdViewerHeader.nameLength);
    vncServerName.assign(buf, isdViewerHeader.nameLength);
    */

    //no need to wait to swap the pixel max values anymore,
    //since we are using a standardized pixel format
    isdViewerHeader.format.redMax = Swap16IfLE(isdViewerHeader.format.redMax);
    isdViewerHeader.format.greenMax = Swap16IfLE(isdViewerHeader.format.greenMax);
    isdViewerHeader.format.blueMax = Swap16IfLE(isdViewerHeader.format.blueMax);

    needsNormalization = !(isdViewerHeader.format.bitsPerPixel==ISD_FORMAT_BITS_PER_PIXEL &&
	isdViewerHeader.format.depth==ISD_FORMAT_COLOUR_DEPTH &&
	isdViewerHeader.format.bigEndian==ISD_FORMAT_BIG_ENDIAN &&
	isdViewerHeader.format.trueColour==ISD_FORMAT_TRUE_COLOUR &&
	isdViewerHeader.format.redMax==ISD_FORMAT_RED_MAX &&
	isdViewerHeader.format.greenMax==ISD_FORMAT_GREEN_MAX &&
	isdViewerHeader.format.blueMax==ISD_FORMAT_BLUE_MAX &&
	isdViewerHeader.format.redShift==ISD_FORMAT_RED_SHIFT &&
	isdViewerHeader.format.greenShift==ISD_FORMAT_GREEN_SHIFT &&
	isdViewerHeader.format.blueShift==ISD_FORMAT_BLUE_SHIFT);

    //write the output init data
    if ((retVal = writeOutputHeader()) != ISD_SUCCESS) {
	cleanup();
	RETURN_ERROR(retVal);
    }

    //init time and process the actual data
    firstFrame = true;
    while ((retVal = processFramebufferUpdate()) == ISD_SUCCESS) {
	//notify and check the callback
	if (callbackClass!=NULL) {
	    if (callbackClass->wasCancelled()) {
		abortCleanup();
		break;
	    }
	    else {
		callbackClass->procentDone((float)inputStream.tellg()/(float)inputFileSize*100.0);
	    }
	}
    }
    //the only good exit value is EOF
    if (retVal!=ISD_EOF_ERROR){
	cleanup();
	//reset the error flags
	inputStream.clear();
	RETURN_ERROR(retVal);
    }

    cleanup();

    RETURN_SUCCESS;
}

//-----PROTECTED FUNCTIONS-----
void ISDVncLogImporter::abortCleanup()
{
    
}
ISDObject::ISDErrorCode ISDVncLogImporter::readMagic(string& magic)
{
    if (inputStream.good()) {
	//magic always occurs at the beginning of the file
	inputStream.seekg(0);
	char buf[magic.length()];
	inputStream.read(buf, magic.length());
	magic.assign(buf, magic.length());
	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }
}
ISDObject::ISDErrorCode ISDVncLogImporter::readRfbInit()
{
    if (inputStream.good()) {
	inputStream.read(reinterpret_cast<char*>(&isdViewerHeader), RFB_INIT_MSG_SIZE);
       
	isdViewerHeader.framebufferWidth = Swap16IfLE(isdViewerHeader.framebufferWidth);
	isdViewerHeader.framebufferHeight = Swap16IfLE(isdViewerHeader.framebufferHeight);
	//wait to endian-swap the RGB max values after they are written to the instrudeo file
	//doing so, we can write the pixel-format structure as an U8 array
	/*
	isdViewerHeader.format.redMax = Swap16IfLE(isdViewerHeader.format.redMax);
	isdViewerHeader.format.greenMax = Swap16IfLE(isdViewerHeader.format.greenMax);
	isdViewerHeader.format.blueMax = Swap16IfLE(isdViewerHeader.format.blueMax);
	*/
	isdViewerHeader.nameLength = Swap32IfLE(isdViewerHeader.nameLength);
	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }
}

ISDObject::ISDErrorCode ISDVncLogImporter::writeOutputHeader()
{
    if (outputStream.good()) {
	string magic = ISDDataFile::getMagicString();
	outputStream.write(magic.c_str(), magic.length());
	U16 fWidth = Swap16IfLE(isdViewerHeader.framebufferWidth);
	U16 fHeight = Swap16IfLE(isdViewerHeader.framebufferHeight);
	outputStream.write(reinterpret_cast<char*>(&fWidth), sizeof(U16));
	outputStream.write(reinterpret_cast<char*>(&fHeight), sizeof(U16));

	//We use our own, standardized pixel format for the ISD file
	RFBPixelFormat writeFormat;
	writeFormat.bitsPerPixel = static_cast<U8>(ISD_FORMAT_BITS_PER_PIXEL);
	writeFormat.depth = static_cast<U8>(ISD_FORMAT_COLOUR_DEPTH);
	writeFormat.bigEndian = static_cast<U8>(ISD_FORMAT_BIG_ENDIAN);
	writeFormat.trueColour = static_cast<U8>(ISD_FORMAT_TRUE_COLOUR);
	writeFormat.redMax = Swap16IfLE(static_cast<U16>(ISD_FORMAT_RED_MAX));
	writeFormat.greenMax = Swap16IfLE(static_cast<U16>(ISD_FORMAT_GREEN_MAX));
	writeFormat.blueMax = Swap16IfLE(static_cast<U16>(ISD_FORMAT_BLUE_MAX));
	writeFormat.redShift = static_cast<U8>(ISD_FORMAT_RED_SHIFT);
	writeFormat.greenShift = static_cast<U8>(ISD_FORMAT_GREEN_SHIFT);
	writeFormat.blueShift = static_cast<U8>(ISD_FORMAT_BLUE_SHIFT);

	outputStream.write(reinterpret_cast<char*>(&writeFormat), RFB_PIXEL_FORMAT_SIZE);
	
	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }   
}

ISDObject::ISDErrorCode ISDVncLogImporter::processFramebufferUpdate()
{
    RFBRectHeader inHeader;
    RFBFbUpdate framebufferUpdate;
    int relTime;
    ISDObject::ISDErrorCode retVal;

    if (inputStream.good()) {
	//read framebuffer-update header
	inputStream.read(reinterpret_cast<char*>(&framebufferUpdate), RFB_FB_UPDATE_SIZE);
       
	//check for EOF
	if (inputStream.eof()) {
	    RETURN_ERROR(ISD_EOF_ERROR);
	}

	framebufferUpdate.startTime.tv_sec = Swap32IfLE(framebufferUpdate.startTime.tv_sec);
	framebufferUpdate.startTime.tv_usec = Swap32IfLE(framebufferUpdate.startTime.tv_usec);
	framebufferUpdate.numRects = Swap16IfLE(framebufferUpdate.numRects);

	//read the different rectangles in this framebuffer update
	for (int i=0;i<framebufferUpdate.numRects; i++) {
	    //read rectangle header
	    inputStream.read(reinterpret_cast<char*>(&inHeader), RFB_RECT_HEADER_SIZE);

	    inHeader.x = Swap16IfLE(inHeader.x);
	    inHeader.y = Swap16IfLE(inHeader.y);
	    inHeader.w = Swap16IfLE(inHeader.w);
	    inHeader.h = Swap16IfLE(inHeader.h);
	    inHeader.encType = Swap32IfLE(inHeader.encType);

	    //check for errors
	    if ((inHeader.x + inHeader.w > isdViewerHeader.framebufferWidth) ||
		(inHeader.y + inHeader.h >  isdViewerHeader.framebufferHeight))
	    {
		LOG_CRITICAL("Encountered a rect that exceeded the framebuffer dimensions, aborting.");
		RETURN_ERROR(ISD_IMPORT_ERROR);
	    }

	    //zero-sized rectangle, skip
	    if (inHeader.h * inHeader.w == 0) {
		continue;
	    }

	    ISDRectangle* rect;
	    switch(inHeader.encType) {
		case RFB_ENCODING_RAW:
		    //init starttime with time of first frame if necessary
		    if (firstFrame) {
			startTime.tv_sec = framebufferUpdate.startTime.tv_sec;
			startTime.tv_usec = framebufferUpdate.startTime.tv_usec;
			
			firstFrame = false;
		    }
		    
		    //calc relative time to startframe in millisecs
		    relTime = (int)((framebufferUpdate.startTime.tv_sec-startTime.tv_sec)*1000
			+ (framebufferUpdate.startTime.tv_usec-startTime.tv_usec) / 1e3);
		    
		    /*
		     * create the rect
		     * seekback will be calculated, init to 0
		     * we don't use the dataLen, init to 0
		     * Note: the filePos of the rect equals the current position of the outputStream
		     */
		    rect = new ISDRectangle((int)relTime, inHeader.x, inHeader.y, inHeader.w, inHeader.h,
						0, defaultRectEncoding, (int)outputStream.tellp(),
						NULL, NULL, 0);

		    //add the rectangle to the rectList
		    if (appendRectToList(rect) != ISD_SUCCESS) {
			LOG_WARNING("Error while appending a rect to the restList");
			RETURN_ERROR(ISD_IMPORT_ERROR);
		    }
		    
		    //calculate the seekback of the new rectangle
		    if (calcSeekBackRects(rect) != ISD_SUCCESS) {
			LOG_WARNING("Error while calculating seekback for rectangle.");
			RETURN_ERROR(ISD_IMPORT_ERROR);
		    }

		    //encode and record the rect
		    if ((retVal = recordRectangle(rect)) != ISD_SUCCESS) {
			LOG_CRITICAL("Error occurred while writing/encoding a rectangle.");
			RETURN_ERROR(retVal);
		    }
		    
		    break;

		default:
		    LOG_CRITICAL("Unknown rfb rect encoding or encoding not (yet?) supported, aborting.");
		    RETURN_ERROR(ISD_IMPORT_ERROR);
	    }
	}

	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }   
}

ISDObject::ISDErrorCode ISDVncLogImporter::recordRectangle(ISDRectangle* rect)
{
    if (inputStream.good() && outputStream.good() && rect!=NULL){
	
	//init buffers
	//Note: we need space for 4 bytes per pixel, as specified in the ISD format,
	//      but this will be re-allocated if necessary by normalizePixels
	int dataSize = rect->width*isdViewerHeader.format.bitsPerPixel/8*rect->height;
	char* data = (char*)malloc(dataSize);
	
	//the compressBound() adds the overhead-size needed to compress.
	//We need to allocate space for the ISD pixel format here, since that is the size
	//  that will be allocated by normalizePixels()
	char returnBuffer[compressBound(rect->width*ISD_FORMAT_BITS_PER_PIXEL/8*rect->height)];
	uLongf compressedDataSize;
	
	//read the data into memory
	inputStream.read(data, dataSize);

	//lazy checking, method will not be called if first test fails
	if (needsNormalization && normalizePixels(rect, data, dataSize) != ISD_SUCCESS) {
	    free(data);
	    LOG_CRITICAL("Erro while normalizing pixels in rect.");
	    RETURN_ERROR(ISD_ENCODING_ERROR);
	}

	/*
	 * Encode the data
	 * In version 0.1.3, this was updated to use zlib's (utility) function compress2,
	 * instead of the intermediary functions in the rle.c file (which was deleted in this release).
	 * compressedDataSize contains the size of the destination buffer on entry.
	 */
	compressedDataSize = compressBound(rect->width*ISD_FORMAT_BITS_PER_PIXEL/8*rect->height);
	int retVal = compress2((Bytef*)returnBuffer, (uLongf*)&compressedDataSize,
			       (Bytef*)data, dataSize,
			       Z_DEFAULT_COMPRESSION);
	if (retVal != Z_OK)
	{
	    free(data);
	    LOG_CRITICAL("Error during encoding of a rect ("+ISDUtils::getInstance()->intToString(retVal)+"), aborting.");
	    RETURN_ERROR(ISD_ENCODING_ERROR);
	}

	//write the header
	U32 time = Swap32IfLE(rect->frameTime);
	U16 x = Swap16IfLE(rect->x);
	U16 y = Swap16IfLE(rect->y);
	U16 w = Swap16IfLE(rect->width);
	U16 h = Swap16IfLE(rect->height);
	U32 compSize = Swap32IfLE(compressedDataSize);
	U8 encType = rect->encType;
	U32 seekBack = Swap32IfLE(rect->seekBackPos);
	outputStream.write(reinterpret_cast<char*>(&time), sizeof(U32));
	outputStream.write(reinterpret_cast<char*>(&x), sizeof(U16));
	outputStream.write(reinterpret_cast<char*>(&y), sizeof(U16));
	outputStream.write(reinterpret_cast<char*>(&w), sizeof(U16));
	outputStream.write(reinterpret_cast<char*>(&h), sizeof(U16));
	outputStream.write(reinterpret_cast<char*>(&compSize), sizeof(U32));
	outputStream.write(reinterpret_cast<char*>(&encType), sizeof(U8));
	outputStream.write(reinterpret_cast<char*>(&seekBack), sizeof(U32));

	//write the data
	outputStream.write(returnBuffer, compressedDataSize);
	
	if (!outputStream.good()){
	    free(data);
	    RETURN_ERROR(ISD_FILE_ERROR);
	}
	
	free(data);
	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }  
}
ISDObject::ISDErrorCode ISDVncLogImporter::normalizePixels(ISDRectangle* rect, char*& buffer, int& bufLen)
{
    //make the 'buffer' large enough
    int pixelSize = isdViewerHeader.format.bitsPerPixel/8;
    char* bufPtr = buffer;

    int newPixelSize = ISD_FORMAT_BITS_PER_PIXEL/8;
    int newBufferSize = rect->width*newPixelSize*rect->height;
    char* newBuffer = (char*)malloc(newBufferSize);
    char* newBufPtr = newBuffer;

    unsigned long pixel;
    float redScale = (float)isdViewerHeader.format.redMax / (float)ISD_FORMAT_RED_MAX;
    float greenScale = (float)isdViewerHeader.format.greenMax / (float)ISD_FORMAT_GREEN_MAX;
    float blueScale = (float)isdViewerHeader.format.blueMax / (float)ISD_FORMAT_BLUE_MAX;

    for (int i=0;i<bufLen;i+=pixelSize) {
	pixel = 0;
	memcpy(&pixel, bufPtr, pixelSize);

	U8 alpha = 0;
	//check for alpha channel
	if (isdViewerHeader.format.bitsPerPixel != isdViewerHeader.format.depth) {
	    alpha = (pixel >> isdViewerHeader.format.depth);
	}
	//remark: we need to re-scale the colors to the new colorspace
	U8 red = static_cast<U8>(((pixel >> isdViewerHeader.format.redShift) & isdViewerHeader.format.redMax) / redScale);
	U8 green = static_cast<U8>(((pixel >> isdViewerHeader.format.greenShift) & isdViewerHeader.format.greenMax) / greenScale);
	U8 blue = static_cast<U8>(((pixel >> isdViewerHeader.format.blueShift) & isdViewerHeader.format.blueMax) / blueScale);
	
	newBufPtr[0] = blue;
	newBufPtr[1] = green;
	newBufPtr[2] = red;
	newBufPtr[3] = alpha;

	newBufPtr += newPixelSize;

	bufPtr += pixelSize;
    }

    free(buffer);
    buffer = newBuffer;
    bufLen = newBufferSize;

    RETURN_SUCCESS;
}

void ISDVncLogImporter::cleanup()
{
    if (inputStream.good())
	inputStream.close();

    if (outputStream.good())
	outputStream.close();
}
