/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDUTILS_H
#define ISDUTILS_H

/**
 * Utility/Helper Singleton class that provides the entire application with various routines.
 *
 * @author Bram Biesbrouck <b@beligum.org>
*/

#include <string>
#include <list>
#include <glibmm/ustring.h>
#include <unistd.h>//for file access checking

#include <libinstrudeo/isdobject.h>

using namespace std;

//-----MACROS-----
//Note: -p means: no error if exists, make parents if needed
#define CREATE_DIR(dirName) \
     system(("mkdir -p " + dirName).c_str()) 

#define REMOVE_FILE(fileName) \
     system(("rm -rf " + fileName).c_str())

#define REMOVE_DIR(dirName) \
     system(("rm -rf " + dirName).c_str())

#define VALID_FILE(fileName) \
     !access(fileName.c_str(), R_OK)

#define ENDIAN_TEST 1
#define Swap16IfLE(s) \
    (ENDIAN_TEST ? ((((s) & 0xff) << 8) | (((s) >> 8) & 0xff)) : (s))
#define Swap32IfLE(l) \
    (ENDIAN_TEST ? ((((l) & 0xff000000) >> 24) | \
			     (((l) & 0x00ff0000) >> 8)  | \
			     (((l) & 0x0000ff00) << 8)  | \
			     (((l) & 0x000000ff) << 24))  : (l))

#define TEMP_FILE_TEMPLATE "/tmp/instrudeoXXXXXX"
#define TEMP_DIR_TEMPLATE "/tmp/instrudeoXXXXXX"
#define DIR_DELIM "/"
#define MAIL_PROGRAM "/usr/bin/mail"

class ISDUtils : public ISDObject
{
 public:

    /**
     * Static accesser that must be called to get/initialise an instance of this class.
     * So use ISDUtils::getInstance()-><method> to use one of the methods in this class.
     *
     * @return An instance of this class.
     */
    static ISDUtils* getInstance();

    /**
     * Creates a temp file in the /tmp directory with the defined filename template.
     *
     * @param name Returns the filename of the newly created file.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode createTempFile(string& name);

    /**
     * Creates a temp directory in the /tmp directory with the defined dirname template.
     * (with trailing slash)
     *
     * @param name Returns the dirname of the newly created file.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode createTempDir(string& name);

    /**
     * Converts an int to a string
     *
     * @param i The int;
     * @return The string.
     */
    string intToString(int i);

    /**
     * Converts a string to an int.
     *
     * @param numStr The string that must be converted to an int.
     * @return The int.
     */
    int stringToInt(string& numStr);

    /**
     * Converts a float to a string
     *
     * @param f The float;
     * @return The string.
     */
    string floatToString(float f);

    /**
     * Converts a string to a float.
     *
     * @param numStr The string that must be converted to a float.
     * @return The float.
     */
    float stringToFloat(string& numStr);

    /**
     * Returns the directory name of this file path (with trailing slash)
     * eg. /home/john/test.txt returns /home/john/
     *
     * @param fileName The file name
     * @return The directorystring.
     */
    string getFileDirectory(string fileName);

    /**
     * Returns a filename without the directories eg. /home/john/test.txt returns test.txt
     *
     * @param fileName The file name
     * @return The pure file string.
     */
    string stripFileDirectories(string fileName);

    /**
     * Explodes the str, using substr as a delimiter and adds every exploded string to the list.
     *
     * @param str The source-string.
     * @param substr The delimiter to explode around.
     * @return The exploded list of strings.
     */
    list<Glib::ustring> explodeString(Glib::ustring str, Glib::ustring substr);

    /**
     * Returns the file size in bytes.
     *
     * @param fileName The fileName.
     * @return The size or -1 on error.
     */
    off64_t getFileSizeInBytes(string& fileName);

    /**
     * Returns the extension (without the dot) of the fileName.
     * If the file had no extension, "" is returned.
     *
     * @param fileName The filename to substract the extension from.
     * @return The extensionstring or ""
     */
    string getExtension(string fileName);

    /**
     * Generates a random string that can be used as, eg. filename.
     *
     * @return the generated string.
     */
    string getRandomString();

    /**
     * Sends a mail to the recipient.
     *
     * @param recipient The target email address.
     * @param subject The subject of the email.
     * @param body The body of the email.
     * @return Returns true on success, false otherwise.
     */
    bool sendMail(string& recipient, string& subject, string& body);

 protected:
    /**
     * Private constructor
     */
    ISDUtils();
    
 private:
    //-----VARIABLES-----
    static ISDUtils* instance;
};

#endif
