/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDIMPORTER_H
#define ISDIMPORTER_H

/**
 * The abstract super class for all importers
 * @author Bram Biesbrouck <b@beligum.org>
*/

#include <string>
#include <iostream>
#include <fstream>

#include <libinstrudeo/isdobject.h>
#include <libinstrudeo/isdseekbackcalculator.h>

using namespace std;

class ISDRectangle;
class ISDProgressCallback;

class ISDImporter : public ISDObject
{

public:

    /**
     * Loads a recording from a different input format to the instrudeo file format.
     * The object is merely contructed with all necessary field filled in. To actually 
     * parse the file and convert it to an instrudeo file, use the "convertTo" function. 
     *
     * @param inputFileName The filename of the file that was produced by the foreign program.
     */
    ISDImporter(const string inputFileName);

    virtual ~ISDImporter();

    /**
     * Converts the file that was given in the constructor to the standard instrudeo file format.
     * OutputFileName will be filled with the tempFile name used if createNew was True.
     * 
     * @param outputFileName The name of the file to write the data to.
     *                       Will be filled with the path of the newly created (temporary) file if createNew was true.
     * @param callbackClass Can be set to monitor the progress or NULL to discard that.
     * @param createNew Indicates if a new file must be created (true) or the existing one must be used (false)
     * @return Returns a code that indicates success or failure.
     */
    virtual ISDErrorCode convertTo(string& outputFileName,
				   ISDProgressCallback* callbackClass = NULL,
				   bool createNew=false) = 0;

 protected:
    //-----METHODS-----
    /**
     * Because of the sequential rectangle-update structure of the file format,
     * the data underlying a rect in an update gets lost, so when one needs to
     * seek back in a video, the image must be "rebuild" from the first rect.
     * To eliminate this, this function calculates the smallest amount of rects
     * to go back, and to rebuild the image from that rectangle on, onto the current
     * image. This number of "seekBacks" is significantly smaller then the total amount
     * of rects between this rect and the first rect, resulting in a much better performance.
     * If this function isn't called, the default behaviour is to go back to the first rect.
     * Make sure the rectList variables are properly initialised before calling this function.
     *
     * See the ISDSeekBackCalculator class for an explanation of the used algorithm
     * @see ISDSeekBackCalculator
     *
     * @param rect Pointer to a rect in the list to calculate the seekback for.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode calcSeekBackRects(ISDRectangle* rect);

    /**
     * Appends the supplied rect to the end of the rectList.
     *
     * @param rect The rectangle to append.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode appendRectToList(ISDRectangle* rect);
    
    //-----VARIABLES-----
    int defaultRectEncoding;
    string inputFileName;
    int inputFileSize;
    string outputFileName;
    ifstream inputStream;
    ofstream outputStream;
    ISDSeekBackCalculator seekBackCalculator;
    ISDRectangle* rectListHead;
    ISDRectangle* rectListTail;
};

#endif
