/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDEXPORTER_H
#define ISDEXPORTER_H

/**
 * Abstract superclass for all exporter classes.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>
#include <glibmm/ustring.h>

#include <libinstrudeo/isdobject.h>

using namespace std;

#define EXPORT_DEFAULT_QUALITY 100
#define EXPORT_DEFAULT_FRAMERATE 24

class ISDRecording;
class ISDProgressCallback;
class ISDRectangle;
class ISDCommentbox;

class ISDExporter : public ISDObject
{
 public:
    //-----CONSTANTS-----
    enum ISDExportFormat {
	ISD_EXPORT_AUTO_FORMAT,
	ISD_EXPORT_AVI_FORMAT
    };

    //-----CONSTRUCTORS-----
    /**
     * Constructor that initialises the exporter with specified values.
     *
     * @param outFileName The name of the file to export.
     * @param rec The recording to export.
     * @param lang The language of the comments to use, or NULL to disable comments.
     * @param callbackClass Can be set to monitor the progress or NULL to discard that.
     * @param quality The quality of the exported video. Range [0, 100].
     * @param framerate The framerate of the video.
     */
    ISDExporter(string outFileName, ISDRecording* rec,
		Glib::ustring* lang, ISDProgressCallback* callbackClass,
		int quality = EXPORT_DEFAULT_QUALITY,
		float framerate = EXPORT_DEFAULT_FRAMERATE);
    
    virtual ~ISDExporter();
    
    //-----METHODS-----
    /**
     * Export the video to the specified file in the constructor.
     * If the file exists, it is deleted first.
     * The optional parameter specifies the pixelbuffer that should be used instead
     * of grabbing the frames from the framebuffer. If offers the possibility
     * to use an offscreen rendering context that doesn't support glReadPixels()
     * (like OSMESA).
     *
     * @param pixelBuffer An optional offscreen pixel buffer.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode exportRecording(char* pixelBuffer = NULL);
    
    /**
     * Does the actual exporting-arithmetic.
     *
     * @return Returns a code that indicates success or failure.
     */
    virtual ISDErrorCode doExport(char* pixelBuffer) = 0;

 protected:
    //-----METHODS-----
    virtual void cleanup();
    void cleanupMesa();
    /**
     * Checks if the rectangle intersects (or overlaps) with the comment.
     *
     * @param rect The rectangle to check.
     * @param comment The commentbox to check.
     * @return true if there's overlap, false if not
     */
    bool intersects(ISDRectangle* rect, ISDCommentbox* comment);

    //-----VARIABLES-----
    string fileName;
    ISDRecording* recording;
    int quality;
    float framerate;
    bool initOK;
    Glib::ustring* lang;
    ISDProgressCallback* callbackClass;
};

#endif
