/** Copyright  2003 by Jean-Hugues de Raigniac <jhraigniac@workingfrog.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.workingfrog.i18n.swing;

import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;

import org.workingfrog.i18n.util.LocaleEvent;
import org.workingfrog.i18n.util.LocaleListener;
import org.workingfrog.i18n.util.Translator;
import org.workingfrog.i18n.util.Wrapper;

/**
 * Wraps a javax.swing.AbstractAction in an i18n version 
 *
 * @author Jean-Hugues de Raigniac
 */

public class I18NActionWrapper extends I18NAction implements LocaleListener {

    /** i18n key */
    private String i18nKey = "action.error";

    /** action */
    private AbstractAction action;

    public I18NActionWrapper () {
        super();
    }

    /**
     * Basic I18NActionWrapper.
     *
     * @param action an AbstractAction
     */
    public I18NActionWrapper (AbstractAction action) {
        this.action = action;
        i18nKey = Wrapper.getI18NKey((String) action.getValue(Action.NAME));
        Object[] keys = action.getKeys();
        for (int i = 0; i < keys.length; i++) {
            putValue((String) keys[i], action.getValue((String) keys[i]));
        }

        String value = Translator.checkValue(i18nKey, this);
        action.putValue(NAME, value);
        putValue(NAME, value);
    }

    public String getKey () {
        return i18nKey;
    }

    public Object[] getKeys () {
        return action.getKeys();
    }

    public Object getValue (String key) {
        return (key == Action.NAME) ? i18nKey : action.getValue(key);
    }

    /**
     * Must be implemented by children.
     *
     * @param event The event fired
     */
    public void actionPerformed (ActionEvent event) {
        action.actionPerformed(event);
    }

    /**
     * Updates Action.NAME when a LocaleEvent is fired.
     *
     * @param event contains the new Locale
     */
    public void localeChanged (LocaleEvent event) {
        action.putValue(NAME, Translator.checkValue(i18nKey, this));
        //action.localeChanged(event);
        putValue(NAME, Translator.checkValue(i18nKey, this));
    }
}
