// -*- C++ -*-
//
// The Hoard Multiprocessor Memory Allocator
// www.hoard.org
//
// Author: Emery Berger, http://www.cs.umass.edu/~emery
//
// Copyright (c) 1998-2003, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * @file wrapper.cpp
 *
 * Implementations of malloc(), free(), etc. in terms of Hoard.
 * This lets us link in hoard in place of the stock malloc.
 *
 */

#include <stdlib.h>
#include <string.h>
#if !defined(_WIN32)
#include <strings.h>
#include <pthread.h>
#endif
#if !defined(__SUNPRO_CC) || __SUNPRO_CC > 0x420
#include <new>
#endif

#include "arch-specific.h"
#include "config.h"
#include "threadheap.h"
#include "processheap.h"

extern "C" void hoardPrepareFork (void);
extern "C" void hoardParentFork (void);

//
// Access exactly one instance of the process heap
// (which contains the thread heaps).
// We create this object dynamically to avoid bloating the object code.
//

inline static processHeap * getAllocator (void) {
  static char * buf = (char *) hoardGetMemory (sizeof(processHeap));
  static processHeap * theAllocator = new (buf) processHeap;
  return theAllocator;
}

// Ensure that the pthread_atfork handlers get installed very early.

class InitializeHoard {
public:
  InitializeHoard (void) {
    volatile processHeap * ph = getAllocator();
#if !defined(_WIN32)
    //    pthread_atfork(hoardPrepareFork, hoardParentFork, hoardParentFork);
#endif
  }
};

static InitializeHoard _hoardInitStub;

extern "C" void hoardPrepareFork (void) {
  getAllocator()->lockAll();
}

extern "C" void hoardParentFork (void) {
  getAllocator()->unlockAll();
}

#define HOARD_MALLOC(x) malloc(x)
#define HOARD_FREE(x) free(x)
#define HOARD_REALLOC(x,y) realloc(x,y)
#define HOARD_CALLOC(x,y) calloc(x,y)
#define HOARD_MEMALIGN(x,y) memalign(x,y)
#define HOARD_VALLOC(x) valloc(x)
#define HOARD_GET_USABLE_SIZE(x) malloc_usable_size(x)
#define HOARD_STRDUP(x) strdup(x)


extern "C" void * HOARD_MALLOC(size_t);
extern "C" void HOARD_FREE(void *);
extern "C" void * HOARD_REALLOC(void *, size_t);
extern "C" void * HOARD_CALLOC(size_t, size_t);
extern "C" void * HOARD_MEMALIGN(size_t, size_t);
extern "C" void * HOARD_VALLOC(size_t);
extern "C" size_t HOARD_GET_USABLE_SIZE(void *);
extern "C" char * HOARD_STRDUP(const char *);

extern "C" void * HOARD_MALLOC (size_t sz)
{
  static processHeap * pHeap = getAllocator();
  if (sz == 0) {
	  sz = 1;
  }
  void * addr = pHeap->getHeap(pHeap->getHeapIndex()).malloc (sz);
  return addr;
}

extern "C" void * HOARD_CALLOC (size_t nelem, size_t elsize)
{
  static processHeap * pHeap = getAllocator();
  size_t sz = nelem * elsize;
  if (sz == 0) {
	  sz = 1;
  }
  void * ptr = pHeap->getHeap(pHeap->getHeapIndex()).malloc (sz);
  // Zero out the malloc'd block.
  memset (ptr, 0, sz);
  return ptr;
}

extern "C" void HOARD_FREE (void * ptr)
{
  static processHeap * pHeap = getAllocator();
  pHeap->free (ptr);
}


extern "C" void * HOARD_MEMALIGN (size_t alignment, size_t size)
{
  return HOARD_MALLOC (size);
#if 0
  static processHeap * pHeap = getAllocator();
  void * addr = pHeap->getHeap(pHeap->getHeapIndex()).memalign (alignment, size);
  return addr;
#endif
}


extern "C" void * HOARD_VALLOC (size_t size)
{
  return HOARD_MEMALIGN (hoardGetPageSize(), size);
}


extern "C" void * HOARD_REALLOC (void * ptr, size_t sz)
{
  if (ptr == NULL) {
    return HOARD_MALLOC (sz);
  }
  if (sz == 0) {
    HOARD_FREE (ptr);
    return NULL;
  }

  // If the existing object can hold the new size,
  // just return it.

  size_t objSize = threadHeap::objectSize (ptr);

  if (objSize >= sz) {
    return ptr;
  }

  // Allocate a new block of size sz.

  void * buf = HOARD_MALLOC (sz);

  if (buf != NULL) {

    // Copy the contents of the original object
    // up to the size of the new block.
    
    size_t minSize = (objSize < sz) ? objSize : sz;
    memcpy (buf, ptr, minSize);
  }

  // Free the old block.
  
  HOARD_FREE (ptr);

  // Return a pointer to the new one.

  return buf;
}

extern "C" size_t HOARD_GET_USABLE_SIZE (void * ptr)
{
  return threadHeap::objectSize (ptr);
}


extern "C" char * HOARD_STRDUP (const char * s)
{
  char * newString = NULL;
  if (s != NULL) {
    if ((newString = (char *) HOARD_MALLOC(strlen(s) + 1))) {
      strcpy(newString, s);
    }
  }
  return newString;
}


#if 0
extern "C" void malloc_stats (void)
{
  TheWrapper.TheAllocator()->stats();
}
#endif

#if defined(_WIN32)
void * operator new (size_t size)
{
  return malloc (size);
}

void * operator new (size_t size, const std::nothrow_t&) throw() {
  return malloc (size);
} 

void * operator new[] (size_t size)
{
  return malloc (size);
}

void * operator new[] (size_t size, const std::nothrow_t&) throw() {
  return malloc (size);
} 

void operator delete (void * ptr)
{
  free (ptr);
}

void operator delete[] (void * ptr)
{
  free (ptr);
}

extern "C" void * __cdecl _heap_alloc_base (size_t size)
{
	return malloc (size);
}

void * operator new (unsigned int cb, int, const char *, int)
{
	return ::operator new (cb);
}

void operator delete(void * p, int, const char *, int)
{
  ::operator delete(p);
}

extern "C" void * __cdecl _malloc_dbg (size_t sz, int, const char *, int) {
	return malloc (sz);
}

extern "C" void * __cdecl _malloc_base (size_t sz) {
	return malloc (sz);
}

extern "C" void * __cdecl _calloc_dbg (size_t num, size_t size, int, const char *, int) {
	return calloc (num, size);
}

extern "C" void * __cdecl _calloc_base (size_t num, size_t size) {
	return calloc (num, size);
}

extern "C" void * __cdecl _realloc_dbg (void * ptr, size_t newSize, int, const char *, int) {
	return realloc (ptr, newSize);
}

extern "C" void * __cdecl _realloc_base (void * ptr, size_t newSize) {
	return realloc (ptr, newSize);
}

extern "C" void __cdecl _free_dbg (void * ptr, int) {
	free (ptr);
}

extern "C" void __cdecl _free_base (void * ptr) {
	free (ptr);
}


/* Don't allow expand to work ever. */

extern "C" void * __cdecl _expand (void * ptr, size_t sz) {
  return NULL;
}

extern "C" void * __cdecl _expand_base (void * ptr, size_t sz) {
  return NULL;
}

extern "C" void * __cdecl _expand_dbg (void * ptr, size_t sz, int, const char *, int) {
  return NULL;
}

extern "C" void * __cdecl _nh_malloc (size_t sz, int) {
	return malloc (sz);
}

extern "C" void * __cdecl _nh_malloc_base (size_t sz, int) {
	return malloc (sz);
}

extern "C" void * __cdecl _nh_malloc_dbg (size_t sz, size_t, int, int, const char *, int) {
	return malloc (sz);
}
#endif

