// -*- C++ -*-
//
// The Hoard Multiprocessor Memory Allocator
// www.hoard.org
//
// Author: Emery Berger, http://www.cs.umass.edu/~emery
//
// Copyright (c) 1998-2003, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#ifndef _THREADHEAP_H_
#define _THREADHEAP_H_

#include "config.h"


#include "hoardheap.h"

class processHeap; // forward declaration

/**
* @class threadHeap
* @brief We use one threadHeap for each thread (processor).
* @author Emery Berger <http://www.cs.umass.edu/~emery>
*/

class threadHeap : public hoardHeap {

public:

  threadHeap (void);

  // Memory allocation routines.

  /// Return a new object of at least sz bytes.
  void * malloc (const size_t sz);

  /// Deallocate an object.
  void free (void * ptr);

#if 0
  /// Return an aligned object.
  inline void * memalign (size_t alignment, size_t sz);
#endif

  /// Find out how large an allocated object is.
  inline static size_t objectSize (void * ptr);

  /// Set our process heap.
  inline void setpHeap (processHeap * p);

private:

  // Prevent copying and assignment.
  threadHeap (const threadHeap&);
  const threadHeap& operator= (const threadHeap&);

  /// Our process heap.
  processHeap *	_pHeap;

#if 0
  // We cache freed objects belonging to exactly one superblock, for now.
  superblock * _currentSuperblock;
  block * _freelistHead;
  block * _freelistTail;
  int _freelistLength;
  int cnt;
#endif

  // We insert a cache pad here to avoid false sharing (the
  // processHeap holds an array of threadHeaps, and we don't want
  // these to share any cache lines).
  double _pad[CACHE_LINE / sizeof(double)];
};

#if 0
void * threadHeap::memalign (size_t alignment,
			     size_t size)
{
  // Calculate the amount of space we need
  // to satisfy the alignment requirements.

  size_t newSize;

  // If the alignment is less than the required alignment,
  // just call malloc.
  if (alignment <= ALIGNMENT) {
    return this->malloc (size);
  }

  if (alignment < sizeof(block)) {
    alignment = sizeof(block);
  }

  // Alignment must be a power of two!
  assert ((alignment & (alignment - 1)) == 0);

  // Leave enough room to align the block within the malloced space.
  newSize = size + sizeof(block) + alignment;

  // Now malloc the space up with a little extra (we'll put the block
  // pointer in right behind the allocated space).

  void * ptr = this->malloc (newSize);
  if ((((size_t) ptr) & -((long) alignment)) == 0) {
    // ptr is already aligned, so return it.
    assert (((size_t) ptr % alignment) == 0);
    return ptr;

  } else {

    // Align ptr.
    char * newptr = (char *)
      (((size_t) ptr + alignment - 1) & -((long) alignment));

    // If there's not enough room for the block header, skip to the
    // next aligned space within the block..
    if ((size_t) newptr - (size_t) ptr < sizeof(block)) {
      newptr += alignment;
    }
    assert (((size_t) newptr % alignment) == 0);

    // Copy the block from the start of the allocated memory.
    block * b = ((block *) ptr - 1);

    assert (b->isValid());
    assert (b->getSuperblock()->isValid());
    
    // Make sure there's enough room for the block header.
    assert (((size_t) newptr - (size_t) ptr) >= sizeof(block));

    block * p = ((block *) newptr - 1);

    // Make sure there's enough room allocated for size bytes.
    assert (((size_t) p - sizeof(block)) >= (size_t) b);

    if (p != b) {
      assert ((size_t) newptr > (size_t) ptr);
      // Copy the block header.
      *p = *b;
      assert (p->isValid());
      assert (p->getSuperblock()->isValid());
      
      // Set the next pointer to point to b with the 1 bit set.
      // When this block is freed, it will be treated specially.
      p->setNext ((block *) ((size_t) b | 1));

    } else {
      assert (ptr != newptr);
    }

    assert (((size_t) ptr + newSize) >= ((size_t) newptr + size));
    return newptr;
  }
}
#endif

size_t threadHeap::objectSize (void * ptr) 
{
  // Find the superblock pointer.
  
  block * b = ((block *) ptr - 1);
  assert (b->isValid());
  superblock * sb = b->getSuperblock ();
  assert (sb);
  
  // Return the size.
  return sizeFromClass (sb->getBlockSizeClass());
}


void threadHeap::setpHeap (processHeap * p) 
{
  _pHeap = p; 
}

#endif // _THREADHEAP_H_

