// -*- C++ -*-
//
// The Hoard Multiprocessor Memory Allocator
// www.hoard.org
//
// Author: Emery Berger, http://www.cs.umass.edu/~emery
//
// Copyright (c) 1998-2003, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/**
* @class processHeap
* @brief The global heap - just one for the whole system.
* @author Emery Berger <http://www.cs.umass.edu/~emery>
*/

#ifndef _PROCESSHEAP_H_
#define _PROCESSHEAP_H_

#define MAX_THREADS 2048

#include "config.h"

#include <stdio.h>

#include "arch-specific.h"
#include "hoardheap.h"
#include "threadheap.h"

#if HEAP_LOG
#include "memstat.h"
#include "log.h"
#endif

class processHeap : public hoardHeap {

public:

  /// Always grab at least this many superblocks' worth of memory which we parcel out.
  enum { REFILL_NUMBER_OF_SUPERBLOCKS = 16 };

  processHeap (void);

  ~processHeap (void) {
#if HEAP_STATS
    stats();
#endif
  }

  /// Memory deallocation routine.
  void free (void * ptr);

  /// Print out statistics information.
  void stats (void);

  /// Get a thread heap index.
  inline int getHeapIndex (void);

  /// Get the thread heap with index i.
  inline threadHeap& getHeap (int i);

  /// Extract a superblock.
  inline superblock * acquire (const int c,
			       hoardHeap * dest);

  /// Insert a superblock.
  inline void release (superblock * sb);

#if HEAP_LOG
  /// Get the log for index i.
  inline Log<MemoryRequest>& getLog (int i);
#endif

#if HEAP_FRAG_STATS
  /// Declare that we have allocated an object.
  void setAllocated (int requestedSize,
		     int actualSize);

  /// Declare that we have deallocated an object.
  void setDeallocated (int requestedSize,
		       int actualSize);

  /// Return the number of wasted bytes at the high-water mark  (maxAllocated - maxRequested)
  inline int getFragmentation (void);

  int getMaxAllocated (void) {
    return _maxAllocated;
  }

  int getInUseAtMaxAllocated (void) {
    return _inUseAtMaxAllocated;
  }

  int getMaxRequested (void) {
    return _maxRequested;
  }
  
#endif

  // Note that the global lock order works like this:
  //   superblock:upLock
  //   threadHeap:Lock
  //   processHeap:Lock

#if 0
  void lockAll (void) {
    for (int i = 0; i < MAX_HEAPS; i++) {
      theap[i].lock();
      theap[i].lockAll();
    }
    lock();
  }

  /// @note This command is to be executed only by a pthread_atfork handler.
  void unlockAll (void) {
    for (int i = 0; i < MAX_HEAPS; i++) {
      theap[i].unlock();
      theap[i].unlockAll();
    }
    unlock();
  }
#endif

private:

  // Hide the lock & unlock methods.

  // Prevent copying and assignment.
  processHeap (const processHeap&);
  const processHeap& operator= (const processHeap&);

  /// The per-thread heaps.
  threadHeap theap[MAX_HEAPS];

#if HEAP_FRAG_STATS
  // Statistics required to compute fragmentation.  We cannot
  // unintrusively keep track of these on a multiprocessor, because
  // this would become a bottleneck.

  int _currentAllocated;
  int _currentRequested;
  int _maxAllocated;
  int _maxRequested;
  int _inUseAtMaxAllocated;
  int _fragmentation;

  /// A lock to protect these statistics.
  hoardLockType _statsLock;
#endif

#if HEAP_LOG
  Log<MemoryRequest> _log[MAX_HEAPS + 1];
#endif

};


threadHeap& processHeap::getHeap (int i)
{
  assert (i >= 0);
  assert (i < MAX_HEAPS);
  return theap[i];
}


#if HEAP_LOG
Log<MemoryRequest>& processHeap::getLog (int i)
{
  assert (i >= 0);
  assert (i < MAX_HEAPS + 1);
  return _log[i];
}
#endif


/// @return ceil(log_2(num)).
/// @note num must be positive.
static int lg (int num)
{
  assert (num > 0);
  int power = 0;
  int n = 1;
  // Invariant: 2^power == n.
  while (n < num) {
    n <<= 1;
    power++;
  }
  return power;
}


extern int tidmap[MAX_THREADS];

// Hash out the thread id to a heap and return an index to that heap.
int processHeap::getHeapIndex (void) {
//  int tid = hoardGetThreadID() & hoardHeap::_numProcessorsMask;
  int tid = hoardGetThreadID() & MAX_HEAPS_MASK;
  assert (tid < MAX_HEAPS);
  return tidmap[tid];
}


superblock * processHeap::acquire (const int sizeclass,
				   hoardHeap * dest)
{
  Guard hoardLock (this);

  // Remove the superblock with the most free space.
  superblock * maxSb = removeMaxSuperblock (sizeclass);
  if (maxSb) {
    maxSb->setOwner (dest);
  }

  return maxSb;
}


// Put a superblock back into our list of superblocks.
void processHeap::release (superblock * sb)
{
  assert (EMPTY_FRACTION * sb->getNumAvailable() > sb->getNumBlocks());

  Guard hoardLock (this);

  // Insert the superblock.
  insertSuperblock (sb->getBlockSizeClass(), sb, this);

}


#endif // _PROCESSHEAP_H_

