// -*- C++ -*-
//
// The Hoard Multiprocessor Memory Allocator
// www.hoard.org
//
// Author: Emery Berger, http://www.cs.umass.edu/~emery
//
// Copyright (c) 1998-2003, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#ifndef _ARCH_SPECIFIC_H_
#define _ARCH_SPECIFIC_H_

/**
 * @file arch-specific.h
 * @brief Architecture-specific definitions.
 * @author Emery Berger
 */

#include "config.h"

// Wrap architecture-specific functions.

#if defined(WIN32) || defined(__WIN32__) || defined(_WIN32)

// Windows

// Set maximal inlining.
#pragma inline_depth(255)
#define inline __forceinline
//#pragma comment(linker, "/FORCE:MULTIPLE")

#include <windows.h>
#include <process.h>
//typedef CRITICAL_SECTION	hoardLockType;
typedef volatile unsigned long hoardLockType;
typedef HANDLE 	hoardThreadType;

#elif USE_SPROC

// SGI's SPROC library

#include <sys/types.h>
#include <sys/prctl.h>

#else

// Generic UNIX

#if defined(__SVR4) // Solaris
#include <thread.h>
#endif

#include <pthread.h>
#include <unistd.h>

#endif


#ifndef _WIN32
#if USER_LOCKS && (defined(i386) || defined(sparc) || defined(__sgi) || defined(__hppa__) || defined(__hppa) || defined(ppc) || defined(__s390__))
typedef unsigned long	hoardLockType;
#else
typedef pthread_mutex_t	hoardLockType;
#endif

#if USE_SPROC
typedef pid_t			hoardThreadType;
#else
typedef pthread_t		hoardThreadType;
#endif

#endif

extern "C" {

  ///// Thread-related wrappers.

#if defined (_WIN32)
  void	hoardCreateThread (hoardThreadType& t,
			   void (*function) (void *),
			   void * arg);
#else
  void	hoardCreateThread (hoardThreadType& t,
			   void * (*function) (void *),
			   void * arg);
#endif

  /// Wait for a thread to terminate.
  void	hoardJoinThread (hoardThreadType& t);

  /// Set the desired concurrency level.
  void  hoardSetConcurrency (int n);

  /**
   * @return A thread identifier appropriate for hashing.
   * @note Some systems do not produce consecutive thread id's so some hackery may be necessary.
   */
  int	hoardGetThreadID (void);

  ///// Lock-related wrappers.

#if !defined(_WIN32) && !USER_LOCKS

  /* We define the lock operations inline to save a little overhead. */

  /// Initialize a lock.
  inline void hoardLockInit (hoardLockType& lock) {
    pthread_mutex_init (&lock, NULL);
  }

  /// Acquire a lock.
  inline void hoardLock (hoardLockType& lock) {
    pthread_mutex_lock (&lock);
  }

  /// Release a lock.
  inline void hoardUnlock (hoardLockType& lock) {
    pthread_mutex_unlock (&lock);
  }

#else
  void	hoardLockInit (hoardLockType& lock);
  void	hoardLock (hoardLockType& lock);
  void	hoardUnlock (hoardLockType& lock);
#endif

  inline void  hoardLockDestroy (hoardLockType&) {
  }

  ///// Memory-related wrapper.

  int	hoardGetPageSize (void);
  void *	hoardGetMemory (size_t size);
  void	hoardFreeMemory (void * ptr);

  ///// Other.

	/// Give up the current thread's quantum.
  void  hoardYield (void);

  /// @return The number of processors (CPUs) on a system.
  int	hoardGetNumProcessors (void);

  /// Atomically assign *oldval = newval.
  /// @return The previous value of *oldval.
  unsigned long hoardInterlockedExchange (unsigned long * oldval,
					  unsigned long newval);
}

#endif // _ARCH_SPECIFIC_H_

