/*
 * This file is part of hildon-help
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Jakub Pavelek <jakub.pavelek@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
 * SECTION:hildon-help
 * @short_description: Hildon Help Functions
 *
 * Use these functions to invoke the help UI, or to add context-sensitive help to your %GtkDialog widgets.
 */
#ifndef _HILDON_HELP_H_
#define _HILDON_HELP_H_

#include <libosso.h>
#include <gtk/gtkdialog.h>

#ifdef __cplusplus
  extern "C" {
#endif

enum {
    HILDON_HELP_SHOW_DIALOG= 0x01,
    HILDON_HELP_SHOW_JUSTASK= 0x02
    };
    
/**
 * hildon_help_show:
 * @osso: OSSO context to use for RPC
 * @topic_id: Help topic ID. This is a string of the form "foo_bar_baz"
 * @flags: Whether to actually display the help topic
 *         (%HILDON_HELP_SHOW_DIALOG), or merely determine whether the
 *         topic is available (%HILDON_HELP_SHOW_JUSTASK)
 * 
 * Show Help topic or folder.  This same function can be used either for
 * Help dialogs (displaying a single topic _without links_ in the
 * calling application's dialog box), or for launching Help Application,
 * providing browsing, search and history functions, too.
 *
 * Normally, applications would only use this to launch help topics, but
 * it can be used for opening a folder view as well.
 *
 * Returns:  %OSSO_OK        All well, help found (and being displayed)
 *           %OSSO_ERROR     No help for such topic ID
 *           %OSSO_RPC_ERROR Unable to contact HelpApp or Browser
 *           %OSSO_INVALID   Incorrect parameters (NULL, or bad ID format)
 */
osso_return_t hildon_help_show( osso_context_t *osso,
                                const gchar *topic_id,
                                guint flags );

/**
 * hildon_help_dialog_help_enable:
 * @dialog: %GtkDialog to enhance
 * @topic: Help topic ID. This is a string of the form "foo_bar_baz"
 * @osso: OSSO context to use for RPC
 * Enable context specific help for a dialog.
 *
 * Calling this function adds a '?' icon to the dialog's titlebar,
 * allowing user to access the context specific help topic. All
 * UI binding is done for you, one line is all it takes.
 *
 * TODO: If the given help topic is not installed on the system, 
 *       the '?' icon should be dimmed (insensitive); is there a 
 *       way to do that?
 *
 * Note: Do this call before gtk_widget_show_all(). Otherwise, the
 *       '?' icon won't show up.
 *
 * Returns: %TRUE if the help topic exists, and '?' has been added, 
 *          and %FALSE if the topic ID was not found on the system.
*/
gboolean hildon_help_dialog_help_enable( GtkDialog *dialog, 
                                         const gchar *topic,
                                         osso_context_t *osso );

#ifdef __cplusplus
  }
#endif

#endif  /* ndef _HILDON_HELP_H_ */

