/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: CommandHyperlinkListener.java,v 1.11 2005/06/15 23:42:57 pietschy Exp $
 */

package org.pietschy.command;

import org.pietschy.command.log.Logger;

import javax.swing.*;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import java.awt.*;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * A {@link javax.swing.event.HyperlinkListener} that will interogate the href of the
 * link and fire the appropriate command.  Commands can be specified in the
 * anchor using the following syntax
 * <pre>command://my.command.id?hintName1=hintValue,hintName2=hintVale</pre><p>
 * A simple command would be specified using<pre>command://my.command.name</pre><P>
 * The <tt>command://</tt> protocol is not registered with the standard URL infrastructure.
 * <p>
 * The {@link #buildAnchorString} method can be used to create a correctly formatted URL
 * string from a specified command.
 *
 */
public class
CommandHyperlinkListener
implements HyperlinkListener
{
   private static final Logger log = CommandManager.getLogger(CommandHyperlinkListener.class);

   private static final String COMMAND_PREFIX = "command://";

   private CommandManager commandManager;
   private HashMap popups = new HashMap();

   /**
    * Creates a new {@link HyperlinkListener} that will search for commands using the
    * specified {@link CommandManager}.
    * @param commandManager the container used to locate commands.
    */
   public
   CommandHyperlinkListener(CommandManager commandManager)
   {
      this.commandManager = commandManager;
   }

   /**
    * Creates a new {@link HyperlinkListener} that will search for commands using the
    * specified default {@link CommandManager} instance.  It is equivalent to calling
    * {@link #CommandHyperlinkListener(org.pietschy.command.CommandManager)}
    * with {@link CommandManager#defaultInstance} as the argument.
    */
   public
   CommandHyperlinkListener()
   {
      this(CommandManager.defaultInstance());
   }

   public void
   hyperlinkUpdate(HyperlinkEvent e)
   {
      String url = e.getDescription().trim();
      String commandId = getCommandIdFromUrl(url);

      Command command;
      if (commandManager.isGroup(commandId))
         command = commandManager.getGroup(commandId);
      else
         command = commandManager.getCommand(commandId);

      if (command == null)
         return;

      if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED)
      {
         if (command.isEnabled())
         {
            if (command instanceof CommandGroup)
            {
               try
               {
                  Element anchor = e.getSourceElement();
                  JEditorPane editor = ((JEditorPane) e.getSource());
                  Rectangle rect = editor.modelToView(anchor.getStartOffset());
                  getPopupFor((CommandGroup) command).show(editor, rect.x, rect.y + rect.height);
               }
               catch (BadLocationException e1)
               {
               }
            }
            else if (command instanceof ActionCommand)
            {
               Map hints = getHints(url);
               ActionCommand actionCommand = ((ActionCommand) command);
               actionCommand.putHint(ActionCommand.HINT_INVOKER, e.getSource());

               if (hints != null)
                  actionCommand.execute(hints);
               else
                  actionCommand.execute();

            }
         }
      }
      else if (e.getEventType() == HyperlinkEvent.EventType.ENTERED)
      {
         command.fireHoverStarted(command.getDefaultFace(), (Component) e.getSource());
      }
      else if (e.getEventType() == HyperlinkEvent.EventType.EXITED)
      {
         command.fireHoverEnded(command.getDefaultFace(), (Component) e.getSource());
      }
   }

   protected String getCommandIdFromUrl(String url)
   {
      int startIndex = url.startsWith(COMMAND_PREFIX) ? COMMAND_PREFIX.length() : 0;
      int endIndex = url.indexOf("?");

      return (endIndex > 0) ? url.substring(startIndex, endIndex) : url.substring(startIndex);
   }

   protected Map getHints(String url)
   {
      int startIndex = url.indexOf("?");

      if (startIndex < 0)
         return null;

      // get past the '?'
      startIndex++;

      String hintString = url.substring(startIndex);
      Map hintMap = new HashMap();
      String[] hints = hintString.split(",");
      for (int i = 0; i < hints.length; i++)
      {
         String[] entry = hints[i].split("=");
         hintMap.put(entry[0], entry[1]);
      }

      return hintMap;
   }

   protected JPopupMenu getPopupFor(CommandGroup group)
   {
      JPopupMenu menu = (JPopupMenu) popups.get(group);
      if (menu == null)
      {
         menu = group.createPopupMenu();
         popups.put(group, menu);
      }

      return menu;
   }

   /**
    *
    * @param command
    * @param hints
    * @param text
    * @return
    */
   public static String buildAnchorString(ActionCommand command, Map hints, String text, String toolTip)
   {
      StringBuffer buf = new StringBuffer("<a href='");

      buf.append(COMMAND_PREFIX);
      buf.append(command.getId());
      if (hints != null)
      {
         buf.append("?");
         Set entries = hints.entrySet();
         for (Iterator iterator = entries.iterator(); iterator.hasNext();)
         {
            Map.Entry entry = (Map.Entry) iterator.next();
            if (entry.getKey() != null && entry.getValue() != null)
               buf.append(entry.getKey()).append("=").append(entry.getValue());
         }
      }
      buf.append("'");
      if (toolTip != null)
      {
         buf.append(" title='").append(toolTip).append("'");
      }

      buf.append(">");
      buf.append(text);
      buf.append("</a>");

      return buf.toString();
   }

   public static String buildAnchorString(ActionCommand command, Map hints, String text)
   {
      return buildAnchorString(command, hints, text, null);
   }

   /**
    * Builds an &lt;a&gt; tag for the specified command but that uses the specified text instead
    * of the Commands normal text.
    * @param command the command to be activated by the link
    * @param text the text to display
    * @return a formated &lt;a&gt; tag to be inserted into html.
    */
   public static String buildAnchorString(ActionCommand command, String text)
   {
      return buildAnchorString(command, null, text, null);
   }

   /**
    * Builds an &lt;a&gt; tag for the specified command using the commands text attribute to
    * populate the links text and the short description to populate the tooltip.  The "html"
    * face is used if specified, the default face is used otherwise.
    * @param command the command to be activated by the link
    * @return a formated &lt;a&gt; tag to be inserted into html.
    */
   public static String buildAnchorString(ActionCommand command)
   {
      // try the html face first.
      Face face = command.getFace("html");
      return buildAnchorString(command, null, face.getText(), face.getDescription());
   }

   /**
    * Builds an &lt;a&gt; tag for the specified command using the commands text attribute to
    * populate the links text and the short description to populate the tooltip.  The "html"
    * face is used if specified, the default face is used otherwise.
    * @param command the command to be activated by the link
    * @param hints a map of hints to be specified to the command when it executes.
    * @return a formated &lt;a&gt; tag to be inserted into html.
    */
   public static String buildAnchorString(ActionCommand command, Map hints)
   {
      // try the html face first.
      Face face = command.getFace("html");
      return buildAnchorString(command, hints, face.getText(), face.getDescription());
   }

   public static void main(String[] args)
   {
      CommandHyperlinkListener l = new CommandHyperlinkListener();

      System.out.println(l.getCommandIdFromUrl("command://a.b.c?x=1,y=2"));
      System.out.println(l.getHints("command://a.b.c?x=1,y=2"));
      System.out.println(l.getCommandIdFromUrl("a.b.c?x=1,y=2"));
      System.out.println(l.getHints("a.b.c?x=1,y=2"));
      System.out.println(l.getHints("command://a.b.c"));
      System.out.println(l.getHints("a.b.c"));
   }

}
