/*******************************************************************************************************************************************
 cstatusbar.c
*******************************************************************************************************************************************/

#include "cstatusbar.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CStatusBar);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CStatusBar::CStatusBar  (CContainer *inOwner, const CWidgetListener *inListener)
	   :CHBoxLayout (NULL, inListener),
	    m_ContextId (0L),
	    m_Caption	()
{
	// widget instanciation handling request
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CStatusBar::~CStatusBar ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CStatusBar::PerformWidgetInstanciate ()
{
	// ok
	return gtk_statusbar_new();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CStatusBar::PerformWidgetInitialize ()
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// get an arbitrary context id
	if (inGtkWidget != NULL) m_ContextId = ::gtk_statusbar_get_context_id (GTK_STATUSBAR(inGtkWidget), "gtkol status bar");

	// generic call
	CHBoxLayout::PerformWidgetInitialize();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// has resize grip
//-----------------------------------------------------------------------------------------------------------------------------------------
void CStatusBar::SetHasResizeGrip (const Bool inHasResizeGrip)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_statusbar_set_has_resize_grip (GTK_STATUSBAR(inGtkWidget), inHasResizeGrip);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// has resize grip ?
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CStatusBar::GetHasResizeGrip () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_statusbar_get_has_resize_grip (GTK_STATUSBAR(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// set caption
//-----------------------------------------------------------------------------------------------------------------------------------------
void CStatusBar::SetCaption (const CString &inCaption)
{
	// attribute local copy
	m_Caption = inCaption;

	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return;

	// remove the message at the top of the context id
	if (m_ContextId > 0L) ::gtk_statusbar_pop (GTK_STATUSBAR(inGtkWidget), m_ContextId);

	// push the new message
	::gtk_statusbar_push (GTK_STATUSBAR(inGtkWidget), m_ContextId, m_Caption.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get caption
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CStatusBar::GetCaption () const
{
	// ok
	return m_Caption;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CStatusBar::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CHBoxLayout::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_STATUSBAR_ELEMENT);

			// add the attribute
			newXMLElement -> AddAttribute (XML_STATUSBAR_ATTR_HAS_RESIZE_GRIP, CString(GetHasResizeGrip()?"true":"false"));

			// change the io xml element node so that the overloaded serialization process would continue under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode();
		}
		break;

		// xml load
		case XML_READ :
		{
			// search for the node this instance is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_STATUSBAR_ELEMENT); 

			// check we got an expected cboxlayout node
			if (inXMLNode == NULL)
				throw new CException (CString("CStatusBar::Serialize, specified xml node is not a \"") + 
							      XML_STATUSBAR_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// retreive and set the attribute
			SetHasResizeGrip (::xml_node_get_attribute (inXMLNode, XML_STATUSBAR_ATTR_HAS_RESIZE_GRIP).GetValue().ToBool());

			// modify the io xml element node
			ioXMLElementNode = inXMLNode;
		}
		break;
	}
}

