/**
 * @file libgalago/galago-status.h Galago Status API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_STATUS_H_
#define _GALAGO_STATUS_H_

typedef struct _GalagoStatus        GalagoStatus;
typedef struct _GalagoStatusClass   GalagoStatusClass;
typedef struct _GalagoStatusPrivate GalagoStatusPrivate;
typedef struct _GalagoStatusAttr    GalagoStatusAttr;

typedef enum
{
	GALAGO_STATUS_UNSET = 0,
	GALAGO_STATUS_OFFLINE,
	GALAGO_STATUS_AVAILABLE,
	GALAGO_STATUS_AWAY,
	GALAGO_STATUS_EXTENDED_AWAY,
	GALAGO_STATUS_HIDDEN

} GalagoStatusType;

typedef enum
{
	GALAGO_STATUS_ATTR_UNSET = -1,
	GALAGO_STATUS_ATTR_BOOL  = 0,
	GALAGO_STATUS_ATTR_STRING,
	GALAGO_STATUS_ATTR_INT,
	GALAGO_STATUS_ATTR_DOUBLE

} GalagoStatusAttrType;

#include <libgalago/galago-presence.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-types.h>

/**************************************************************************/
/** @name Common Status Definitions                                       */
/**************************************************************************/
/*@{*/

#define GALAGO_STATUS_ID_AVAILABLE       "available"
#define GALAGO_STATUS_ID_AWAY            "away"
#define GALAGO_STATUS_ID_BRB             "brb"
#define GALAGO_STATUS_ID_BUSY            "busy"
#define GALAGO_STATUS_ID_DND             "dnd"
#define GALAGO_STATUS_ID_EXTENDED_AWAY   "xa"
#define GALAGO_STATUS_ID_HIDDEN          "hidden"
#define GALAGO_STATUS_ID_OFFLINE         "offline"

/*@}*/

/**************************************************************************/
/** @name Status API                                                      */
/**************************************************************************/
/*@{*/

/**
 * A status in a presence.
 */
struct _GalagoStatus
{
	GalagoObject parent_object;

	GalagoStatusPrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * GalagoStatus's class.
 */
struct _GalagoStatusClass
{
	GalagoObjectClass parent_class;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

#define GALAGO_CLASS_STATUS (galago_status_get_class())
#define GALAGO_IS_STATUS(obj) (GALAGO_IS_OBJECT(obj) && galago_object_check_cast((obj), GALAGO_CLASS_STATUS))

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Status API                                                      */
/**************************************************************************/
/*@{*/

/**
 * Returns the class for a GalagoStatus.
 *
 * @return The GalagoStatus class.
 */
GalagoObjectClass *galago_status_get_class(void);

/**
 * Creates a new status.
 *
 * If @a exclusive is TRUE, the status will be exclusive. Only one
 * exclusive status can be set at a time. If another exclusive status
 * is set, the previously set exclusive status will be removed.
 *
 * If @a exclusive is FALSE, the status will not be removed unless
 * manually removed.
 *
 * @param type      The type of status.
 * @param id        The status ID.
 * @param name      The name of the status.
 * @param exclusive TRUE if the status is exclusive.
 *
 * @return The status.
 */
GalagoStatus *galago_status_new(GalagoStatusType type, const char *id,
								const char *name, galago_bool exclusive);

/**
 * Duplicates a status.
 *
 * @param status The status to duplicate.
 *
 * @return The duplicate status.
 */
GalagoStatus *galago_status_duplicate(const GalagoStatus *status);

/**
 * Sets the parent presence of a status.
 *
 * This should only be used internally.
 *
 * @param status   The status.
 * @param presence The presence.
 */
void galago_status_set_presence(GalagoStatus *status,
								GalagoPresence *presence);

/**
 * Returns the parent presence of a status.
 *
 * @param status The status.
 *
 * @return The presence.
 */
GalagoPresence *galago_status_get_presence(const GalagoStatus *status);

/**
 * Returns the primitive type of a status.
 *
 * @param status The status.
 *
 * @return The status's primitive type.
 */
GalagoStatusType galago_status_get_type(const GalagoStatus *status);

/**
 * Returns the status's ID.
 *
 * @param status The status.
 *
 * @return The status's ID.
 */
const char *galago_status_get_id(const GalagoStatus *status);

/**
 * Returns the status's name.
 *
 * @param status The status.
 *
 * @return The status's name.
 */
const char *galago_status_get_name(const GalagoStatus *status);

/**
 * Returns whether or not a status is exclusive.
 *
 * @param status The status.
 *
 * @return TRUE if the status is exclusive, or FALSE.
 */
galago_bool galago_status_is_exclusive(const GalagoStatus *status);

/**
 * Returns whether or not a status is considered available.
 *
 * @param status The status.
 *
 * @return TRUE if the status is available, or FALSE.
 */
galago_bool galago_status_is_available(const GalagoStatus *status);

/**
 * Sets a boolean status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 * @param value   The boolean value.
 */
void galago_status_set_attr_bool(GalagoStatus *status, const char *attr_id,
								 galago_bool value);

/**
 * Sets a string status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 * @param value   The string value.
 */
void galago_status_set_attr_string(GalagoStatus *status, const char *attr_id,
								   const char *value);

/**
 * Sets an integer status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 * @param value   The integer value.
 */
void galago_status_set_attr_int(GalagoStatus *status, const char *attr_id,
								dbus_int32_t value);

/**
 * Sets a double status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 * @param value   The double value.
 */
void galago_status_set_attr_double(GalagoStatus *status, const char *attr_id,
								   double value);

/**
 * Removes a status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 */
void galago_status_remove_attr(GalagoStatus *status, const char *attr_id);

/**
 * Returns a boolean status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 *
 * @return The boolean value.
 */
galago_bool galago_status_get_attr_bool(const GalagoStatus *status,
										const char *attr_id);

/**
 * Returns a string status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 *
 * @return The string value.
 */
const char *galago_status_get_attr_string(const GalagoStatus *status,
										  const char *attr_id);

/**
 * Returns an integer status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 *
 * @return The integer value.
 */
dbus_int32_t galago_status_get_attr_int(const GalagoStatus *status,
										const char *attr_id);

/**
 * Returns a double status attribute.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 *
 * @return The double value.
 */
double galago_status_get_attr_double(const GalagoStatus *status,
									 const char *attr_id);

/**
 * Returns whether or not a status attribute exists.
 *
 * @param status  The status.
 * @param attr_id The attribute ID.
 *
 * @return TRUE if the status attribute exists, or FALSE otherwise.
 */
galago_bool galago_status_attr_exists(const GalagoStatus *status,
									  const char *attr_id);

/**
 * Returns the list of status attributes in this status.
 *
 * @param status The status.
 *
 * @return The status attributes in the status.
 */
const GalagoList *galago_status_get_attributes(const GalagoStatus *status);

/*@}*/

/**************************************************************************/
/** @name GalagoStatusAttr API                                            */
/**************************************************************************/
/*@{*/

/**
 * Returns the type of a status attribute.
 *
 * @param attr The status attribute.
 *
 * @return The status attribute's type.
 */
GalagoStatusAttrType galago_status_attr_get_type(const GalagoStatusAttr *attr);

/**
 * Returns the ID of a status attribute.
 *
 * @param attr The status attribute.
 *
 * @return The status attribute's ID.
 */
const char *galago_status_attr_get_id(const GalagoStatusAttr *attr);

/**
 * Returns the value of a boolean status attribute.
 *
 * @param attr The status attribute.
 *
 * @return The boolean value.
 */
galago_bool galago_status_attr_get_bool(const GalagoStatusAttr *attr);

/**
 * Returns the value of a string status attribute.
 *
 * @param attr The status attribute.
 *
 * @return The string value.
 */
const char *galago_status_attr_get_string(const GalagoStatusAttr *attr);

/**
 * Returns the value of an integer status attribute.
 *
 * @param attr The status attribute.
 *
 * @return The integer value.
 */
dbus_int32_t galago_status_attr_get_int(const GalagoStatusAttr *attr);

/**
 * Returns the value of a double status attribute.
 *
 * @param attr The status attribute.
 *
 * @return The double value.
 */
double galago_status_attr_get_double(const GalagoStatusAttr *attr);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_STATUS_H_ */
