/**
 * @file libgalago/galago-service.h Galago Service API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_SERVICE_H_
#define _GALAGO_SERVICE_H_

typedef struct _GalagoService        GalagoService;
typedef struct _GalagoServiceClass   GalagoServiceClass;
typedef struct _GalagoServicePrivate GalagoServicePrivate;

/*
 * Service flags
 */
typedef enum
{
	GALAGO_PRESERVE_SPACES = 0x01, /**< Preserve spaces during normalization */
	GALAGO_PRESERVE_CASE   = 0x02, /**< Preserve case during normalization   */
	GALAGO_STRIP_SLASH     = 0x04  /**< Strip a slash and everything after it
	                                    during normalization.                */
} GalagoServiceFlags;

#include <libgalago/galago-account.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-signals.h>

/**
 * A service capable of having accounts and presences.
 */
struct _GalagoService
{
	GalagoObject parent_object;

	GalagoServicePrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * A service's class.
 */
struct _GalagoServiceClass
{
	GalagoObjectClass parent_class;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

#define GALAGO_CLASS_SERVICE (galago_service_get_class())
#define GALAGO_DBUS_SERVICE_INTERFACE "org.freedesktop.Galago.Service"
#define GALAGO_IS_SERVICE(obj) (GALAGO_IS_OBJECT(obj) && galago_object_check_cast((obj), GALAGO_CLASS_SERVICE))

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Common Service Definitions                                      */
/**************************************************************************/
/*@{*/

#define GALAGO_SERVICE_ID_AIM          "aim"
#define GALAGO_SERVICE_ID_GADUGADU     "gadugadu"
#define GALAGO_SERVICE_ID_GROUPWISE    "groupwise"
#define GALAGO_SERVICE_ID_ICQ          "icq"
#define GALAGO_SERVICE_ID_IRC          "irc"
#define GALAGO_SERVICE_ID_JABBER       "jabber"
#define GALAGO_SERVICE_ID_MSN          "msn"
#define GALAGO_SERVICE_ID_NAPSTER      "napster"
#define GALAGO_SERVICE_ID_SILC         "silc"
#define GALAGO_SERVICE_ID_TREPIA       "trepia"
#define GALAGO_SERVICE_ID_YAHOO        "yahoo"
#define GALAGO_SERVICE_ID_ZEPHYR       "zephyr"

/*@}*/

/**************************************************************************/
/** @name Service API                                                     */
/**************************************************************************/
/*@{*/

/**
 * Returns the class for a GalagoService.
 *
 * @return The GalagoService class.
 */
GalagoObjectClass *galago_service_get_class(void);

/**
 * Creates a service.
 *
 * If @a native is TRUE, this service, and all accounts beneath it, will
 * be native. Native services and accounts are fed through the local
 * feed(s). Non-native services are a result of queries or watches.
 *
 * @param id     The service ID.
 * @param name   The service name.
 * @param native TRUE if this service belongs to the local feed.
 * @param flags  Service flags.
 *
 * @return The service.
 */
GalagoService *galago_service_new(const char *id, const char *name,
								  galago_bool native,
								  GalagoServiceFlags flags);

/**
 * Returns a service's ID.
 *
 * @param service The service.
 *
 * @return The service's ID.
 */
const char *galago_service_get_id(const GalagoService *service);

/**
 * Returns a service's name.
 *
 * @param service The service.
 *
 * @return The service's name.
 */
const char *galago_service_get_name(const GalagoService *service);

/**
 * Returns whether or not a service is native to the local application.
 *
 * @param service The service.
 *
 * @return TRUE if the service is native to the local application, or
 *         FALSE if it came from another feed.
 */
galago_bool galago_service_is_native(const GalagoService *service);

/**
 * Returns a service's flags.
 *
 * @param service The service.
 *
 * @return The flags.
 */
GalagoServiceFlags galago_service_get_flags(const GalagoService *service);

/**
 * Returns the account with the specified username from a service.
 *
 * @param service  The service.
 * @param username The account's username.
 * @param query    TRUE if a remote query should be done if there is no
 *                 local account found, or FALSE.
 *
 * @return The account, if found, or NULL.
 */
GalagoAccount *galago_service_get_account(const GalagoService *service,
										  const char *username,
										  galago_bool query);

/**
 * Returns a list of all accounts in the service.
 *
 * @param service The service.
 * @param query   TRUE if a remote query should be done if there is no
 *                local account found, or FALSE.
 *
 * @return A list of all accounts in the service.
 */
const GalagoList *galago_service_get_accounts(const GalagoService *service,
											  galago_bool query);

/**
 * Adds an account to a service.
 *
 * This is used internally. You don't need to call this.
 *
 * @param service The service.
 * @param account The account to add.
 */
void galago_service_add_account(GalagoService *service,
								GalagoAccount *account);

/**
 * Removes an account from a service.
 *
 * This is used internally. You don't need to call this.
 *
 * @param service The service.
 * @param account The account to remove.
 */
void galago_service_remove_account(GalagoService *service,
								   GalagoAccount *account);

/**
 * Normalizes an account username based on the service's normalization
 * flags.
 *
 * The returned string is stored internally in a status buffer. If you need
 * the string for long, make sure to strdup() it.
 *
 * @param service  The service.
 * @param username The username to normalize.
 *
 * @return The string, stored in a static buffer.
 */
const char *galago_service_normalize(const GalagoService *service,
									 const char *username);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_SERVICE_H_ */
