/**
 * @file libgalago/galago-log.h Log API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBGALAGO_LOG_H_
#define _LIBGALAGO_LOG_H_

#include <stdarg.h>
#include <libgalago/galago-types.h>

/**
 * Log levels.
 */
typedef enum
{
	GALAGO_LOG_ALL = 0,    /**< All log levels.                */
	GALAGO_LOG_MISC,       /**< General chatter.               */
	GALAGO_LOG_INFO,       /**< General operation information. */
	GALAGO_LOG_WARNING,    /**< Warnings.                      */
	GALAGO_LOG_ERROR,      /**< Errors.                        */
	GALAGO_LOG_FATAL,      /**< Fatal errors.                  */
	GALAGO_LOG_ASSERT,     /**< Assertion errors.              */
	GALAGO_LOG_NUM_LEVELS  /**< The number of log levels.      */

} GalagoLogLevel;

typedef void (*GalagoLogHandlerFunc)(GalagoLogLevel, const char *,
									 va_list args, void *);

#ifndef __GNUC__
# define __attribute__(x)
#endif

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Sets whether or not log output is enabled for a level.
 *
 * If @a level is GALAGO_LOG_ALL, the enabled state will be set for
 * all levels. The other levels may then be enabled or disabled on
 * a per-level basis.
 *
 * @param level   The log level.
 * @param enabled Whether or not log output is enabled.
 */
void galago_log_set_enabled(GalagoLogLevel level, galago_bool enabled);

/**
 * Sets up a handler for logging information.
 *
 * @param level     The log level.
 * @param func      The handler function.
 * @param user_data Optional user-specific data.
 */
void galago_log_set_handler(GalagoLogLevel level, GalagoLogHandlerFunc func,
							void *user_data);

/**
 * Unsets a logging handler.
 *
 * This is equivalent to calling galago_log_set_handler(level, func, NULL);
 *
 * @param level The log level.
 */
void galago_log_unset_handler(GalagoLogLevel level);

/**
 * Outputs log information.
 *
 * @param level  The log level.
 * @param format The format string.
 * @param ...    The format parameters.
 */
void galago_log(GalagoLogLevel level, const char *format, ...)
	__attribute__((format(printf, 2, 3)));

/**
 * Outpus log information.
 *
 * This is different from galagoLog() in that it takes a va_list.
 *
 * @param level  The log level.
 * @param format The format string.
 * @param args   The format parameters.
 *
 * @see galago_log()
 */
void galago_log_args(GalagoLogLevel level, const char *format, va_list args);

/**
 * Outputs misc. level log information.
 *
 * This is a wrapper for galago_log(), and uses GALAGO_LOG_MISC as
 * the level.
 *
 * @param format The format string.
 * @param ...    The format parameters.
 */
void galago_log_misc(const char *format, ...)
	__attribute__((format(printf, 1, 2)));

/**
 * Outputs info level log information.
 *
 * This is a wrapper for galago_log(), and uses GALAGO_LOG_INFO as
 * the level.
 *
 * @param format The format string.
 * @param ...    The format parameters.
 */
void galago_log_info(const char *format, ...)
	__attribute__((format(printf, 1, 2)));

/**
 * Outputs warning level log information.
 *
 * This is a wrapper for galago_log(), and uses GALAGO_LOG_WARNING as
 * the level.
 *
 * @param format The format string.
 * @param ...    The format parameters.
 */
void galago_log_warning(const char *format, ...)
	__attribute__((format(printf, 1, 2)));

/**
 * Outputs error level log information.
 *
 * This is a wrapper for galago_log(), and uses GALAGO_LOG_ERROR as
 * the level.
 *
 * @param format The format string.
 * @param ...    The format parameters.
 */
void galago_log_error(const char *format, ...)
	__attribute__((format(printf, 1, 2)));

/**
 * Outputs fatal level log information.
 *
 * This is a wrapper for galago_log(), and uses GALAGO_LOG_FATAL as
 * the level.
 *
 * @param format The format string.
 * @param ...    The format parameters.
 */
void galago_log_fatal(const char *format, ...)
	__attribute__((format(printf, 1, 2)));

#ifdef __cplusplus
}
#endif

#endif /* _LIBGALAGO_LOG_H_ */
