/**
 * @file libgalago/galago-list.c List API
 *
 * @Copyright (C) 2004-2005 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-list.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-utils.h>
#include <stdlib.h>

GalagoList *
galago_list_new(void)
{
	GalagoList *list;

	list = galago_new0(GalagoList, 1);

	return list;
}

void
galago_list_destroy(GalagoList *list)
{
	GalagoList *last;

	while (list != NULL)
	{
		last = list;
		list = list->next;
		free(last);
	}
}

void
galago_list_destroy_1(GalagoList *list)
{
	if (list == NULL)
		return;

	free(list);
}

GalagoList *
galago_list_dup(const GalagoList *list)
{
	const GalagoList *l;
	GalagoList *new_list = NULL;

	if (list == NULL)
		return NULL;

	for (l = list; l != NULL; l = l->next)
		new_list = galago_list_append(new_list, l->data);

	return new_list;
}

GalagoList *
galago_list_append(GalagoList *list, void *data)
{
	GalagoList *list_item;
	GalagoList *last_item;

	list_item = galago_list_new();
	list_item->data = data;

	if (list == NULL)
		return list_item;

	last_item = galago_list_get_last(list);
	list_item->prev = last_item;
	last_item->next = list_item;

	return list;
}

GalagoList *
galago_list_prepend(GalagoList *list, void *data)
{
	GalagoList *list_item;

	list_item = galago_list_new();
	list_item->data = data;

	if (list == NULL)
		return list_item;

	if (list->prev != NULL)
	{
		list->prev->next = list_item;
		list_item->prev  = list->prev;
	}
	else
	{
		list->prev = list_item;
		list_item->next = list;
	}

	return list_item;
}

GalagoList *
galago_list_remove(GalagoList *list, void *data)
{
	GalagoList *l, *l_next;

	for (l = list; l != NULL; l = l_next)
	{
		l_next = l->next;

		if (l->data == data)
		{
			list = galago_list_delete_link(list, l);

			break;
		}
	}

	return list;
}

GalagoList *
galago_list_remove_link(GalagoList *list, GalagoList *link)
{
	if (link == NULL)
		return list;

	if (link->prev != NULL)
		link->prev->next = link->next;

	if (link->next != NULL)
		link->next->prev = link->prev;

	if (link == list)
		list = link->next;

	link->next = NULL;
	link->prev = NULL;

	return list;
}

GalagoList *
galago_list_delete_link(GalagoList *list, GalagoList *link)
{
	if (list == NULL || link == NULL)
		return list;

	list = galago_list_remove_link(list, link);
	galago_list_destroy_1(link);

	return list;
}

GalagoList *
galago_list_find(GalagoList *list, void *data)
{
	GalagoList *l;

	if (list == NULL || data == NULL)
		return NULL;

	for (l = list; l != NULL; l = l->next)
	{
		if (l->data == data)
			return l;
	}

	return NULL;
}

void
galago_list_foreach(const GalagoList *list, GalagoForEachFunc func,
					void *user_data)
{
	const GalagoList *l, *l_next;

	if (list == NULL)
		return;

	galago_return_if_fail(func != NULL);

	for (l = list; l != NULL; l = l_next)
	{
		l_next = l->next;

		func(l->data, user_data);
	}
}

GalagoList *
galago_list_get_first(GalagoList *list)
{
	GalagoList *first_item;

	for (first_item = list;
		 first_item != NULL && first_item->prev != NULL;
		 first_item = first_item->prev)
		;

	return first_item;
}

GalagoList *
galago_list_get_last(GalagoList *list)
{
	GalagoList *last_item;

	for (last_item = list;
		 last_item != NULL && last_item->next != NULL;
		 last_item = last_item->next)
		;

	return last_item;
}

GalagoList *
galago_list_get_prev(GalagoList *list)
{
	if (list == NULL)
		return NULL;

	return list->prev;
}

GalagoList *
galago_list_get_next(GalagoList *list)
{
	if (list == NULL)
		return NULL;

	return list->next;
}

void *
galago_list_get_data(GalagoList *list)
{
	if (list == NULL)
		return NULL;

	return list->data;
}

size_t
galago_list_get_count(const GalagoList *list)
{
	const GalagoList *l;
	size_t i = 0;

	for (l = list; l != NULL; l = l->next)
		i++;

	return i;
}
