/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"
#include <stdlib.h>
#include <string.h>


void tt_reparent_siblings(TT *parent_tt, TT *tt)
{
  TT *old_parent_tt, *next_tt;
  
  old_parent_tt = tt_get_parent(tt);
  if (!old_parent_tt) return;

  for (tt = tt_get_first_child(old_parent_tt); tt; tt = next_tt)
  {
    next_tt = tt_get_next(tt);
    
    tt_detach(tt);
    tt_add_as_last_child(parent_tt, tt);
  }

  tt_del(old_parent_tt);
}


TT *tt_get_root(TT *tt)
{
  TT *tt_last;
  
  for (tt_last = tt; tt; tt = tt_get_parent(tt)) tt_last = tt;

  return(tt_last);
}


TT *tt_get_first_sibling(TT *tt)
{
  TT *tt_ret;
  
  tt_ret = tt_get_parent(tt);
  if (tt_ret) return(tt_get_first_child(tt_ret));
  
  return(tt);
}


TT *tt_get_last_sibling(TT *tt)
{
  TT *tt_ret;

  tt_ret = tt_get_parent(tt);
  if (tt_ret) return(tt_get_last_child(tt_ret));
  
  return(tt);
}


TT *tt_get_common_parent(TT *tt0, TT *tt1)
{
  int depth_0, depth_1;
  
  depth_0 = tt_depth(tt0);
  depth_1 = tt_depth(tt1);
  
  while (depth_0 > depth_1)
  {
    tt0 = tt_get_parent(tt0);
    depth_0--;
  }

  while (depth_1 > depth_0)
  {
    tt1 = tt_get_parent(tt1);
    depth_1--;
  }
  
  /* Depths are now equal */
  
  while (tt0 != tt1)
  {
    tt0 = tt_get_parent(tt0);
    tt1 = tt_get_parent(tt1);
  }

  return(tt0);  /* If NULL, nodes belong to different trees */
}


void tt_add_as_first_child(TT *parent_tt, TT *tt)
{
  TT *first_tt;
  
  if (tt_get_parent(tt)) tt_reparent_siblings(parent_tt, tt);
  else
  {
    first_tt = tt_get_first_child(parent_tt);

    parent_tt->child_first = tt;
    tt->prev = 0;
    tt->next = first_tt;
    tt->parent = parent_tt;

    if (first_tt) first_tt->prev = tt;
    else parent_tt->child_last = tt;
  }
}


void tt_add_as_last_child(TT *parent_tt, TT *tt)
{
  TT *last_tt;

  if (tt_get_parent(tt)) tt_reparent_siblings(parent_tt, tt);
  else
  {
    last_tt = tt_get_last_child(parent_tt);

    parent_tt->child_last = tt;
    tt->prev = last_tt;
    tt->next = 0;
    tt->parent = parent_tt;

    if (last_tt) last_tt->next = tt;
    else parent_tt->child_first = tt;
  }
}


void tt_add_as_first_sibling(TT *sibling_tt, TT *tt)
{
  TT *parent_tt;

  parent_tt = tt_get_parent(sibling_tt);

  if (!parent_tt)
  {
    /* Need a new root */
    parent_tt = tt_new();
    tt_set_fake_root(parent_tt, 1);
    tt_add_as_first_child(parent_tt, sibling_tt);
  }

  tt_add_as_first_child(parent_tt, tt);
}


void tt_add_as_last_sibling(TT *sibling_tt, TT *tt)
{
  TT *parent_tt;
  
  parent_tt = tt_get_parent(sibling_tt);
  
  if (!parent_tt)
  {
    /* Need a new root */
    parent_tt = tt_new();
    tt_set_fake_root(parent_tt, 1);
    tt_add_as_first_child(parent_tt, sibling_tt);
  }

  tt_add_as_last_child(parent_tt, tt);
}


void tt_add_before(TT *next_tt, TT *tt)
{
  TT *parent_tt;

  parent_tt = tt_get_parent(next_tt);

  if (!parent_tt)
  {
    /* Need a new root */
    parent_tt = tt_new();
    tt_set_fake_root(parent_tt, 1);
    tt_add_as_first_child(parent_tt, tt);
    tt_add_as_last_child(parent_tt, next_tt);
  }
  else
  {
    if (!next_tt->prev) parent_tt->child_first = tt;
    else next_tt->prev->next = tt;
    tt->next = next_tt;
    tt->prev = next_tt->prev;
    tt->parent = parent_tt;
    next_tt->prev = tt;
  }
}


void tt_add_after(TT *prev_tt, TT *tt)
{
  TT *parent_tt;
  
  parent_tt = tt_get_parent(prev_tt);
  
  if (!parent_tt)
  {
    /* Need a new root */
    parent_tt = tt_new();
    tt_set_fake_root(parent_tt, 1);
    tt_add_as_first_child(parent_tt, prev_tt);
    tt_add_as_last_child(parent_tt, tt);
  }
  else
  {
    if (!prev_tt->next) parent_tt->child_last = tt;
    else prev_tt->next->prev = tt;
    tt->prev = prev_tt;
    tt->next = prev_tt->next;
    tt->parent = parent_tt;
    prev_tt->next = tt;
  }
}


/* TODO: Check if this works when nodes already reference each other */

void tt_swap(TT *tt0, TT *tt1)
{
  TT tt_buf;
  
  memcpy(&tt_buf, tt1, sizeof(tt_buf));
  memcpy(tt1, tt0, sizeof(*tt1));
  memcpy(tt0, &tt_buf, sizeof(*tt0));
  
  if (tt1->prev) tt1->prev->next = tt1;
  else if (tt1->parent) tt1->parent->child_first = tt1;
  
  if (tt1->next) tt1->next->prev = tt1;
  else if (tt1->parent) tt1->parent->child_last = tt1;  /* FIXME: Optimize */
  
  /* - */
  
  if (tt0->prev) tt0->prev->next = tt0;
  else if (tt0->parent) tt0->parent->child_first = tt0;
  
  if (tt0->next) tt0->next->prev = tt0;
  else if (tt0->parent) tt0->parent->child_last = tt0;  /* FIXME: Optimize */
}


void tt_detach(TT *tt)
{
  if (tt->prev) tt->prev->next = tt->next;
  else if (tt->parent) tt->parent->child_first = tt->next;
  
  if (tt->next) tt->next->prev = tt->prev;
  else if (tt->parent) tt->parent->child_last = tt->prev;
  
  tt->prev = tt->next = tt->parent = 0;
}


int tt_is_in_path(TT *tt0, TT *tt1)
{
  for (; tt1 && tt0 != tt1; tt1 = tt_get_parent(tt1)) ;
  
  return(tt1 ? 1 : 0);
}


u32 tt_depth(TT *tt)
{
  int depth;
  
  for (depth = 0; tt; tt = tt_get_parent(tt), depth++) ;
  
  return(depth - 1);
}


TT *tt_get_next_in_breadth_with_level(TT *tt, int depth, int level)
{
  TT *tt0;

  tt0 = tt;

  /* If we're too shallow, try to go down first */

  while (level < depth && tt_get_first_child(tt0))
  {
    tt0 = tt_get_first_child(tt0);
    level++;
    if (level == depth) return(tt0);
  }

  for (;;)
  {
    if (level == depth)
    {
      if (tt_get_next(tt0))
      {
        /* Proceed to next node on this branch/right level */

        tt0 = tt_get_next(tt0);
        return(tt0);
      }
      else
      {
        /* Take us out of the local minimum */
        
        while (tt0 && !tt_get_next(tt0))
        {
          tt0 = tt_get_parent(tt0);
          level--;
        }
        
        if (!tt0) break;
        tt0 = tt_get_next(tt0);
      }
    }
    else if (level < depth && tt_get_first_child(tt0))
    {
      /* Try to go deeper if we have to */
      
      while (tt_get_first_child(tt0))
      {
        tt0 = tt_get_first_child(tt0);
        level++;
        if (level == depth) return(tt0);
      }
    }
    else if (level < depth && tt_get_next(tt0))
      tt0 = tt_get_next(tt0);
    else
    {
      /* Back up, this branch doesn't go deep enough 
       * (or we're too deep) */

      while (!tt_get_next(tt0))
      {
        tt0 = tt_get_parent(tt0);
        if (!tt0) break;  /* We hit root, thus no match */
        level--;
      }
      
      if (!tt0) break;  /* Somewhat ugly, but gets the job done */
      tt0 = tt_get_next(tt0);
    }
  }

  return(0);  /* No more at this depth */
}


TT *tt_get_next_in_same_depth(TT *tt)
{
  int level;

  level = tt_depth(tt);
  return(tt_get_next_in_breadth_with_level(tt, level, level));
}


TT *tt_get_next_infix(TT *tt, TT *top)
{
  if (tt_get_first_child(tt)) return(tt_get_first_child(tt));
  if (tt_get_next(tt)) return(tt_get_next(tt));

  while (tt_get_parent(tt) && !tt_get_next(tt_get_parent(tt)))
  {
    tt = tt_get_parent(tt);
    if (tt == top) return(0);
  }

  tt = tt_get_parent(tt);
  if (!tt) return(0);
  if (tt == top) return(0);

  return(tt_get_next(tt));
}
