/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

/* -----------------------
   Includes for token nets
   ----------------------- */

#ifndef _TN_H
#define _TN_H 1

#ifndef _STDIO_H
#include <stdio.h>
#endif

extern unsigned int edges_step_max[], edges_step_min[];

/* --- Token net node --- */

/* Flags */

#define TN_DOALL_FLIP (1 << 0)
#define TN_SEEN       (1 << 1)  /* Seen from another node, but not processed */
#define TN_TRAVERSED  (1 << 2)  /* Started processing subnodes */
#define TN_NOPACK     (1 << 3)  /* Don't pack edges when neighbors are deleted */
#define TN_ID         (1 << 3)  /* For separating traversed nodes in two groups */

/* Node struct */

typedef struct tn_node
{
  struct tn_node **edge_in, **edge_out;
  unsigned long edges_in_num, edges_out_num;
  unsigned char edges_in_step, edges_out_step;

  unsigned short dist;
  unsigned long flags;
  unsigned long size;

  void *data;
} TN;

/* --- Inline functions --- */

/* Flags and cocitos */

#define tn_doall(n)         ((n)->flags & TN_DOALL_FLIP)
#define tn_seen(n)          ((n)->flags & TN_SEEN)
#define tn_seen_dist(n, d)  ((((n)->flags & TN_SEEN) && ((n)->dist == d)) ? 1 : 0)
#define tn_seen_set(n, d)   do { (n)->flags |= TN_SEEN; (n)->dist = d; } while (0)
#define tn_seen_clear(n)    do { (n)->flags &= ~TN_SEEN; (n)->dist = 0; } while (0)
#define tn_trav(n)          ((n)->flags & TN_TRAVERSED)
#define tn_trav_set(n)      (n)->flags |= TN_TRAVERSED
#define tn_trav_clear(n)    (n)->flags &= ~TN_TRAVERSED
#define tn_nopk(n)          ((n)->flags & TN_NOPACK)
#define tn_nopk_set(n)      (n)->flags |= TN_NOPACK
#define tn_nopk_clear(n)    (n)->flags &= ~TN_NOPACK
#define tn_id(n)            ((n)->flags & TN_ID)

/* Reallocating edge arrays */

#define tn_edges_pack(n) do { tn_edges_in_pack(n); tn_edges_out_pack(n); } while (0)

#define _tn_edges_step_set_in(n, step) do \
{ \
  (n)->edges_in_step = step; \
  (n)->edge_in = realloc((n)->edge_in, edges_step_max[step] * sizeof(TN *)); \
} while (0)

#define _tn_edges_step_set_out(n, step) do \
{ \
  (n)->edges_out_step = step; \
  (n)->edge_out = realloc((n)->edge_out, edges_step_max[step] * sizeof(TN *)); \
} while (0)


/* --- Functions --- */

/* Allocates a new node structure, initialized with data of size <size>, or
 * nothing if either data or size is 0. */

TN *tn_add(unsigned char *data, unsigned long size);

/* Always frees a node structure, disassociating it first if necessary. */

void tn_del(TN *n);

/* Frees a node structure. Returns 1 if successful, or 0 if there were
 * still edges pointing to/from it. */

int tn_del_try(TN *n);

/* Frees a node structure, paying no heed to net integrity. */

void tn_del_fast(TN *n);

/* Deletes all nodes (like tn_del) separated from source node by at least
 * dist_min edges and at most dist_max edges (see description of
 * tn_distance). Returns number of nodes freed.
 *
 * Useful for punching holes in a network, separating out a sub-network or
 * limiting the extent of a network (removing all distant nodes). */

int tn_del_distance(TN *n, unsigned short dist_min, unsigned short dist_max, int dir);

/* Removes all incoming and outgoing edges from node, disassociating it
 * entirely from a network. Same as tn_disconnect_distance(n, 0, 0, 0). */

int tn_disassociate(TN *n);

/* Removes edges separated from source node by at least dist_min nodes and
 * at most dist_max nodes. The dir argument works like in tn_connect and
 * tn_disconnect. Returns number of edges removed. */

int tn_disconnect_distance(TN *n, unsigned short dist_min,
                               unsigned short dist_max, int dir);

/* Tries to connect two nodes. A dir of 0 means connect both ways, 1 means
 * a -> b, -1 means a <- b. */

int tn_connect(TN *a, TN *b, int dir);

/* Tries to disconnect two nodes. A dir of 0 means disconnect both ways,
 * 1 means a -/-> b, -1 means a <-/- b. */

int tn_disconnect(TN *a, TN *b, int dir);

/* Checks if two nodes are separated by at least dist_min and at most
 * dist_max edges. A distance of 0 means same node. Returns the actual
 * shortest distance between the nodes, -1 if no connection was found,
 * or -2 if there was a connection below the dist_min threshold. */

int tn_distance(TN *a, TN *b, unsigned short dist_min, unsigned short dist_max);

/* --- Internal functions --- */

int tn_edge_in_del(TN *a, TN *b);
int tn_edge_out_del(TN *a, TN *b);
void tn_edges_in_pack(TN *n);
void tn_edges_out_pack(TN *n);
int _tn_del_distance_r(TN *n, unsigned short dist_min, unsigned short dist_max, int dir, unsigned short dist);
                         
#endif  /* _TN_H */
