/*
 * libexplain - Explain errno values returned by libc functions
 * Copyright (C) 2009 Peter Miller
 * Written by Peter Miller <pmiller@opensource.org.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef CODEGEN_STRING_LIST_H
#define CODEGEN_STRING_LIST_H

#include <libexplain/ac/stddef.h>

typedef struct string_list_t string_list_t;
struct string_list_t
{
    size_t          length;
    size_t          maximum;
    char            **string;
};


/**
  * The split function may be used to break a string to to several
  * string in a string_list_t.  The parts are separated by white space.
  *
  * @param result
  *    Where to put the strings.
  * @param text
  *    The text to be split into words.
  */
void split(string_list_t *result, const char *text);

/**
  * The string_list_constructor function is used to prepare a string
  * list for use.
  *
  * @param slp
  *     The string list to operate on.
  */
void string_list_constructor(string_list_t *slp);

/**
  * The string_list_destruct function is used to release the resources
  * held by a string list, when you are done with it.
  *
  * @param slp
  *     The string list to operate on.
  */
void string_list_destructor(string_list_t *slp);

/**
  * The string_list_append function is used to add another string to the
  * end of the list.  The string is *always* copied.
  *
  * @param slp
  *     The string list to operate on.
  * @param text
  *     The string to be appended.
  */
void string_list_append(string_list_t *slp, const char *text);

/**
  * The string_list_append_n function is used to add another string to
  * the end of the list.  The string is *always* copied.
  *
  * @param slp
  *     The string list to operate on.
  * @param text
  *     The string to be appended.
  * @param tsize
  *     The length of the string to be appended.
  */
void string_list_append_n(string_list_t *slp, const char *text, size_t tsize);

/**
  * The string_list_sort function is used to sort a string list into
  * ascending lexicographical order.
  *
  * @param slp
  *     The string list to operate on.
  */
void string_list_sort(string_list_t *slp);

#endif /* CODEGEN_STRING_LIST_H */
