#include <config.h>
#include <libxml/parser.h>
#include <string.h>

#include "e-cookies.h"
#include "e-date-rfc.h"

typedef struct _ECookie ECookie;
typedef struct _ECookieData ECookieData;

struct _ECookies {
  GObject parent_object;

  GList * session;
  GList * persist;
};

struct _ECookie {
  gchar * domain;
  gchar * path;
  gchar * name;
  gchar * value;
  gchar * expires;
  time_t expires_t;
  gboolean secure;
};

enum {
  COOKIE_REMOVED,
  LAST_SIGNAL,
};

static GObjectClass * parent_class = NULL;
static guint signals[LAST_SIGNAL] = { 0, };

static void e_cookies_class_init (ECookiesClass * klass);
static void e_cookies_init (ECookies * cookies);
static void e_cookies_finalize (GObject * object);

static void e_cookie_free (ECookie * cookie);
static ECookie * e_cookie_dup (ECookie * cookie);

GType e_cookies_get_type (void) {
  static GType type = 0;

  if (!type) {
    GTypeInfo info = {
      sizeof (ECookiesClass),
      NULL,
      NULL,
      (GClassInitFunc) e_cookies_class_init,
      NULL,
      NULL,
      sizeof (ECookies),
      0,
      (GInstanceInitFunc) e_cookies_init,
      NULL
    };

    type = g_type_register_static (G_TYPE_OBJECT, "ECookies", &info, 0);
  }

  return type;
}
      
static gboolean e_cookie_domcmp (const gchar * host, const gchar * domain) {
  gint domain_len, host_len;

  if (host == NULL || domain == NULL) {
    return FALSE;
  }

  domain_len = strlen (domain);
  host_len = strlen (host);

  if (domain_len > host_len) {
    return FALSE;
  } else {
    if (!strcmp (host + (host_len - domain_len), domain)) {
      return TRUE;
    } else if (!strcmp (host, domain)) {
      return TRUE;
    } else {
      return FALSE;
    }
  }
}

static gboolean e_cookie_xml_get_bool (const xmlNode * parent,
				       const gchar * prop_name) {
  gchar * prop;
  gboolean ret_val = FALSE;

  g_return_val_if_fail (parent != NULL, FALSE);
  g_return_val_if_fail (prop_name != NULL, FALSE);

  prop = xmlGetProp ((xmlNode *) parent, prop_name);
  if (prop != NULL) {
    if (!g_strcasecmp (prop, "true") || !g_strcasecmp (prop, "1")) {
      ret_val = TRUE;
    } else {
      ret_val = FALSE;
    }
    g_free (prop);
  }

  return ret_val;
}

static void e_cookie_xml_set_bool (const xmlNode * parent,
				   const gchar * prop_name, gboolean value) {
  g_return_if_fail (parent != NULL);
  g_return_if_fail (prop_name != NULL);

  if (value) {
    xmlSetProp ((xmlNode *) parent, prop_name, "true");
  } else {
    xmlSetProp ((xmlNode *) parent, prop_name, "false");
  }
}

static void e_cookies_class_init (ECookiesClass * klass) {
  GObjectClass * object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek (G_TYPE_OBJECT);

  object_class->finalize = e_cookies_finalize;

  /*
  signals[COOKIE_REMOVED] = g_signal_new ("cookie_removed",
					  G_TYPE_FROM_CLASS (klass),
					  G_SIGNAL_RUN_LAST,
					  G_STRUCT_OFFSET (ECookiesClass,
							   cookie_removed),
					  NULL, NULL,
					  g_cclosure_marshal_VOID__STRING,
					  G_TYPE_NONE, 1, G_TYPE_STRING);

  klass->cookie_removed = NULL;
  */
}

static gboolean e_cookies_timeout (ECookies * cookies) {
  GList * list;

  for (list = cookies->persist; list != NULL; list = list->next) {
    ECookie * cookie = list->data;

    if (cookie == NULL) {
      continue;
    }

    if (cookie->expires_t <= time (NULL)) {
      cookies->persist = g_list_remove (cookies->persist, cookie);
      e_cookies_save_cookies (cookies);
    }
  }

  return TRUE;
}

static void e_cookies_init (ECookies * cookies) {
  xmlDoc * ecookies;
  xmlNode * root, * list;
  gchar * cookiedb;

  cookies->persist = NULL;
  cookies->session = NULL;

  cookiedb = g_strconcat (g_get_home_dir (), "/.gnome2/cookies.xml", NULL);
  
  ecookies = xmlParseFile (cookiedb);

  if (ecookies == NULL) {
    ecookies = xmlNewDoc ("1.0");
    xmlCreateIntSubset (ecookies, "cookies", NULL, "e-cookies.dtd");
    root = xmlNewNode (NULL, "cookies");
    xmlDocSetRootElement (ecookies, root);
    xmlSaveFormatFile (cookiedb, ecookies, 1);
  } else {
    root = xmlDocGetRootElement (ecookies);
  }

  if (root == NULL || strcmp (root->name, "cookies")) {
    xmlCreateIntSubset (ecookies, "cookies", NULL, "e-cookies.dtd");
    root = xmlNewNode (NULL, "cookies");
    xmlDocSetRootElement (ecookies, root);
    xmlSaveFormatFile (cookiedb, ecookies, 1);
  } else {
    for (list = root->children; list != NULL; list = list->next) {
      if (!strcmp (list->name, "cookie")) {
	ECookie * cookie;

	cookie = g_new0 (ECookie, 1);

	cookie->domain = xmlGetProp (list, "domain");
	cookie->path = xmlGetProp (list, "path");
	cookie->name = xmlGetProp (list, "name");
	cookie->value = xmlGetProp (list, "value");
	cookie->expires = xmlGetProp (list, "expires");
	cookie->expires_t = e_date_rfc_parse (cookie->expires);
	cookie->secure = e_cookie_xml_get_bool (list, "secure");

	if (cookie->expires_t > time (NULL)) {
	  cookies->persist = g_list_append (cookies->persist,
					    e_cookie_dup (cookie));
	}

	e_cookie_free (cookie);
      }
    }
  }

  xmlFreeDoc (ecookies);
  g_free (cookiedb);

  g_timeout_add (30000, (GSourceFunc) e_cookies_timeout, cookies);
}

static void e_cookies_finalize (GObject * object) {
  ECookies * cookies = E_COOKIES (object);
  GList * list = NULL;

  for (list = cookies->persist; list != NULL; list = list->next) {
    ECookie * cookie = list->data;

    cookies->persist = g_list_remove (cookies->persist, cookie);
  }
  g_list_free (list);

  for (list = cookies->session; list != NULL; list = list->next) {
    ECookie * cookie = list->data;

    cookies->session = g_list_remove (cookies->session, cookie);
  }
  g_list_free (list);

  g_list_free (cookies->persist);
  g_list_free (cookies->session);

  parent_class->finalize (object);
}

ECookies * e_cookies_new (void) {
  return g_object_new (E_TYPE_COOKIES, NULL);
}

static void e_cookie_free (ECookie * cookie) {
  if (cookie == NULL ) {
    return;
  }

  g_free (cookie->domain);
  g_free (cookie->path);
  g_free (cookie->name);
  g_free (cookie->value);
  g_free (cookie->expires);
  
  g_free (cookie);
  cookie = NULL;
}

static ECookie * e_cookie_dup (ECookie * cookie) {
  ECookie * newcookie;

  newcookie = g_new0 (ECookie, 1);

  newcookie->domain = g_strdup (cookie->domain);
  newcookie->path = g_strdup (cookie->path);
  newcookie->name = g_strdup (cookie->name);
  newcookie->value = g_strdup (cookie->value);
  newcookie->expires = g_strdup (cookie->expires);
  newcookie->expires_t = cookie->expires_t;
  newcookie->secure = cookie->secure;

  return newcookie;
}

static ECookie * e_cookies_find_cookie (GList * cookies, ECookie * cookie) {
  ECookie * newcookie;
  GList * list;
  gboolean found = FALSE;

  if (cookies == NULL || cookie == NULL) {
    return NULL;
  }

  for (list = cookies; list != NULL; list = list->next) {
    newcookie = list->data;

    if (!strcmp (newcookie->domain, cookie->domain) &&
	!strcmp (newcookie->path, cookie->path) &&
	!strcmp (newcookie->name, cookie->name)) {

      found = TRUE;
      break;
    }
  }

  if (found) {
    return newcookie;
  } else {
    return NULL;
  }
}

gboolean e_cookies_add_cookie (ECookies * cookies, const gchar * header,
			       const gchar * hostname, const gchar * path) {
  ECookie * cookie = NULL;
  gchar ** pairs;
  gboolean cookie_added = FALSE;
  gint i;

  cookie = g_new0 (ECookie, 1);

  pairs = g_strsplit (header, ";", -1);
  for (i = 0; pairs && pairs[i] && *pairs[i]; i++) {
    pairs[i] = g_strchug (pairs[i]);
    if (!strchr (pairs[i], '=')) {
      if (!g_strcasecmp (pairs[i], "true") ||
	  !strcmp (pairs[i], "1")) {
	cookie->secure = TRUE;
      } else {
	cookie->secure = FALSE;
      }
    } else {
      gchar * foo;
      foo = g_strndup (pairs[i], strlen (pairs[i]) -
					 strlen (strchr (pairs[i], '=')));
      if (!g_strcasecmp (foo, "domain")) {
	cookie->domain = g_strndup (pairs[i] + strlen (pairs[i]) -
				    strlen (strchr (pairs[i], '=')) + 1,
				    strlen (strchr (pairs[i], '=')) + 1);
      } else if (!g_strcasecmp (foo, "path")) {
	cookie->path = g_strndup (pairs[i] + strlen (pairs[i]) -
				  strlen (strchr (pairs[i], '=')) + 1,
				  strlen (strchr (pairs[i], '=')) + 1);
      } else if (!g_strcasecmp (foo, "expires")) {
	cookie->expires = g_strndup (pairs[i] + strlen (pairs[i]) -
				     strlen (strchr (pairs[i], '=')) + 1,
				     strlen (strchr (pairs[i], '=')) + 1);
	cookie->expires_t = e_date_rfc_parse (cookie->expires);
      } else {
	if (pairs[i][strlen (pairs[i]) - 1] != '=') {
	  cookie->name = g_strndup (pairs[i], strlen(pairs[i]) -
				    strlen (strchr (pairs[i], '=')));
	  cookie->value = g_strndup (pairs[i] + strlen (pairs[i]) -
				     strlen (strchr (pairs[i], '=')) + 1,
				     strlen (strchr (pairs[i], '=')) + 1);
	}
      }
      g_free (foo);
    }
  }

  if (cookie->domain == NULL || !strcmp (cookie->domain, "")) {
    cookie->domain = g_strdup (hostname);
  }
  if (cookie->path == NULL || !strcmp (cookie->path, "")) {
    cookie->path = g_strdup (path);
  }

  if (cookie->expires != NULL & *cookie->expires &&
      cookie->name != NULL && *cookie->name) {
    ECookie * pcookie;

    pcookie = e_cookies_find_cookie (cookies->persist, cookie);
    if (pcookie != NULL) {
      if (!strcmp (pcookie->domain, cookie->domain) &&
	  !strcmp (pcookie->path, cookie->path) &&
	  !strcmp (pcookie->name, cookie->name)) {
	cookies->persist = g_list_remove (cookies->persist, pcookie);
	e_cookie_free (pcookie);
      }
      if (cookie->expires_t >= time (NULL)) {
	cookies->persist = g_list_append (cookies->persist,
					  e_cookie_dup (cookie));
	cookie_added = TRUE;
      }
    } else if (cookie->expires_t >= time (NULL)) {
      cookies->persist = g_list_append (cookies->persist,
					 e_cookie_dup (cookie));
      cookie_added = TRUE;
    }
  } else if (cookie->name != NULL && *cookie->name) {
    ECookie * scookie;

    scookie = e_cookies_find_cookie (cookies->session, cookie);
    if (scookie != NULL) {
      if (!strcmp (scookie->domain, cookie->domain) &&
	  !strcmp (scookie->path, cookie->path) &&
	  !strcmp (scookie->name, cookie->name)) {
	cookies->session = g_list_remove (cookies->session, scookie);
	e_cookie_free (scookie);
      }
      cookies->session = g_list_append (cookies->session,
					e_cookie_dup (cookie));
      cookie_added = TRUE;
    } else {
      cookies->session = g_list_append (cookies->session,
					e_cookie_dup (cookie));
      cookie_added = TRUE;
    }
  }

  e_cookie_free (cookie);
  g_strfreev (pairs);

  return cookie_added;
}

gchar * e_cookies_find_cookies (ECookies * cookies, const gchar * domain,
				const gchar * path) {
  GList * plist, * slist;
  gchar * header = g_strdup ("");

  for (plist = cookies->persist; plist != NULL; plist = plist->next) {
    ECookie * pcookie = e_cookie_dup (plist->data);

    if (e_cookie_domcmp (domain, pcookie->domain) &&
	(strlen (pcookie->path) <= strlen (path) &&
	 !strncmp (pcookie->path, path, strlen (pcookie->path)))) {
      header = g_strconcat (header, pcookie->name, "=",
			    pcookie->value, "; ", NULL);
    }
    e_cookie_free (pcookie);
  }

  for (slist = cookies->session; slist != NULL; slist = slist->next) {
    ECookie * scookie = e_cookie_dup (slist->data);

    if (e_cookie_domcmp (domain, scookie->domain) &&
	(strlen (scookie->path) <= strlen (path) &&
	 !strncmp (scookie->path, path, strlen (scookie->path)))) {
      header = g_strconcat (header, scookie->name, "=",
			    scookie->value, "; ", NULL);
    }
    e_cookie_free (scookie);
  }

  return header;
}

void e_cookies_save_cookies (ECookies * cookies) {
  xmlDoc * ecookies;
  xmlNode * root;
  GList * list;
  gchar * cookiedb;

  cookiedb = g_strconcat (g_get_home_dir (), "/.gnome2/cookies.xml", NULL);
  
  ecookies = xmlNewDoc ("1.0");
  xmlCreateIntSubset (ecookies, "cookies", NULL, "e-cookies.dtd");
  root = xmlNewNode (NULL, "cookies");
  xmlDocSetRootElement (ecookies, root);

  for (list = cookies->persist; list != NULL; list = list->next) {
    ECookie * cookie = e_cookie_dup (list->data);
    xmlNode * node;

    node = xmlNewChild (root, NULL, "cookie", NULL);
    xmlSetProp (node, "domain", cookie->domain);
    xmlSetProp (node, "path", cookie->path);
    xmlSetProp (node, "name", cookie->name);
    xmlSetProp (node, "value", cookie->value);
    xmlSetProp (node, "expires", cookie->expires);
    e_cookie_xml_set_bool (node, "secure", cookie->secure);

    e_cookie_free (cookie);
  }

  xmlSaveFormatFile (cookiedb, ecookies, 1);
  xmlFreeDoc (ecookies);

  g_free (cookiedb);
}
