/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Query declarations
 */

#ifndef __DRIZZLE_QUERY_H
#define __DRIZZLE_QUERY_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @addtogroup drizzle_query Query Functions
 * @ingroup drizzle_client
 * These functions are used to issue queries on a connection. Single queries are
 * made using the drizzle_query function, or you can queue multiple queries and
 * run them concurrently using the other query functions.
 * @{
 */

/**
 * Send query to server.
 */
DRIZZLE_API
drizzle_result_st *drizzle_query(drizzle_con_st *con, drizzle_result_st *result,
                                 const char *query, size_t size,
                                 drizzle_return_t *ret_ptr);

/**
 * Send query to server, using strlen to get the size of query buffer..
 */
DRIZZLE_API
drizzle_result_st *drizzle_query_str(drizzle_con_st *con,
                                     drizzle_result_st *result,
                                     const char *query,
                                     drizzle_return_t *ret_ptr);

/**
 * Send query incrementally.
 */
DRIZZLE_API
drizzle_result_st *drizzle_query_inc(drizzle_con_st *con,
                                     drizzle_result_st *result,
                                     const char *query, size_t size,
                                     size_t total, drizzle_return_t *ret_ptr);

/**
 * Add a query to be run concurrently.
 */
DRIZZLE_API
drizzle_query_st *drizzle_query_add(drizzle_st *drizzle,
                                    drizzle_query_st *query,
                                    drizzle_con_st *con,
                                    drizzle_result_st *result,
                                    const char *query_string, size_t size,
                                    drizzle_query_options_t options,
                                    const void *data);

/**
 * Initialize a query structure.
 */
DRIZZLE_API
drizzle_query_st *drizzle_query_create(drizzle_st *drizzle,
                                       drizzle_query_st *query);

/**
 * Free a query structure.
 */
DRIZZLE_API
void drizzle_query_free(drizzle_query_st *query);

/**
 * Get connection struct for a query.
 */
DRIZZLE_API
drizzle_con_st *drizzle_query_con(drizzle_query_st *query);

/**
 * Set connection struct for a query.
 */
DRIZZLE_API
void drizzle_query_set_con(drizzle_query_st *query, drizzle_con_st *con);

/**
 * Get result struct for a query.
 */
DRIZZLE_API
drizzle_result_st *drizzle_query_result(drizzle_query_st *query);

/**
 * Set result struct for a query.
 */
DRIZZLE_API
void drizzle_query_set_result(drizzle_query_st *query,
                              drizzle_result_st *result);

/**
 * Get query string for a query.
 */
DRIZZLE_API
char *drizzle_query_string(drizzle_query_st *query, size_t *size);

/**
 * Set query string for a query.
 */
DRIZZLE_API
void drizzle_query_set_string(drizzle_query_st *query, const char *string,
                              size_t size);

/**
 * Get options for a query. 
 */
DRIZZLE_API
drizzle_query_options_t drizzle_query_options(drizzle_query_st *query);

/**
 * Set options for a query.
 */
DRIZZLE_API
void drizzle_query_set_options(drizzle_query_st *query,
                               drizzle_query_options_t options);

/**
 * Add options for a query.
 */
DRIZZLE_API
void drizzle_query_add_options(drizzle_query_st *query,
                               drizzle_query_options_t options);

/**
 * Remove options for a query.
 */
DRIZZLE_API
void drizzle_query_remove_options(drizzle_query_st *query,
                                  drizzle_query_options_t options);

/**
 * Get application data for a query.
 */
DRIZZLE_API
void *drizzle_query_data(drizzle_query_st *query);

/**
 * Set application data for a query.
 */
DRIZZLE_API
void drizzle_query_set_data(drizzle_query_st *query, const void *data);

/**
 * Set callback function when the data pointer should be freed.
 */
DRIZZLE_API
void drizzle_query_set_data_free(drizzle_query_st *query,
                                 drizzle_query_data_free_fn *free_fn);

/**
 * Run queries concurrently, returning when one is complete.
 */
DRIZZLE_API
drizzle_query_st *drizzle_query_run(drizzle_st *drizzle,
                                    drizzle_return_t *ret_ptr);

/**
 * Run queries until they are all complete.
 */
DRIZZLE_API
drizzle_return_t drizzle_query_run_all(drizzle_st *drizzle);

/*
 * Escape a string or encode a string in hexadecimal. The return value is the
 * size of the output string in to.
 */
DRIZZLE_API
uint64_t drizzle_escape_string(char *to, const char *from, uint64_t from_size);
DRIZZLE_API
uint64_t drizzle_hex_string(char *to, const char *from, uint64_t from_size);

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* __DRIZZLE_QUERY_H */
