/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Connection declarations for servers
 */

#ifndef __DRIZZLE_CON_SERVER_H
#define __DRIZZLE_CON_SERVER_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @addtogroup drizzle_con_server Connection Handling for Servers
 * @ingroup drizzle_server
 * These functions extend the core connection functions with a set of functions
 * for server application use. These functions allow you to set raw handshake in
 * formation for use with the handshake write functions.
 * @{
 */

/**
 * Set protocol version for a connection.
 */
DRIZZLE_API
void drizzle_con_set_protocol_version(drizzle_con_st *con,
                                      uint8_t protocol_version);

/**
 * Set server version string for a connection.
 */
DRIZZLE_API
void drizzle_con_set_server_version(drizzle_con_st *con,
                                    const char *server_version);

/**
 * Set thread ID for a connection.
 */
DRIZZLE_API
void drizzle_con_set_thread_id(drizzle_con_st *con, uint32_t thread_id);

/**
 * Set scramble buffer for a connection.
 */
DRIZZLE_API
void drizzle_con_set_scramble(drizzle_con_st *con, const uint8_t *scramble);

/**
 * Set capabilities for a connection.
 */
DRIZZLE_API
void drizzle_con_set_capabilities(drizzle_con_st *con,
                                  drizzle_capabilities_t capabilities);

/**
 * Set charset for a connection.
 */
DRIZZLE_API
void drizzle_con_set_charset(drizzle_con_st *con, drizzle_charset_t charset);

/**
 * Set status for a connection.
 */
DRIZZLE_API
void drizzle_con_set_status(drizzle_con_st *con, drizzle_con_status_t status);

/**
 * Set max packet size for a connection.
 */
DRIZZLE_API
void drizzle_con_set_max_packet_size(drizzle_con_st *con,
                                     uint32_t max_packet_size);

/**
 * Copy all handshake information from one connection into another.
 */
DRIZZLE_API
void drizzle_con_copy_handshake(drizzle_con_st *con, drizzle_con_st *from);

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* __DRIZZLE_CON_SERVER_H */
