/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Command declarations for clients
 */

#ifndef __DRIZZLE_COMMAND_CLIENT_H
#define __DRIZZLE_COMMAND_CLIENT_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @addtogroup drizzle_command_client Command Functions for Clients
 * @ingroup drizzle_client
 * These functions are used to issue commands on a connection. Normal SQL
 * queries are issued using the drizzle_query* functions defined in query.h.
 * @{
 */

/**
 * Send quit command on a connection.
 */
DRIZZLE_API
drizzle_result_st *drizzle_quit(drizzle_con_st *con, drizzle_result_st *result,
                                drizzle_return_t *ret_ptr);

/**
 * Select a new default database for a connection.
 */
DRIZZLE_API
drizzle_result_st *drizzle_select_db(drizzle_con_st *con,
                                     drizzle_result_st *result, const char *db,
                                     drizzle_return_t *ret_ptr);


/**
 * Refresh using the given options.
 */
DRIZZLE_API
drizzle_result_st *drizzle_refresh(drizzle_con_st *con,
                                   drizzle_result_st *result,
                                   drizzle_refresh_options_t options,
                                   drizzle_return_t *ret_ptr);

/**
 * Send a shutdown message at the given level.
 */
DRIZZLE_API
drizzle_result_st *drizzle_shutdown(drizzle_con_st *con,
                                    drizzle_result_st *result,
                                    drizzle_shutdown_level_t level,
                                    drizzle_return_t *ret_ptr);

/**
 * Send request to get statistics for a connection.
 */
DRIZZLE_API
drizzle_result_st *drizzle_stat(drizzle_con_st *con, drizzle_result_st *result,
                                drizzle_return_t *ret_ptr);

/**
 * Send request to get debug info for a connection.
 */
DRIZZLE_API
drizzle_result_st *drizzle_debug_info(drizzle_con_st *con,
                                      drizzle_result_st *result,
                                      drizzle_return_t *ret_ptr);

/**
 * Send ping request on a connection.
 */
DRIZZLE_API
drizzle_result_st *drizzle_ping(drizzle_con_st *con, drizzle_result_st *result,
                                drizzle_return_t *ret_ptr);

/**
 * Change user for a connection.
 */
DRIZZLE_API
drizzle_result_st *drizzle_change_user(drizzle_con_st *con,
                                       drizzle_result_st *result,
                                       const char *user, const char *password,
                                       const char *db,
                                       drizzle_return_t *ret_ptr);

/**
 * Send raw command to server, possibly in parts.
 */
DRIZZLE_API
drizzle_result_st *drizzle_command_write(drizzle_con_st *con,
                                         drizzle_result_st *result,
                                         drizzle_command_t command,
                                         const uint8_t *data, size_t size,
                                         size_t total,
                                         drizzle_return_t *ret_ptr);

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* __DRIZZLE_COMMAND_CLIENT_H */
