# vim: ts=8 sw=2 sts=0 noexpandtab:
##########################################################
## This script is part of the Devel::NYTProf distribution
##
## Copyright, contact and other information can be found
## at the bottom of this file, or by going to:
## http://search.cpan.org/dist/Devel-NYTProf/
##
###########################################################
# $Id: Makefile.PL 723 2009-03-26 11:38:14Z thatsafunnyname $
###########################################################

use 5.008001;
use warnings;
use strict;

use ExtUtils::MakeMaker;
use Getopt::Long;
use Config;

my $is_developer = (-d '.svn');

# This lets me build it under ext/Devel, and profile parts of the core in place
unless($ENV{PERL_CORE}) {
    $ENV{PERL_CORE} = 1 if grep { $_ eq 'PERL_CORE=1' } @ARGV;
}
my @man;
if ($ENV{PERL_CORE}) {
  @man = ( MAN1PODS => {}, MAN3PODS => {} );
} else {
  @man = ( MAN1PODS  => {
        'bin/nytprofhtml' => '$(INST_MAN1DIR)/nytprofhtml.1',
        'bin/nytprofcsv'  => '$(INST_MAN1DIR)/nytprofcsv.1'
  } );
}

# --- Options
GetOptions(
    'g!'  => \my $opt_g,
    'pg!' => \my $opt_pg,
) or exit 1;


# --- make sure t/test40pmc.pmc is newer than t/test40pmc.pmc

utime time(), time(), "t/test40pmc.pmc"
    or die "Can't update mod time of t/test40pmc.pmc";


# --- Discover how much of stdio is implemented

my $cpp = $Config{cpp} || do {
    warn "Warning: cpp not found in your perl config.  Falling back to 'cat'\n";
    'cat';
};

print "Looking for header files and functions...\n";
my $INCLUDE;

my $h_files;
my @h_dirs;
push @h_dirs, split /:/, $ENV{INCLUDE} if $ENV{INCLUDE};
push @h_dirs, split ' ', $Config{libsdirs};
push @h_dirs, qw(/include /usr/include /usr/local/include);
@h_dirs = grep { -d $_ } @h_dirs;

$h_files = find_h_files(@h_dirs);

# See lib/ExtUtils/MakeMaker.pm for details of how to influence
# the contents of the Makefile that is written.
my %mm_opts;
my @libs = ();

foreach (['time.h', qr/(clock_gettime)\s*\(/, '-DHAS_CLOCK_GETTIME', '-lrt'],
	 ['zlib.h', qr/(deflateInit2)(?:_)?\s*\(/, '-DHAS_ZLIB', '-lz'],
	) {
    my ($header, $regexp, $define, $libs) = @$_;
    if (my $result = search_h_file($header, $regexp)) {
	print "Found $result in $header\n";
	push @libs, $libs;
	$mm_opts{DEFINE} .= " $define";
    }
}

$mm_opts{LICENSE}  = 'perl' if $ExtUtils::MakeMaker::VERSION >= 6.3002;
$mm_opts{OPTIMIZE} = '-g'   if $opt_g;
$mm_opts{CCFLAGS}  = "-pg"  if $opt_pg;

if (my $gccversion = $Config{gccversion}) {    # ask gcc to be more pedantic
    print "Your perl was compiled with gcc (version $Config{gccversion}), okay.\n";
    $gccversion =~ s/[^\d\.]//g;               # just a number please
    $mm_opts{DEFINE} .= ' -W -Wall -Wpointer-arith -Wbad-function-cast';
    $mm_opts{DEFINE} .= ' -Wno-comment -Wno-sign-compare -Wno-cast-qual';
    $mm_opts{DEFINE} .= ' -Wmissing-noreturn -Wno-unused-parameter' if $gccversion ge "3.0";
    if ($is_developer && $opt_g) {
        $mm_opts{DEFINE} .= ' -DPERL_GCC_PEDANTIC -ansi -pedantic' if $gccversion ge "3.0";
        $mm_opts{DEFINE} .= ' -Wdisabled-optimization -Wformat'    if $gccversion ge "3.0";
        $mm_opts{DEFINE} .= ' -Wmissing-prototypes';
    }
}


WriteMakefile(
    NAME          => 'Devel::NYTProf',
    VERSION_FROM  => 'lib/Devel/NYTProf/Core.pm',    # finds $VERSION
    ABSTRACT_FROM => 'lib/Devel/NYTProf.pm',         # retrieve abstract from module
    AUTHOR    => 'Adam Kaplan <akaplan@cpan.org>, Tim Bunce <timb@cpan.org>',
    PREREQ_PM => {
        'List::Util'   => 0,
        'Test::More'   => 0,
        'XSLoader'     => 0,
        'Getopt::Long' => 0,
    },
    LIBS      => [join ' ', @libs],
    EXE_FILES => ['bin/nytprofhtml', 'bin/nytprofcsv'],
    @man,
    INC   => $INCLUDE,
    clean => { FILES => join " ",
		"nytprof demo-out",
		map { ("t/$_", "xt/$_") }
			qw(nytprof nytprof*.out *_outdir test*.*.new auto)
    },
    dist  => {
        DIST_DEFAULT => 'clean distcheck disttest tardist',
        PREOP        => '$(MAKE) -f Makefile.old distdir',
        COMPRESS     => 'gzip -v9',
        SUFFIX       => 'gz',
    },
    %mm_opts,
);

exit 0;

# --- Utility functions ---


sub find_h_files {
    my @dirs = @_;
    my %h_files;
    foreach my $dir (@dirs) {
        next unless $dir;
        opendir(DIR, $dir)
            or next;    # silently ignore missing directories

        while (my $file = readdir(DIR)) {
            next unless $file =~ /\.h$/;
            $h_files{$file} ||= $dir;    # record first found
        }
    }
    close DIR;
    return \%h_files;
}

sub search_h_file {
    my ($h_file, $regex) = @_;
    my $dir = $h_files->{$h_file}
        or return undef;
    open H, "$cpp $dir/$h_file |";
    while (<H>) {
        return $1 if m/$regex/;
    }
    close H;
    return undef;
}


# --- MakeMaker overrides ---

package MY;

# add some extra utility targets to the make file
sub post_constants {
    q{
svnmanifest::
	svn list -R .@HEAD | sort | grep -v '/$$' > MANIFEST
	svn diff MANIFEST

checkkeywords:
	$(RM_RF) blib
	find .  \( -name .svn -prune -or -name t -prune -or -name \*.pm -or -name \*.PL -or -name \*.pl \) -type f \
	    -exec bash -c '[ "$$(svn pg svn:keywords {})" != "Id Revision Date" ] && echo svn propset svn:keywords \"Id Revision Date\" {}' \;

checkpod:
	$(RM_RF) blib
	find . -type f \( -name .svn -prune -o -name \*.pm -o -name \*.PL -o -name \*.pl \) \
	    -exec podchecker {} \; 2>&1 | grep -v "pod syntax OK"

PERLTIDY=perltidy --profile=.perltidyrc -nst -b
perltidy:
	$(PERLTIDY) bin/nytprofhtml bin/nytprofcsv
	$(PERLTIDY) lib/Devel/NYTProf.pm lib/Devel/NYTProf/*.pm

# the XS portion of the file will be mangled and require manual fixup
ctidy_bcpp:
	bcpp -f 2 -i 4 -bcl -qb 10 -ylcnc -yb NYTProf.xs

}
}

sub dynamic {
    my $make = shift->SUPER::dynamic(@_);

    my $xsl_dest_dir = '$(INST_LIB)/$(PARENT_NAME)/auto/$(FULLEXT)';
    my $xsl_dest     = '$(XSL_DEST_DIR)/$(DLBASE).$(DLEXT)';

    if($^O eq 'VMS'){
        $xsl_dest_dir = File::Spec->catdir('blib','lib','Devel','auto','Devel','NYTProf');
        $xsl_dest     = File::Spec->catfile('blib','lib','Devel','auto','Devel','NYTProf','PL_Devel__NYTProf'.'.exe');
    }

    $make .= join "\n",
        '# Copy extension to where XSLoader looks to avoid fallback to DynaLoader',
        '# See t/test14.p for more details',
        "XSL_DEST_DIR = $xsl_dest_dir",
        "XSL_DEST = $xsl_dest",
        '',
        'dynamic :: $(INST_DYNAMIC)',
        "\t" . '$(NOECHO) $(MKPATH) $(XSL_DEST_DIR)',
        "\t" . '$(CP) $(INST_DYNAMIC) $(XSL_DEST)',
        '';
    return $make;
}


# vim:ts=8:sw=4:sts=0:noexpandtab
