package com.tildemh.debbug;

/**
 * Interface for events related to a listing object.
 * 
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: ListingListener.java,v 1.5 2004/03/04 17:38:04 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public interface ListingListener{


	/**
	 * Called whenever the numbers of bugs of each type (including number of read
	 * bugs) changes. If this is likely to happen a lot in a short period of
	 * time (e.g. when loading a listing from disk), it may not be called for
	 * every change.
	 */  
	public void bugCountsChanged( Listing listing );

	/**
	 * Called when the listing is about to contact the server to perform an
	 * update.
	 */
	public void listingUpdateStart( Listing listing );

	/**
	 * Called when the listing has completed it's interactions with the server.
	 */
	public void listingUpdateDone( Listing listing );

	/**
	 * Called occasionally when downloading a package listing. This may be used
	 * as part of a progress screen. 
	 */
	public void downloadingListing( Listing listing );

	/**
	 * Called at the start of interpreting the listing - deciding which bugs are
	 * new, modified or removed.
	 */
	public void interpretingListing( Listing listing );

	/**
	 * Called at the start of te download of each bug report from the server as
	 * part of a listing update process.
	 * @param bugNumber The number of the bug being downloaded
	 * @param pending Number of bugs downloaded so far + 1
	 * @param total Number of bugs to be downloaded. Note that this is not the
	 * number of bugs in the listing, merely the number of bugs which need
	 * updating.
	 */
	public void downloadingListingBug( Listing listing, Integer bugNumber, int pending, int total);

	/**
	 * Called when a bug is removed from the listing (e.g. when the server says
	 * the bug has been fixed for a list of open reports)
	 */
	public void bugRemoved( Listing listing, Integer bugNumber );

	/**
	 * Called when a bug is added to the listing for the first time.
	 */
	public void bugAdded( Listing listing, Bug bug );

	/**
	 * Called when a listing has completed loading all reports into memory (and
	 * so called bugAdded for all of them). This can be used to flush queues.
	 */
	public void loadReportsDone( Listing listing );
	

	/**
	 * Called when a modification is made to a bug in the listing (including
	 * change of read/unread status) (excluding addition or removal of bugs).
	 */
	public void bugChanged( Listing listing, Bug bug );

	/**
	 * Called when an exception occurred while updating the report. 
	 * @return true if you want the execution to continue, false otherwise.
	 */
	public boolean  listingException( Listing listing, Exception e );

}
