package org.lobobrowser.html.renderer;

import javax.swing.*;
import java.awt.event.*;
import java.util.*;

import org.lobobrowser.html.domimpl.*;
import org.lobobrowser.util.gui.WrapperLayout;
import org.w3c.dom.Node;
import org.w3c.dom.html2.*;

class InputSelectControl extends BaseInputControl {
	private final JComboBox comboBox;
	
	public InputSelectControl(HTMLBaseInputElement modelNode) {
		super(modelNode);
		this.setLayout(WrapperLayout.getInstance());
		JComboBox comboBox = new JComboBox();
		comboBox.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				OptionItem item = (OptionItem) e.getItem();
				if(item != null) {
					switch(e.getStateChange()) {
					case ItemEvent.SELECTED:
						item.setSelected(true);
						break;
					case ItemEvent.DESELECTED:
						// Ignore deselection here. It must necessarily
						// be followed by selection. If we deselect, that
						// changes the state of the control.
						break;
					}					
				}
			}
		});
		this.add(comboBox);

		//Note: Value attribute cannot be set in reset() method.
		//Otherwise, layout revalidation causes typed values to
		//be lost (including revalidation due to hover.)

		this.comboBox = comboBox;
		this.resetItemList();
	}

	private void resetItemList() {
		JComboBox comboBox = this.comboBox;
		HTMLSelectElementImpl selectElement = (HTMLSelectElementImpl) this.controlElement;
		ArrayList list = selectElement.getDescendents(HTMLOptionsCollectionImpl.OPTION_FILTER, false);
		// First determine current selected option
		HTMLOptionElement priorSelectedOption = null;
		int priorIndex = selectElement.getSelectedIndex();
		if(priorIndex != -1) {
			Iterator i1 = list.iterator();
			int index = 0;
			while(i1.hasNext()) {
				HTMLOptionElement option = (HTMLOptionElement) i1.next();
				if(index++ == priorIndex) {
					priorSelectedOption = option;
				}			
			}
		}
		comboBox.removeAllItems();
		OptionItem defaultItem = null;
		OptionItem selectedItem = null;
		OptionItem firstItem = null;
		Iterator i = list.iterator();
		while(i.hasNext()) {
			HTMLOptionElement option = (HTMLOptionElement) i.next();
			OptionItem item = new OptionItem(option); 
			if(firstItem == null) {
				firstItem = item;
				comboBox.addItem(item);
				// Undo automatic selection that occurs when adding the first item.
				// This might set the deferred index as well.
				selectElement.setSelectedIndex(-1);
				if(priorSelectedOption != null) {
					priorSelectedOption.setSelected(true);
				}
			}
			else {
				comboBox.addItem(item);				
			}
			if(option.getSelected()) {
				selectedItem = item;
			}
			if(option.getDefaultSelected()) {
				defaultItem = item;
			}
		}
		if(selectedItem != null) {
			comboBox.setSelectedItem(selectedItem);			
		}
		else if(defaultItem != null) {
			comboBox.setSelectedItem(defaultItem);
		}
		else if(firstItem != null) {
			comboBox.setSelectedItem(firstItem);
		}		
	}
	
	public void reset(int availWidth, int availHeight) {
		super.reset(availWidth, availHeight);
		// Need to do this here in case element was incomplete
		// when first rendered.
		this.resetItemList();
	}
	
	public String getValue() {
		OptionItem item = (OptionItem) this.comboBox.getSelectedItem();
		if(item == null) {
			return null;
		}
		else {
			return item.getValue();
		}
	}

	public boolean getMultiple() {
		return false;
	}
	
	public void setMultiple(boolean value) {
		if(value) {
			//TODO ?
			throw new UnsupportedOperationException();
		}
	}

	private int selectedIndex = -1;
	
	public int getSelectedIndex() {
		return this.selectedIndex;
	}
	
	public void setSelectedIndex(int value) {
		this.selectedIndex = value;
		JComboBox comboBox = this.comboBox;
		if(comboBox.getSelectedIndex() != value) {
			// This check is done to avoid an infinite recursion
			// on ItemListener.
			int size = comboBox.getItemCount();
			if(value < size) {
				comboBox.setSelectedIndex(value);
			}
		}
	}
	
	public int getVisibleSize() {
		return this.comboBox.getMaximumRowCount();
	}
	
	public void setVisibleSize(int value) {
		this.comboBox.setMaximumRowCount(value);
	}
	
	public void resetInput() {
		this.comboBox.setSelectedIndex(-1);
	}
	
	private static class OptionItem {
		private final HTMLOptionElement option;
		private final String caption;
		
		public OptionItem(HTMLOptionElement option) {
			this.option = option;
			String label = option.getLabel();
			if(label == null) {
				this.caption = option.getText();
			}
			else {
				this.caption = label;
			}
		}
		
		public void setSelected(boolean value) {
			this.option.setSelected(value);
		}
		
		public String toString() {
			return this.caption;
		}
		
		public String getValue() {
			String value = this.option.getValue();
			if(value == null) {
				value = this.option.getText();
			}
			return value;
		}
	}
}
