# This class is one of the most important classes in CGIKit.
# CKApplication has two major roles. 
# 
# One role is that CKAplication provides parameters which 
# effect the whole behavior of CGIKit application.
# The other role is that CKApplication communicates with CKAdapter, 
# an interface between a web server and CGIKit.
# 
# When CGIKit receives a request from a client,
# CKApplication arranges its parameters and creates a CKAdapter object.
# Next, CKApplication gets a request object from the CKAdapter object. 
# In reponse to the request object, CKApplication creates 
# a response object and sends it to the CKAdapter object.  
#
# == Application Process
#
# === 1. Ready a request and response object
# First, CKApplication creates a CKAdapter object and gets a request
# ( CKRequest ) object from it. Then, CKApplication creates a response 
# ( CKResponse ) object. 
#
# === 2. Event loop
# Next, CKApplication goes into event loop. In this loop, CKApplication creates 
# components depending on the request and invokes the specified method of 
# the components until it returns an object whose class is not CKComponent.
#
# === 3. Return the response object to the CKAdapter object
# Finally, CKApplication converts the components created in event loop to HTML
# and adds the HTML to the CKResponse object created in 1.
# CKApplication sends the response object to the CKAdapter object.
# Then, the CKAdapter object displays the response object to a client.
# 
# == Programming Topics
# * Architecture[www.spice-of-life.net/download/cgikit/en/userguide/architecture.html]
# * SessionManagement[www.spice-of-life.net/download/cgikit/en/userguide/session.html]
class CKApplication
	class SessionAuthorizationError < CKError; end
	class SessionTimeoutError       < CKError; end

	# Main component. If the element ID  isn't specified, this component is shown.
	# The default value is 'MainPage'.
	attr_accessor :main

	# Locale of an application in a transaction. If the locale is specified,
	# CGIKit change a template for a component. The name of the template
	# includes the component name and the locale name. Also, the template name
	# is seprated by underscore("_"). For example, if the locale is "ja" and
	# the component name is "MainPage", the template name is "MainPage_ja.html".
	attr_accessor :locale

	# Main locale of an application in a transaction. If the master_locale
	# is equal to the specified locale, the components use templates whose name
	# doesn't include the locale name. For instance, if the master_locale is
	# "ja", the locale is "ja" and the component name is "MainPage",
	# the template name is "MainPage.html".
	attr_accessor :master_locale

	# Document root directory.
	attr_accessor :document_root

	# The file system path of the application.
	attr_accessor :path

	# The file system path for components. Components are searched under it.
	attr_accessor :component_path

	# Resource directory. It has files that don't be send to browser.
	attr_accessor :resources

	# Web server resources directory.
	# If "file" attribute of CKImage is set,
	# the element searches an image file from the directory.
	attr_accessor :web_server_resources

	# CKResourceManager object.
	attr_accessor :resource_manager

	# Adapter object.
	attr_accessor :adapter

	# Interface of adapter. The default value is an interface for CGI.
	attr_accessor :interface

	# HTTP request object ( CKRequest ).
	attr_accessor :request

	# HTTP response object ( CKResponse ).
	attr_accessor :response

	# Hash of logging options.
	#
	# level::         Log level. Select in CKLog::DEBUG, CKLog::INFO, CKLog::WARN,
	#                 CKLog::ERROR, CKLog::FATAL.
	# name::          Program name.
	# out::           Outputter.
	# file::          File name to output logs. The attribute has priority over "out".
	# max_file_size:: Max file size to log. Enables if "file" is setted.
	attr_accessor :log_options

	# Name of an error page component to show caught errors.
	attr_accessor :error_page

	# Temporary directory for CGIKit.
	attr_accessor :tmpdir

	# Session object. If session don't exist, creates a new session.
	attr_writer :session

	# Session ID.
	attr_accessor :session_id

	# Session key.
	attr_accessor :session_key

	# Seconds until the session has timed out.
	attr_accessor :timeout

	# Expiry date of cookie for session. If you set the value to nil,
	# session cookies will be invalid when closing browser.
	attr_accessor :session_cookie_expires

	# Enables or disables the use of URLs for storing session IDs.
	attr_accessor :store_in_url

	# Enables or disables the use of cookies for storing session IDs.
	attr_accessor :store_in_cookie

	# Enables or disables automatic session management.
	attr_accessor :manage_session

	# Enables or disables session authorization by browsers.
	attr_accessor :auth_by_user_agent

	# Enables or disables session authorization by IP addresses.
	attr_accessor :auth_by_remote_addr

	# Database manager class saving the session.
	attr_accessor :database_manager

	# Element ID ( CKElementID object ).
	attr_accessor :element_id

	# Session database object ( CKSessionStore ).
	attr_accessor :session_store

	# Character code to convert form data used in CKComponent#canvert_char_code.
	# Select these Japanese codes: 'jis', 'sjis' or 'euc'. If the value is nil,
	# raw form data is substituted for variables. The default value is nil.
	#
	# The attribute and method is setted for Japanese character codes by default.
	# If you convert form data to non-Japanese character codes, override the method.
	attr_accessor :char_code

	# Handled exception.
	attr_reader :error

	VERSION  = '1.2.1'

	class << self
		# Returns version of CGIKit.
		def version
			VERSION
		end
	end

	def initialize
		@path                   = $0
		@component_path         = Dir.pwd
		@main                   = 'MainPage'
		@error_page             = 'CKErrorPage'
		@tmpdir                 = './tmp' || ENV['TMP'] || ENV['TEMP']
		@session_key            = '_session_id'
		@manage_session         = false
		@timeout                = 60 * 60 * 24 * 7
		@session_cookie_expires = 60 * 60 * 24 * 7
		@store_in_url           = true
		@store_in_cookie        = true
		@auth_by_user_agent     = false
		@auth_by_remote_addr    = false
		@database_manager       = CKSessionStore::FileStore
		@log_options            = {}
		@char_code              = nil
		@resources              = './'

		# decides interface of adapter
		if defined?(MOD_RUBY) then
			@interface = CKAdapter::ModRuby
		else
			@interface = CKAdapter::CGI
		end

		CKElement.load_element_file(@component_path, 'CKErrorPage')

		init
	end

	# Hook method to initialize for convenience.
	def init; end

	# Returns the name of the application without file extension.
	def name
		File.basename( @path, '.*' )
	end

	# The application URL based on SCRIPT_NAME with session ID.
	def baseurl( session = false )
		if (session == true) and session? then
			@baseurl + "?#@session_key=#@session_id"
		else
			@baseurl
		end
	end

	# Creates a session.
	def create_session
		session             = CKSession.new
		session.user_agent  = @request.user_agent
		session.remote_addr = @request.remote_addr
		session
	end

	# Returns a restored session objects with the session ID.
	def restore_session( session_id )
		session_store.restore session_id
	end

	# Deletes the session.
	def clear_session( session )
		session.clear
		save_session session
		@session_id = nil
		@session = nil
	end

	# Saves the session, and set a cookie if "store_in_cookie" attribute is
	# setted. If "clear" method of the session is called, the session is deleted.
	def save_session( session )
		if session.clear? then
			if session.session_id == @session_id then
				_clear_session_cookie
			end
		elsif (@session_id == session.session_id) and @store_in_cookie then
			cookie = CKCookie.new(@session_key, @session_id)
			if @session_cookie_expires then
				cookie.expires = Time.new + @session_cookie_expires
			end
			@response.add_cookie cookie
		end

		session_store.save session
	end

	private

	def _clear_session_cookie
		@session_id = nil

		if @store_in_cookie then
			cookie         = CKCookie.new @session_key
			cookie.expires = Time.new - 60
			@response.add_cookie cookie
		end
	end

	public

	# Returns the session object. If the session isn't existed, returns a new session.
	def session
		@session ||= create_session
	end

	# Returns true if the session is existed.
	def session?
		if @session and @session_key and @session_id then
			true
		else
			false
		end
	end

	# Returns true if the locale is master locale.
	def master_locale?
		if @locale == @master_locale then
			true
		else
			false
		end
	end

	# Runs the application.
	# This method calls hook methods "pre_run" and "pre_respond".
	def run( request = nil, response = nil )
		@adapter = create_adapter
		@adapter.run( request, response ) do | ckrequest, ckresponse |
			@request  = ckrequest
			@response = ckresponse

			begin
				# trap generic components
				_ready_request_response
				_ready_session
				pre_action # hook

				unless ckresponse.redirect?
					_event_loop ckresponse
				end
			rescue Exception => e
				@error = e

				# trap custom components' error
				begin
					component = handle_error e
					ckresponse.content = component.to_s
				rescue Exception => e
					component = _default_error_page e
					ckresponse.content = component.to_s
				end
			end

			pre_respond # hook
		end
	end

	# Hook method called before generating specified component.
	# When calling the method, objects of request and session are already setted.
	def pre_action; end

	# Hook method called just before sending the response to browser.
	# When calling the method, a web page converted to HTML code
	# has already setted to the response.
	def pre_respond; end

	private

	def _ready_request_response
		@baseurl       = @request.script_name              unless @baseurl
		@document_root = @request.headers['DOCUMENT_ROOT'] unless @document_root
		@web_server_resources ||= @document_root
		@resource_manager       = CKResourceManager.new self
		unless @request.languages.empty? then
			@locale = @request.languages.first
		end

		if @element_id then
			@target = @element_id.component
		elsif @request['element_id'] then
			@element_id = CKElementID.new @request['element_id'].to_s
			@target     = @element_id.component
		else
			@target = @main.dup
		end
	end

	def _ready_session
		@session_store = CKSessionStore.new self

		# get the session ID.
		cookie    = @request.cookie(@session_key)
		id_cookie = cookie.value if cookie
		id_query  = @request[@session_key]
		unless CKSession.session_id? id_cookie then id_cookie   = nil       end
		unless CKSession.session_id? id_query  then id_query    = nil       end
		if     @store_in_cookie and id_cookie then @session_id = id_cookie
		elsif                       id_query  then @session_id = id_query  end

		# restore the session.
		if @session_id then
			@session = restore_session @session_id
		else
			@session_id = nil
		end

		# ready a new session in automatic session management
		if @manage_session and @session_id.nil? and @session.nil? then
			@session    = create_session
			@session_id = @session.session_id
			save_session @session

			if @store_in_url and (@store_in_cookie == false) then
				@response.set_redirect url(@element_id, @request.form_values)
			end
		end

		# check 1 - session ID and session
		if @session_id and @session.nil? then
			_session_timeout
		end

		# check 2 - authorization and timeout
		if @session then
			if (@auth_by_user_agent and \
			   (not (@session.user_agent? @request.user_agent))) or \
			   (@auth_by_remote_addr and \
			   (not (@session.remote_addr? @request.remote_addr))) then
				raise SessionAuthorizationError, 'Your session is not authorized.'
			elsif (@session.timeout? @timeout) then
				_session_timeout
			end
		end
	end

	def _session_timeout
		if @session then
			clear_session @session
		elsif @session_id then
			_clear_session_cookie
		end

		raise SessionTimeoutError, 'Your session has timed out.'
	end

	def _event_loop( response )
		result = page @target
		result.is_top = true

		begin
			component = result
			result    = component.run

			if @manage_session and @session then
				save_session @session
			end
		end while result.is_a? CKComponent

		# Recreates a main component if the component includes CKPartsMaker.
		if component.is_a? CKPartsMaker
			@target   = component.substitute_page || @main
			component = page @target
			component.run
		end

		# for displaying images
		element = nil
		if @element_id then
			if defs = component.definitions[@element_id.element] then
				element = defs['element']
			end
		end

		if (CKByteData === result) and (element == 'CKImage') then
			raw_mime = component.definitions[@element_id.element]['mime']
			unless raw_mime then
				mime = result.content_type
			else
				mime = component.parse_ckd_value raw_mime
			end
			response.headers['Content-Type'] = mime
			response.content = result.to_s
		else
			response.content = component.to_s
		end
	end

	public

	# Creates an adapter object.
	def create_adapter
		@interface.new
	end

	# Handles every errors and return an error page component.
	def handle_error( error )
		error_page = nil
		if ( error.class == SessionTimeoutError )       or \
		   ( error.class == SessionAuthorizationError ) then
			error_page = handle_session_error error
		end
		unless error_page.is_a? CKComponent then
			error_page       = page @error_page
			error_page.error = error
		end
		error_page
	end

	# Hook method to handle session errors.
	def handle_session_error( error ); end

	private

	# Return a default error page component.
	def _default_error_page( error )
		error_page       = page 'CKErrorPage'
		error_page.error = error
		error_page
	end

	public

	# Creates a specified page component.
	def page( name )
		CKElement.instance( name, self, nil, nil, nil )
	end

	# Creates a URL.
	def url( id = nil, query = nil, secure = false, direct = false )
		protocol = nil
		string   = '?'

		if @secure then
			protocol = 'https://'
		end

		string << "element_id=#{id};"  if id
		string << _query_string(query) if query
		string << _query_for_session   if direct != true
		string.chop!                   if string =~ /&$/

		url = ''
		if protocol then
			url << protocol
			url << @request.server_name if @request.server_name
		end
		url << self.baseurl if self.baseurl
		url << string
	end

	private

	def _query_string( hash )
		str = ''
		hash.each do | key, value |
			if key == @session_key then
				next
			elsif value.is_a? Array then
				value.each do | _value |
					str << "#{key}=#{CKUtilities.escape_url(_value)};"
				end
			else
				str << "#{key}=#{CKUtilities.escape_url(value)};"
			end
		end
		str
	end

	def _query_for_session
		if @store_in_url and @session_id and @session_key then
			"#@session_key=#@session_id;"
		else
			''
		end		
	end

end


# CKElementID is a class for element IDs.
# The object is used to specify an element when substituting form values and
# running an action.
#
# Element ID has 3 elements, these are "component, repetitions, element".
# "repetitions" are definition names enclosed the element with CKRepetition
# and indexes in the CKRepetition elements.
#
# The each elements are splitted by '.'. Definition names and indexes in
# repetitions are splitted by ':'.
# 
# * MainPage component
#  MainPage
#
# * String element of MainPage component
#  MainPage.String
#
# * String element in third repeat for Repetition element as 
#   CKRepetition of MainPage component 
#  MainPage.Repetition:3.String
#
# Component is required. If you specify elements in CKRepetition, 
# repetitions are required.
class CKElementID
	# Name of the component.
	attr_reader :component

	# Array of the repetitions.
	attr_accessor :repetitions

	# Name of the element.
	attr_accessor :element

	def initialize( string = nil )
		@repetitions = []
		if string then
			tokens     = string.split '.'
			@component = tokens.shift
			@element   = tokens.pop

			# parses repetitions
			token = repetition = nil
			unless tokens.empty?
				tokens.each do | token |
					repetition = token.split ':'
					@repetitions << [ repetition.first, repetition.last.to_i ]
				end
			end
		end
	end

	# Returns true if the ID is in repetitions.
	def repetitions?
		if @repetitions.empty? then
			false
		else
			true
		end
	end

	# Executes the block for every definition names and repetition indexes.
	def each
		definition = index = nil
		@repetitions.each do | name, index |
			yield name, index
		end
	end

	# Returns true if a component of the ID and specified are equal.
	def component?( name )
		if @component == name then
			true
		else
			false
		end
	end

	def component=( name )
		@component = name.split('::').last
	end

	# Returns the object as a string.
	def to_s
		str = @component + '.'
		if repetitions? then
			@repetitions.each do | name, index |
				str << "#{name}:#{index}."
			end
		end
		str << @element if @element
		str.chop! if str =~ /\.$/
		str
	end
end


# CKKeyValueCoding provides methods to access instance variables of the object.
# The methods try accessing instance variables by using accessor method.
# If it is failure, its try accessing directly.
#
# If a class method "access_instance_variables?" defines in the class and
# the method returns "true", the direct access is success. Or failure.
module CKKeyValueCoding
	class UnknownKeyError < CKError; end

	# Retrieves value of the instance variable with method chain.
	def retrieve_value( key )
		keypath = key.split '.'
		object  = self
		keypath.each do |path|
			if object.respond_to? path then
				object = object.__send__ path
			elsif _directly?(object) == true then
				object = object.instance_eval "@#{path}"
			else
				_raise_error(object, path)
			end
		end
		object
	end

	# Sets value for the instance variable with method chain.
	def take_value( key, value )
		keypath = key.split '.'
		object  = self
		keypath.each_with_index do |path, index|
			writer = "#{path}="

			if (index + 1) == keypath.size then
				# takes value for key
				if object.respond_to? writer then
					object.__send__(writer, value)
				elsif _directly?(object) == true then
					object.instance_eval "@#{path}=value"
				else
					_raise_error(object, writer)
				end
			else
				# get and set value for the object
				if object.respond_to? path then
					object = object.__send__ path
				elsif _directly?(object) == true then
					object = object.instance_eval "@#{path}"
				else
					_raise_error(object, path)
				end
			end
		end
	end

	alias []  retrieve_value
	alias []= take_value

	private

	def _directly?( object )
		if object.class.respond_to? 'access_instance_variables?' then
			object.class.access_instance_variables?
		else
			false
		end
	end

	def _raise_error( object, key )
		msg =  "This \"#{object.class}\" object does not have a method "
		msg << "\"#{key}\", nor an instance variable \"@#{key.sub(/=$/,'')}\"."
		raise UnknownKeyError, msg
	end

end


