#ifndef __BFT_MEM_USAGE_H__
#define __BFT_MEM_USAGE_H__

/*============================================================================
 * Base memory usage information (System and Library dependent)
 *============================================================================*/

/*
  This file is part of the "Base Functions and Types" library, intended to
  simplify and enhance portability, memory and I/O use for scientific codes.

  Copyright (C) 2004-2008  EDF

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*----------------------------------------------------------------------------*/

/*
 * Obtain definitions such as that of size_t through stddef.h (C99 standard)
 * if available (preferred method), or through stdlib.h (which defines
 * malloc() and family and so must define size_t some way) otherwise.
 */

#if defined(__STDC_VERSION__)
#  if (__STDC_VERSION__ == 199901L)
#    include <stddef.h>
#  else
#    include <stdlib.h>
#  endif
#else
#  include <stdlib.h>
#endif

/* BFT headers */

#include "bft_config.h"

/*-----------------------------------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#if 0
} /* Fake brace to force Emacs auto-indentation back to column 0 */
#endif
#endif /* __cplusplus */

/*============================================================================
 * Public types
 *============================================================================*/

/*
 * Memory usage tracking mode options
 */

#define BFT_MEM_USAGE_TRACK_PR_SIZE    (1 << 0)
#define BFT_MEM_USAGE_TRACK_ALLOC_SIZE (1 << 1)
#define BFT_MEM_USAGE_TRACK_HEAP_SIZE  (1 << 2)

/*============================================================================
 * Public function prototypes
 *============================================================================*/

/*
 * Initialize memory usage count depending on system.
 *
 * This functions checks if it has already been called, so
 * it is safe to call more than once (though it is not
 * thread-safe). Only the first call is effective.
 */

void
bft_mem_usage_init(void);

/*
 * End memory usage count depending on system.
 */

void
bft_mem_usage_end(void);

/*
 * Indicates if bft_mem_usage_...() functions are initialized.
 *
 * returns:
 *   1 if bft_mem_usage_init has been called, 0 otherwise.
 */

int
bft_mem_usage_initialized(void);

/*
 * Change memory usage tracking options for certain systems.
 *
 * On certain systems, such as GNU/Linux systems, we use hooks to the libc's
 * memory allocation functions to automatically update the maximum memory
 * allocation info (unless the BFT library was configured with malloc hooks
 * disabled or we prelink with another memory allocation library). In that
 * case, we may choose to selectively deactivate tracking with functions of
 * the bft_mem_usage_pr_size() family if we do not intend to use the
 * corresponding function of the bft_mem_usage_max_pr_size() family. This is
 * done by calling this function with the corresponding options set to 0. By
 * default (i.e. if this function is not called), full tracking is activated.
 *
 * If a tracking mode is known to be unnecessary, it is deactivated regardless
 * of the argument; for example, tracking is necessary for older Linux
 * kernels with the /proc filesystem, but not on more recent 2.6.x kernels
 * in which the "/proc/pid/status" file contains a "VmPeak" record.
 *
 * This function may be called at any time, whether before or after
 * bft_mem_usage_init(). It is not thread-safe.
 *
 * parameters:
 *   mode: <-- mask indicating activation or deactivation of tracking
 *             with the different available methods.
 */

void
bft_mem_usage_set_options(const int mode);

/*
 * Return memory usage tracking options for certain systems.
 *
 * Tracking is only necessary on certain systems ;
 * see bft_mem_usage_set_options() for explanations.
 *
 * This function may be called at any time, whether before or after
 * bft_mem_usage_init().
 *
 * parameters:
 *   mode: --> mask indicating activation or deactivation of tracking
 *             with the different available methods.
 */

void
bft_mem_usage_get_options(int  *const mode);

/*
 * Return current process memory use (in kB) depending on OS.
 */

size_t
bft_mem_usage_pr_size(void);

/*
 * Return current measurable dynamic memory use (in kB).
 *
 * If the information is not measurable in a dependable fashion
 * or bft_mem_usage_init() has not been called, 0 is returned.
 */

size_t
bft_mem_usage_alloc_size(void);

/*
 * Return current heap size (in kB).
 *
 * The sbrk() function is used to know the size of the heap. This is not
 * of any use to guess at allocated memory when some part of the memory may
 * be allocated with mmap(), such as with glibc on Linux.
 *
 * If the sbrk() function call is not available or bft_mem_usage_init()
 * has not been called, 0 is returned.
 */

size_t
bft_mem_usage_heap_size(void);

/*
 * Return maximum process memory use (in kB) depending on OS.
 *
 * The returned value is the maximum returned by bft_mem_usage_pr_size()
 * during the program's lifetime. With memory allocations which return
 * memory to the system (such as the GNU glibc on Linux systems),
 * this value will be correct only if allocation is tracked. This should
 * be the case if malloc hooks are used with the glibc allocation
 * functions (BFT library's default configuration/installation option),
 * but may give results lower than the true maximum in other cases.
 */

size_t
bft_mem_usage_max_pr_size(void);

/*
 * Return maximum measurable dynamic memory use (in kB).
 *
 * The returned value is the maximum returned by bft_mem_usage_alloc_size()
 * during the program's lifetime. With memory allocations which return
 * memory to the system (such as the GNU glibc on Linux systems),
 * this value will be correct only if allocation is tracked. This should
 * be the case if malloc hooks are used with the glibc allocation
 * functions (BFT library's default configuration/installation option),
 * but may give results lower than the true maximum in other cases.
 */

size_t
bft_mem_usage_max_alloc_size(void);

/*
 * Return maximum heap size (in kB).
 *
 * The returned value is the maximum returned by bft_mem_usage_heap_size()
 * during the program's lifetime. With memory allocations which return
 * memory to the system (such as the GNU glibc on Linux systems),
 * this value will be correct only if allocation is tracked. This should
 * be the case if malloc hooks are used with the glibc allocation
 * functions (BFT library's default configuration/installation option),
 * but may give results lower than the true maximum in other cases.
 */

size_t
bft_mem_usage_max_heap_size(void);

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __BFT_MEM_USAGE_H__ */
