/***************************************************************************
 $RCSfile: rdhkeyfile.c,v $
                             -------------------
    cvs         : $Id: rdhkeyfile.c,v 1.25 2005/02/15 10:11:40 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "rdhkeyfile_p.h"

#include "aqhbci_l.h"
#include <aqhbci/mediumheader.h>

#include <gwenhywfar/debug.h>
#include <gwenhywfar/misc.h>
#include <gwenhywfar/padd.h>
#include <gwenhywfar/md.h>
#include <gwenhywfar/waitcallback.h>

#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#ifdef OS_WIN32
# define ftruncate chsize
#endif



GWEN_INHERIT(AH_MEDIUM, AH_MEDIUMRDHKEYFILE);


AH_MEDIUMRDHKEYFILE *AH_MediumRDHKeyFile_Data_new(){
  AH_MEDIUMRDHKEYFILE *mrdh;

  GWEN_NEW_OBJECT(AH_MEDIUMRDHKEYFILE, mrdh);
  mrdh->localSignSeq=1;
  return mrdh;
}



void AH_MediumRDHKeyFile_Data_free(AH_MEDIUMRDHKEYFILE *mrdh){
  if (mrdh) {
    /* overwrite password */
    memset(mrdh->password, 0, sizeof(mrdh->password));
    free(mrdh->bankId);
    free(mrdh->userId);
    free(mrdh->systemId);
    GWEN_CryptKey_free(mrdh->localSignKey);
    GWEN_CryptKey_free(mrdh->localCryptKey);
    GWEN_CryptKey_free(mrdh->remoteSignKey);
    GWEN_CryptKey_free(mrdh->remoteCryptKey);
    GWEN_CryptKey_free(mrdh->localSignKeyBak);
    GWEN_CryptKey_free(mrdh->localCryptKeyBak);

    GWEN_FREE_OBJECT(mrdh);
  }
}




AH_MEDIUM *AH_MediumRDHKeyFile_new(AH_HBCI *hbci,
                                   const char *mediumName) {
  AH_MEDIUM *m;
  AH_MEDIUMRDHKEYFILE *mrdh;

  m=AH_MediumRDH_new(hbci,
                     AH_MEDIUMRDHKEYFILE_NAME,
                     mediumName);
  mrdh=AH_MediumRDHKeyFile_Data_new();
  GWEN_INHERIT_SETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE,
                       m, mrdh,
                       AH_MediumRDHKeyFile_FreeData);

  AH_Medium_SetDeviceType(m, AH_MediumDeviceFile);

  /* set implemented functions for Medium */
  AH_Medium_SetChangePinFn(m, AH_MediumRDHKeyFile_ChangePin);
  AH_Medium_SetMountFn(m, AH_MediumRDHKeyFile_Mount);
  AH_Medium_SetCreateFn(m, AH_MediumRDHKeyFile_Create);
  AH_Medium_SetUnmountFn(m, AH_MediumRDHKeyFile_Unmount);
  AH_Medium_SetSignFn(m, AH_MediumRDHKeyFile_Sign);
  AH_Medium_SetGetNextSignSeqFn(m, AH_MediumRDHKeyFile_GetNextSignSeq);
  AH_Medium_SetSetLocalSignSeqFn(m, AH_MediumRDHKeyFile_SetLocalSignSeq);
  AH_Medium_SetVerifyFn(m, AH_MediumRDHKeyFile_Verify);
  AH_Medium_SetEncryptKeyFn(m, AH_MediumRDHKeyFile_EncryptKey);
  AH_Medium_SetDecryptKeyFn(m, AH_MediumRDHKeyFile_DecryptKey);
  AH_Medium_SetGenerateMsgKeyFn(m, AH_MediumRDHKeyFile_GenerateMsgKey);
  AH_Medium_SetSelectContextFn(m, AH_MediumRDHKeyFile_SelectContext);
  AH_Medium_SetCreateContextFn(m, AH_MediumRDHKeyFile_CreateContext);
  AH_Medium_SetRemoveContextFn(m, AH_MediumRDHKeyFile_RemoveContext);
  AH_Medium_SetReadContextFn(m, AH_MediumRDHKeyFile_ReadContext);
  AH_Medium_SetWriteContextFn(m, AH_MediumRDHKeyFile_WriteContext);

  /* set implemented functions for MediumRDH */
  AH_MediumRDH_SetCreateKeysFn(m, AH_MediumRDHKeyFile_CreateKeys);
  AH_MediumRDH_SetActivateKeysFn(m,
                                      AH_MediumRDHKeyFile_ActivateKeys);

  /* not implemented are fromDb and toDb, they are not needed for now */

  /* set implemented functions for MediumRDH */
  AH_MediumRDH_SetGetLocalPubSignKeyFn
    (m, AH_MediumRDHKeyFile_GetLocalPubSignKey);
  AH_MediumRDH_SetGetLocalPubCryptKeyFn
    (m, AH_MediumRDHKeyFile_GetLocalPubCryptKey);
  AH_MediumRDH_SetGetLocalTmpSignKeyFn
    (m, AH_MediumRDHKeyFile_GetLocalTmpSignKey);
  AH_MediumRDH_SetGetLocalTmpCryptKeyFn
    (m, AH_MediumRDHKeyFile_GetLocalTmpCryptKey);

  AH_MediumRDH_SetSetPubSignKeyFn
    (m, AH_MediumRDHKeyFile_SetPubSignKey);
  AH_MediumRDH_SetGetPubSignKeyFn
    (m, AH_MediumRDHKeyFile_GetPubSignKey);
  AH_MediumRDH_SetSetPubCryptKeyFn
    (m, AH_MediumRDHKeyFile_SetPubCryptKey);
  AH_MediumRDH_SetGetPubCryptKeyFn
    (m, AH_MediumRDHKeyFile_GetPubCryptKey);

  return m;
}



void AH_MediumRDHKeyFile_FreeData(void *bp, void *p){
  AH_MEDIUM *m;
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(bp);
  assert(p);
  m=(AH_MEDIUM*)bp;
  mrdh=(AH_MEDIUMRDHKEYFILE*)p;
  AH_MediumRDHKeyFile_Data_free(mrdh);
}



GWEN_CRYPTKEY *AH_MediumRDHKeyFile_GetLocalPubSignKey(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_ERRORCODE err;
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!mrdh->localSignKey) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  db=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(mrdh->localSignKey, db, 1);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_DB_Group_free(db);
    return 0;
  }

  key=GWEN_CryptKey_FromDb(db);
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key from previous export");
    GWEN_DB_Group_free(db);
    return 0;
  }
  GWEN_DB_Group_free(db);

  return key;
}



GWEN_CRYPTKEY *AH_MediumRDHKeyFile_GetLocalPubCryptKey(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_ERRORCODE err;
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!mrdh->localCryptKey) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  db=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(mrdh->localCryptKey, db, 1);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return 0;
  }
  key=GWEN_CryptKey_FromDb(db);
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key from previous export");
    return 0;
  }

  return key;
}



GWEN_CRYPTKEY *AH_MediumRDHKeyFile_GetLocalTmpSignKey(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_ERRORCODE err;
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!mrdh->localSignKeyBak) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  db=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(mrdh->localSignKeyBak, db, 1);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return 0;
  }

  key=GWEN_CryptKey_FromDb(db);
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key from previous export");
    return 0;
  }

  return key;
}



GWEN_CRYPTKEY *AH_MediumRDHKeyFile_GetLocalTmpCryptKey(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_ERRORCODE err;
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!mrdh->localCryptKeyBak) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  db=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(mrdh->localCryptKeyBak, db, 1);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return 0;
  }
  key=GWEN_CryptKey_FromDb(db);
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key from previous export");
    return 0;
  }

  return key;
}



GWEN_CRYPTKEY *AH_MediumRDHKeyFile_GetPubSignKey(AH_MEDIUM *m){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (mrdh->remoteSignKey)
    return GWEN_CryptKey_dup(mrdh->remoteSignKey);
  else
    return 0;
}



int AH_MediumRDHKeyFile_SetPubSignKey(AH_MEDIUM *m,
                                      const GWEN_CRYPTKEY *key){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  assert(key);
  GWEN_CryptKey_free(mrdh->remoteSignKey);
  mrdh->remoteSignKey=GWEN_CryptKey_dup(key);
  return 0;
}



GWEN_CRYPTKEY *AH_MediumRDHKeyFile_GetPubCryptKey(AH_MEDIUM *m){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (mrdh->remoteCryptKey)
    return GWEN_CryptKey_dup(mrdh->remoteCryptKey);
  else
    return 0;
}



int AH_MediumRDHKeyFile_SetPubCryptKey(AH_MEDIUM *m,
                                       const GWEN_CRYPTKEY *key){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  assert(key);
  GWEN_CryptKey_free(mrdh->remoteCryptKey);
  mrdh->remoteCryptKey=GWEN_CryptKey_dup(key);
  return 0;
}



int AH_MediumRDHKeyFile_CreateKeys(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key1;
  GWEN_CRYPTKEY *key2;
  GWEN_ERRORCODE err;
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!mrdh->selected) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No context selected");
    return -1;
  }

  /* create and generate local sign key */
  key1=GWEN_CryptKey_Factory("RSA");
  if (!key1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create an RSA key");
    return -1;
  }
  err=GWEN_CryptKey_Generate(key1, AH_DEFAULT_KEYLEN);
  if (!GWEN_Error_IsOk(err)){
    GWEN_CryptKey_free(key1);
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return -1;
  }
  GWEN_CryptKey_SetKeyName(key1, "S");
  GWEN_CryptKey_SetOwner(key1, mrdh->userId);

  /* create and generate local crypt key */
  key2=GWEN_CryptKey_Factory("RSA");
  if (!key2) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create an RSA key");
    GWEN_CryptKey_free(key1);
    return -1;
  }
  err=GWEN_CryptKey_Generate(key2, AH_DEFAULT_KEYLEN);
  if (!GWEN_Error_IsOk(err)){
    GWEN_CryptKey_free(key1);
    GWEN_CryptKey_free(key2);
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return -1;
  }
  GWEN_CryptKey_SetKeyName(key2, "V");
  GWEN_CryptKey_SetOwner(key2, mrdh->userId);

  /* store keys as backup keys (can be activated later) */
  GWEN_CryptKey_free(mrdh->localSignKeyBak);
  mrdh->localSignKeyBak=key1;
  GWEN_CryptKey_free(mrdh->localCryptKeyBak);
  mrdh->localCryptKeyBak=key2;

  return 0;
}



int AH_MediumRDHKeyFile_ActivateKeys(AH_MEDIUM *m){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!mrdh->localSignKeyBak ||
      !mrdh->localCryptKeyBak) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No keys created");
    return -1;
  }
  GWEN_CryptKey_free(mrdh->localSignKey);
  mrdh->localSignKey=mrdh->localSignKeyBak;
  mrdh->localSignKeyBak=0;
  GWEN_CryptKey_free(mrdh->localCryptKey);
  mrdh->localCryptKey=mrdh->localCryptKeyBak;
  mrdh->localCryptKeyBak=0;
  return 0;
}



int AH_MediumRDHKeyFile__OpenFile(AH_MEDIUM *m, int wr){
#ifndef OS_WIN32
  struct flock fl;
#endif
  int fd;
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (wr) {
    /* write file */
    fd=open(AH_Medium_GetMediumName(m),
            O_RDWR|O_CREAT
#ifdef OS_WIN32
            | O_BINARY
#endif
            ,
            S_IRUSR|S_IWUSR);
  }
  else {
    fd=open(AH_Medium_GetMediumName(m),
	    O_RDONLY
#ifdef OS_WIN32
	    | O_BINARY
#endif
	   );
  }

  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
	      "open(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    return -1;
  }

#ifndef OS_WIN32
  /* lock file for reading or writing */
  memset(&fl, 0, sizeof(fl));
  fl.l_type=wr?F_WRLCK:F_RDLCK;
  fl.l_whence=SEEK_SET;
  fl.l_start=0;
  fl.l_len=0;
  if (fcntl(fd, F_SETLKW, &fl)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "fcntl(%s, F_SETLKW): %s",
	      AH_Medium_GetMediumName(m), strerror(errno));
    close(fd);
    return -1;
  }
#endif

  return fd;
}



int AH_MediumRDHKeyFile__CloseFile(AH_MEDIUM *m, int fd){
#ifndef OS_WIN32
  struct flock fl;
#endif
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Keyfile \"%s\"not open",
	      AH_Medium_GetMediumName(m));
    return -1;
  }

#ifndef OS_WIN32
  /* unlock file */
  memset(&fl, 0, sizeof(fl));
  fl.l_type=F_UNLCK;
  fl.l_whence=SEEK_SET;
  fl.l_start=0;
  fl.l_len=0;
  if (fcntl(fd, F_SETLK, &fl)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "fcntl(%s, F_SETLK): %s",
	      AH_Medium_GetMediumName(m), strerror(errno));
    close(fd);
    return -1;
  }
#endif

  if (close(fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "close(%s): %s",
	      AH_Medium_GetMediumName(m), strerror(errno));
    return -1;
  }

  return 0;
}



int AH_MediumRDHKeyFile__ReadFile(AH_MEDIUM *m, int fd){
  AH_MEDIUMRDHKEYFILE *mrdh;
  int rv;
  GWEN_BUFFER *fbuf;
  GWEN_BUFFER *rawbuf;
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_DB_NODE *gr;
  GWEN_ERRORCODE err;
  GWEN_BUFFEREDIO *bio;
  struct stat st;
  char password[64];
  unsigned int i;
  const char *p;
  AH_MEDIUMHEADER *mh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* some file checks */
  if (fstat(fd, &st)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "stat(%s): %s",
              AH_Medium_GetMediumName(m),
              strerror(errno));
    return -1;
  }
  if (!S_ISREG(st.st_mode)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "\"%s\" is not a regular file",
              AH_Medium_GetMediumName(m));
    return -1;
  }
  if (st.st_mode & 0077) {
    DBG_WARN(AQHBCI_LOGDOMAIN,
             "WARNING: Your keyfile \"%s\" is group/world accessible!\n"
             "Please consider a more strict permission policy on "
             "that file,\n"
             "nobody but you should have access rights for it!",
             AH_Medium_GetMediumName(m));
  }
  mrdh->mtime=st.st_mtime;
  mrdh->ctime=st.st_ctime;

  fbuf=GWEN_Buffer_new(0, 256, 0, 1);

  if (lseek(fd, 0, SEEK_SET)==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "lseek(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    GWEN_Buffer_free(fbuf);
    return -1;
  }

  while (1) {
    char buffer[256];

    rv=read(fd, buffer, sizeof(buffer));
    if (rv==-1) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "read(%s): %s",
                AH_Medium_GetMediumName(m),
                strerror(errno));
      GWEN_Buffer_free(fbuf);
      return -1;
    }
    else if (rv==0)
      break;
    GWEN_Buffer_AppendBytes(fbuf, buffer, rv);
  } /* while */

  mh=AH_MediumHeader_FromBuffer(fbuf);
  if (!mh) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No medium header");
    GWEN_Buffer_free(fbuf);
    return -1;
  }
  else {
    if (strcasecmp(AH_MediumHeader_GetMediumType(mh),
		   AH_MEDIUMRDHKEYFILE_NAME)) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "This file is not of type \"%s\" (it is \"%s\")",
		AH_MEDIUMRDHKEYFILE_NAME,
		AH_MediumHeader_GetMediumType(mh));
      AH_MediumHeader_free(mh);
      GWEN_Buffer_free(fbuf);
      return -1;
    }

    /* check for keyfile versions */
    if (AH_MediumHeader_GetMajorVersion(mh)!=AH_MEDIUMRDHKEYFILE_VMAJOR) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
		"Major versions of keyfile do not match (%d!=%d)",
		rv, AH_MEDIUMRDHKEYFILE_VMAJOR);
      GWEN_WaitCallback_Log(0,
                            "Basically this file type is supported.\n"
                            "However, the major versions do not match,\n"
                            "so this particular version is not supported");
      AH_MediumHeader_free(mh);
      GWEN_Buffer_free(fbuf);
      return -1;
    }
    if (AH_MediumHeader_GetMinorVersion(mh)>AH_MEDIUMRDHKEYFILE_VMINOR) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
		"The keyfile has been created with a newer "
		"library version.\n"
		"Using this file with the current version "
		"would degrade it.\n"
		"So for safety reasons I refuse to work "
		"with this file (%d>%d).",
		AH_MediumHeader_GetMinorVersion(mh),
		AH_MEDIUMRDHKEYFILE_VMINOR);
      GWEN_WaitCallback_Log(0,
                            "Basically this file type is supported.\n"
                            "However, this file has been created with a "
                            "newer library version.\n"
                            "Using this file with the current version "
                            "would degrade it.\n"
                            "So for safety reasons I refuse to work "
                            "with this file.");
      AH_MediumHeader_free(mh);
      GWEN_Buffer_free(fbuf);
      return -1;
    }
    AH_MediumHeader_free(mh);
    /* cut off the header */
    GWEN_Buffer_Crop(fbuf, GWEN_Buffer_GetPos(fbuf),
		     GWEN_Buffer_GetUsedBytes(fbuf)-GWEN_Buffer_GetPos(fbuf));
    GWEN_Buffer_Rewind(fbuf);
  }

  if (mrdh->passWordIsSet==0) {
    /* create key from password */
    password[0]=0;
    for (i=0; i<sizeof(mrdh->password); i++)
      mrdh->password[i]=0;
    if (AH_Medium_InputPin(m,
                           password,
                           AH_MEDIUMRDHKEYFILE_PINMINLENGTH,
                           sizeof(password),
                           0)) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not get PIN");
      GWEN_Buffer_free(fbuf);
      return -1;
    }
    if (strlen(password)<AH_MEDIUMRDHKEYFILE_PINMINLENGTH) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Your program returned a shorter PIN than instructed!");
      GWEN_Buffer_free(fbuf);
      return -1;
    }

    if (GWEN_CryptKey_FromPassword(password,
				   mrdh->password, sizeof(mrdh->password))) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
      GWEN_Buffer_free(fbuf);
      return -1;
    }
    /* overwrite password as soon as possible */
    memset(password, 0, sizeof(password));
    mrdh->passWordIsSet=1;
  }

  key=GWEN_CryptKey_Factory("DES");
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create DES key");
    GWEN_Buffer_free(fbuf);
    return -1;
  }

  err=GWEN_CryptKey_SetData(key, mrdh->password, sizeof(mrdh->password));
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_CryptKey_free(key);
    GWEN_Buffer_free(fbuf);
    return -1;
  }

  /* decrypt file */
  rawbuf=GWEN_Buffer_new(0, 256, 0, 1);
  err=GWEN_CryptKey_Decrypt(key, fbuf, rawbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_Buffer_free(fbuf);
    GWEN_Buffer_free(rawbuf);
    GWEN_CryptKey_free(key);
    return -1;
  }

  /* unpadd raw data */
  if (GWEN_Padd_UnpaddWithANSIX9_23(rawbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not unpadd keyfile");
    GWEN_Buffer_free(fbuf);
    GWEN_Buffer_free(rawbuf);
    GWEN_CryptKey_free(key);
    return -1;
  }
  GWEN_Buffer_free(fbuf);
  GWEN_CryptKey_free(key);

  /* parse raw data */
  GWEN_Buffer_Rewind(rawbuf);
  bio=GWEN_BufferedIO_Buffer_new(rawbuf);
  GWEN_BufferedIO_SetReadBuffer(bio, 0, 256);
  db=GWEN_DB_Group_new("keyfile");
  if (GWEN_DB_ReadFromStream(db, bio,
                             GWEN_DB_FLAGS_DEFAULT |
                             GWEN_PATH_FLAGS_CREATE_GROUP)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Error parsing key file");
    GWEN_BufferedIO_Abandon(bio);
    GWEN_BufferedIO_free(bio);
    GWEN_DB_Group_free(db);
    return -1;
  }
  GWEN_BufferedIO_Close(bio);
  GWEN_BufferedIO_free(bio);

  mrdh->country=GWEN_DB_GetIntValue(db, "country", 0, 280);

  free(mrdh->bankId);
  p=GWEN_DB_GetCharValue(db, "bankId", 0, 0);
  if (p) mrdh->bankId=strdup(p);
  else mrdh->bankId=0;

  free(mrdh->userId);
  p=GWEN_DB_GetCharValue(db, "userId", 0, 0);
  if (p) mrdh->userId=strdup(p);
  else mrdh->userId=0;

  free(mrdh->systemId);
  p=GWEN_DB_GetCharValue(db, "systemId", 0, 0);
  if (p) {
    DBG_NOTICE(AQHBCI_LOGDOMAIN, "SystemId is: %s", p);
    mrdh->systemId=strdup(p);
    AH_Medium_SetSecurityId(m, mrdh->systemId, strlen(mrdh->systemId));
  }
  else mrdh->systemId=0;

  p=GWEN_DB_GetCharValue(db, "serverAddr", 0, 0);
  if (p)
    AH_Medium_SetPeerAddr(m, p);
  AH_Medium_SetPeerPort(m, GWEN_DB_GetIntValue(db, "serverPort", 0, 3000));

  mrdh->localSignSeq=GWEN_DB_GetIntValue(db, "localSignSeq", 0, 1);
  mrdh->remoteSignSeq=GWEN_DB_GetIntValue(db, "remoteSignSeq", 0, 0);

  /* read keys from DB */
  GWEN_CryptKey_free(mrdh->localSignKey);
  mrdh->localSignKey=0;
  GWEN_CryptKey_free(mrdh->localCryptKey);
  mrdh->localCryptKey=0;
  GWEN_CryptKey_free(mrdh->remoteSignKey);
  mrdh->remoteSignKey=0;
  GWEN_CryptKey_free(mrdh->remoteCryptKey);
  mrdh->remoteCryptKey=0;
  GWEN_CryptKey_free(mrdh->localSignKeyBak);
  mrdh->localSignKeyBak=0;
  GWEN_CryptKey_free(mrdh->localCryptKeyBak);
  mrdh->localCryptKeyBak=0;

  gr=GWEN_DB_GetGroup(db,
                      GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                      "localSignKey");
  if (gr) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Loading localSignKey");
    mrdh->localSignKey=GWEN_CryptKey_FromDb(gr);
    if (!mrdh->localSignKey) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Error creating key from db");
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  gr=GWEN_DB_GetGroup(db,
                      GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                      "localCryptKey");
  if (gr) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Loading localCryptKey");
    mrdh->localCryptKey=GWEN_CryptKey_FromDb(gr);
    if (!mrdh->localCryptKey) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Error creating key from db");
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  gr=GWEN_DB_GetGroup(db,
                      GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                      "remoteSignKey");
  if (gr) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Loading remoteSignKey");
    mrdh->remoteSignKey=GWEN_CryptKey_FromDb(gr);
    if (!mrdh->remoteSignKey) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Error creating key from db");
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  gr=GWEN_DB_GetGroup(db,
                      GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                      "remoteCryptKey");
  if (gr) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Loading remoteCryptKey");
    mrdh->remoteCryptKey=GWEN_CryptKey_FromDb(gr);
    if (!mrdh->remoteCryptKey) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Error creating key from db");
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  gr=GWEN_DB_GetGroup(db,
                      GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                      "localSignKeyBak");
  if (gr) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Loading localSignKeyBak");
    GWEN_CryptKey_free(mrdh->localSignKeyBak);
    mrdh->localSignKeyBak=GWEN_CryptKey_FromDb(gr);
    if (!mrdh->localSignKeyBak) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Error creating key from db");
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  gr=GWEN_DB_GetGroup(db,
                      GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                      "localCryptKeyBak");
  if (gr) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Loading localCryptKeyBak");
    mrdh->localCryptKeyBak=GWEN_CryptKey_FromDb(gr);
    if (!mrdh->localCryptKeyBak) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Error creating key from db");
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  /* done */
  GWEN_DB_Group_free(db);
  return 0;
}




AH_MEDIUM_CHECKRESULT AH_MediumRDHKeyFile__CheckFile(AH_MEDIUM *m, int fd){
  AH_MEDIUMRDHKEYFILE *mrdh;
  int rv;
  GWEN_BUFFER *fbuf;
  AH_MEDIUMHEADER *mh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* some file checks */
  fbuf=GWEN_Buffer_new(0, 256, 0, 1);

  if (lseek(fd, 0, SEEK_SET)==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "lseek(%s): %s",
	      AH_Medium_GetMediumName(m),
              strerror(errno));
    GWEN_Buffer_free(fbuf);
    return AH_MediumCheckResultGenericError;
  }

  while (1) {
    char buffer[256];

    rv=read(fd, buffer, sizeof(buffer));
    if (rv==-1) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "read(%s): %s",
                AH_Medium_GetMediumName(m),
                strerror(errno));
      GWEN_Buffer_free(fbuf);
      return AH_MediumCheckResultGenericError;
    }
    else if (rv==0)
      break;
    GWEN_Buffer_AppendBytes(fbuf, buffer, rv);
  } /* while */

  mh=AH_MediumHeader_FromBuffer(fbuf);
  if (!mh) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No medium header");
    GWEN_Buffer_free(fbuf);
    return AH_MediumCheckResultWrongMedium;
  }
  else {
    if (strcasecmp(AH_MediumHeader_GetMediumType(mh),
		   AH_MEDIUMRDHKEYFILE_NAME)) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "This file is not of type \"%s\" (it is \"%s\")",
		AH_MEDIUMRDHKEYFILE_NAME,
		AH_MediumHeader_GetMediumType(mh));
      AH_MediumHeader_free(mh);
      GWEN_Buffer_free(fbuf);
      return AH_MediumCheckResultWrongMedium;
    }

    /* check for keyfile versions */
    if (AH_MediumHeader_GetMajorVersion(mh)!=AH_MEDIUMRDHKEYFILE_VMAJOR) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
		"Major versions of keyfile do not match (%d!=%d)",
		rv, AH_MEDIUMRDHKEYFILE_VMAJOR);
      GWEN_WaitCallback_Log(0,
                            "Basically this file type is supported.\n"
                            "However, the major versions do not match,\n"
                            "so this particular version is not supported");
      AH_MediumHeader_free(mh);
      GWEN_Buffer_free(fbuf);
      return AH_MediumCheckResultWrongMedium;
    }
    if (AH_MediumHeader_GetMinorVersion(mh)>AH_MEDIUMRDHKEYFILE_VMINOR) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
		"The keyfile has been created with a newer "
		"library version.\n"
		"Using this file with the current version "
		"would degrade it.\n"
		"So for safety reasons I refuse to work "
		"with this file (%d>%d).",
		AH_MediumHeader_GetMinorVersion(mh),
		AH_MEDIUMRDHKEYFILE_VMINOR);
      GWEN_WaitCallback_Log(0,
                            "Basically this file type is supported.\n"
                            "However, this file has been created with a "
                            "newer library version.\n"
                            "Using this file with the current version "
                            "would degrade it.\n"
                            "So for safety reasons I refuse to work "
                            "with this file.");
      AH_MediumHeader_free(mh);
      GWEN_Buffer_free(fbuf);
      return AH_MediumCheckResultWrongMedium;
    }
  }
  AH_MediumHeader_free(mh);
  GWEN_Buffer_free(fbuf);

  return AH_MediumCheckResultOk;
}











int AH_MediumRDHKeyFile_Mount(AH_MEDIUM *m){
  AH_MEDIUMRDHKEYFILE *mrdh;
  /*   GWEN_BUFFEREDIO *bio; */
  char password[64];
  unsigned int i;
  int fd;
  int rv;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* create key from password */
  password[0]=0;
  mrdh->passWordIsSet=0;
  for (i=0; i<sizeof(mrdh->password); i++)
    mrdh->password[i]=0;
  if (AH_Medium_InputPin(m,
                         password,
                         AH_MEDIUMRDHKEYFILE_PINMINLENGTH,
                         sizeof(password),
                         0)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not get PIN");
    /*     GWEN_BufferedIO_free(bio); */
    return -1;
  }
  if (strlen(password)<AH_MEDIUMRDHKEYFILE_PINMINLENGTH) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Your program returned a shorter PIN than instructed!");
    /*     GWEN_BufferedIO_free(bio); */
    return -1;
  }

  if (GWEN_CryptKey_FromPassword(password,
				 mrdh->password, sizeof(mrdh->password))) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
    /*     GWEN_BufferedIO_free(bio); */
    return -1;
  }
  /* overwrite password as soon as possible */
  for (i=0; i<sizeof(password); i++)
    password[i]=0;
  mrdh->passWordIsSet=1;

  /* open file */
  fd=AH_MediumRDHKeyFile__OpenFile(m, 0);
  if (fd==-1) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not open keyfile for reading");
    return -1;
  }

  /* read file */
  rv=AH_MediumRDHKeyFile__ReadFile(m, fd);
  if (rv) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reading keyfile");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return rv;
  }

  /* close file */
  if (AH_MediumRDHKeyFile__CloseFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return -1;
  }

  return 0;
}



int AH_MediumRDHKeyFile__ReloadIfNeeded(AH_MEDIUM *m, int fd){
  AH_MEDIUMRDHKEYFILE *mrdh;
  struct stat st;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (fstat(fd, &st)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "stat(%s): %s",
              AH_Medium_GetMediumName(m),
              strerror(errno));
    return -1;
  }
  if (mrdh->mtime!=st.st_mtime ||
      mrdh->ctime!=st.st_ctime) {
    int rv;

    /* file has chenged, reload it */
    DBG_NOTICE(AQHBCI_LOGDOMAIN, "Keyfile changed externally, reloading it");
    /* read file */
    rv=AH_MediumRDHKeyFile__ReadFile(m, fd);
    if (rv) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Error reading keyfile");
      return rv;
    }
  }
  else {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Keyfile unchanged, not reloading");
  }
  return 0;
}



int AH_MediumRDHKeyFile_Create(AH_MEDIUM *m){
  AH_MEDIUMRDHKEYFILE *mrdh;
  struct stat st;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!AH_Medium_GetMediumName(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No medium name given");
    return -1;
  }

  if (stat(AH_Medium_GetMediumName(m), &st)) {
    if (errno!=ENOENT) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "stat(%s): %s",
                AH_Medium_GetMediumName(m),
                strerror(errno));
      return -1;
    }
    else {
      char password[64];
      unsigned int i;

      DBG_INFO(AQHBCI_LOGDOMAIN,
               "Keyfile \"%s\" does not exists, ok to create it",
               AH_Medium_GetMediumName(m));
      /* create key from password */
      password[0]=0;
      if (AH_Medium_InputPin(m,
                             password,
                             AH_MEDIUMRDHKEYFILE_PINMINLENGTH,
                             sizeof(password),
                             AB_BANKING_INPUT_FLAGS_CONFIRM)) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not get PIN");
        return -1;
      }
      if (strlen(password)<AH_MEDIUMRDHKEYFILE_PINMINLENGTH) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Your program returned a shorter PIN than instructed!");
        return -1;
      }

      if (GWEN_CryptKey_FromPassword(password,
                                     mrdh->password, sizeof(mrdh->password))) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
        return -1;
      }
      /* overwrite password as soon as possible */
      for (i=0; i<sizeof(password); i++)
        password[i]=0;
      mrdh->passWordIsSet=1;
    }
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "Keyfile \"%s\" already exists, will not create it",
              AH_Medium_GetMediumName(m));
    return -1;
  }

  return 0;
}



int AH_MediumRDHKeyFile__WriteFile(AH_MEDIUM *m, int fd){
  AH_MEDIUMRDHKEYFILE *mrdh;
  int rv;
  GWEN_BUFFER *fbuf;
  GWEN_BUFFER *hbuf;
  GWEN_BUFFER *rawbuf;
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_DB_NODE *gr;
  GWEN_ERRORCODE err;
  GWEN_BUFFEREDIO *bio;
  unsigned int i;
  struct stat st;
  AH_MEDIUMHEADER *mh;
  const char *s;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!AH_Medium_GetMediumName(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No medium name given");
    return -1;
  }

  db=GWEN_DB_Group_new("keyfile");

  GWEN_DB_SetIntValue(db,
                      GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "vmajor", AH_MEDIUMRDHKEYFILE_VMAJOR);
  GWEN_DB_SetIntValue(db,
                      GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "vminor", AH_MEDIUMRDHKEYFILE_VMINOR);

  GWEN_DB_SetIntValue(db,
                      GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "localsignseq", mrdh->localSignSeq);
  GWEN_DB_SetIntValue(db,
                      GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "remotesignseq", mrdh->remoteSignSeq);

  /* store context data */
  GWEN_DB_SetIntValue(db,
                      GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "country", mrdh->country);

  if (mrdh->bankId)
    GWEN_DB_SetCharValue(db,
                         GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "bankId", mrdh->bankId);

  if (mrdh->userId)
    GWEN_DB_SetCharValue(db,
                         GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "userId", mrdh->userId);

  if (mrdh->systemId)
    GWEN_DB_SetCharValue(db,
                         GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "systemId", mrdh->systemId);

  s=AH_Medium_GetPeerAddr(m);
  if (s) {
    GWEN_DB_SetCharValue(db,
                         GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "serverAddr", s);

    GWEN_DB_SetIntValue(db,
                        GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "serverPort",
                        AH_Medium_GetPeerPort(m));
  }

  /* store keys to DB */
  if (mrdh->localSignKey) {
    gr=GWEN_DB_GetGroup(db,
                        GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                        "localSignKey");
    assert(gr);
    err=GWEN_CryptKey_ToDb(mrdh->localSignKey, gr, 0);
    if (!GWEN_Error_IsOk(err)) {
      DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  if (mrdh->localCryptKey) {
    gr=GWEN_DB_GetGroup(db,
                        GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                        "localCryptKey");
    assert(gr);
    err=GWEN_CryptKey_ToDb(mrdh->localCryptKey, gr, 0);
    if (!GWEN_Error_IsOk(err)) {
      DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  if (mrdh->remoteSignKey) {
    gr=GWEN_DB_GetGroup(db,
                        GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                        "remoteSignKey");
    assert(gr);
    err=GWEN_CryptKey_ToDb(mrdh->remoteSignKey, gr, 1);
    if (!GWEN_Error_IsOk(err)) {
      DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  if (mrdh->remoteCryptKey) {
    gr=GWEN_DB_GetGroup(db,
                        GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                        "remoteCryptKey");
    assert(gr);
    err=GWEN_CryptKey_ToDb(mrdh->remoteCryptKey, gr, 1);
    if (!GWEN_Error_IsOk(err)) {
      DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  if (mrdh->localSignKeyBak) {
    gr=GWEN_DB_GetGroup(db,
                        GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                        "localSignKeyBak");
    assert(gr);
    err=GWEN_CryptKey_ToDb(mrdh->localSignKeyBak, gr, 0);
    if (!GWEN_Error_IsOk(err)) {
      DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  if (mrdh->localCryptKeyBak) {
    gr=GWEN_DB_GetGroup(db,
                        GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                        "localCryptKey");
    assert(gr);
    err=GWEN_CryptKey_ToDb(mrdh->localCryptKeyBak, gr, 0);
    if (!GWEN_Error_IsOk(err)) {
      DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
      GWEN_DB_Group_free(db);
      return -1;
    }
  }

  /* transform DB to buffer */
  rawbuf=GWEN_Buffer_new(0, 128, 0, 1);
  bio=GWEN_BufferedIO_Buffer_new(rawbuf);
  GWEN_BufferedIO_SetWriteBuffer(bio, 0, 256);
  if (GWEN_DB_WriteToStream(db, bio,
                            GWEN_DB_FLAGS_DEFAULT)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Error writing DB to buffer");
    GWEN_BufferedIO_Abandon(bio);
    GWEN_BufferedIO_free(bio);
    GWEN_DB_Group_free(db);
    return -1;
  }
  err=GWEN_BufferedIO_Close(bio);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_BufferedIO_free(bio);
    GWEN_DB_Group_free(db);
    return -1;
  }
  GWEN_DB_Group_free(db);

  /* create key from password */
  if (!mrdh->passWordIsSet) {
    char password[64];

    password[0]=0;
    if (AH_Medium_InputPin(m,
                           password,
                           AH_MEDIUMRDHKEYFILE_PINMINLENGTH,
                           sizeof(password),
                           AB_BANKING_INPUT_FLAGS_CONFIRM)) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not get PIN");
      GWEN_BufferedIO_free(bio);
      return -1;
    }
    if (strlen(password)<AH_MEDIUMRDHKEYFILE_PINMINLENGTH) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Your program returned a shorter PIN than instructed!");
      GWEN_BufferedIO_free(bio);
      return -1;
    }

    if (GWEN_CryptKey_FromPassword(password,
				   mrdh->password, sizeof(mrdh->password))) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
      GWEN_BufferedIO_free(bio);
      return -1;
    }
    /* overwrite password as soon as possible */
    for (i=0; i<sizeof(password); i++)
      password[i]=0;
    mrdh->passWordIsSet=1;
  } /* if password is not set */

  key=GWEN_CryptKey_Factory("DES");
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create DES key");
    GWEN_BufferedIO_free(bio);
    return -1;
  }
  err=GWEN_CryptKey_SetData(key, mrdh->password, sizeof(mrdh->password));
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_BufferedIO_free(bio);
    GWEN_CryptKey_free(key);
  }

  /* padd raw data */
  if (GWEN_Padd_PaddWithANSIX9_23(rawbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not padd keyfile");
    GWEN_BufferedIO_free(bio);
    GWEN_CryptKey_free(key);
    return -1;
  }

  /* encrypt file */
  fbuf=GWEN_Buffer_new(0, 256, 0, 1);
  GWEN_Buffer_ReserveBytes(fbuf, 128);

  err=GWEN_CryptKey_Encrypt(key, rawbuf, fbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_Buffer_free(fbuf);
    GWEN_BufferedIO_free(bio);
    GWEN_CryptKey_free(key);
  }
  GWEN_BufferedIO_free(bio);
  GWEN_CryptKey_free(key);

  GWEN_Buffer_Rewind(fbuf);

  /* create header */
  hbuf=GWEN_Buffer_new(0, 128, 0, 1);
  mh=AH_MediumHeader_new(AH_MEDIUMRDHKEYFILE_NAME,
			 AH_MEDIUMRDHKEYFILE_VMAJOR,
                         AH_MEDIUMRDHKEYFILE_VMINOR);
  AH_MediumHeader_ToBuffer(mh, hbuf);
  GWEN_Buffer_SetPos(fbuf, 0);
  GWEN_Buffer_InsertBuffer(fbuf, hbuf);
  AH_MediumHeader_free(mh);
  GWEN_Buffer_free(hbuf);

  if (lseek(fd, SEEK_SET, 0)==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
	      "lseek(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    GWEN_Buffer_free(fbuf);
    return -1;
  }
  if (ftruncate(fd, 0)==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
	      "ftruncate(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    GWEN_Buffer_free(fbuf);
    return -1;
  }

  while (1) {
    rv=write(fd,
             GWEN_Buffer_GetPosPointer(fbuf),
             GWEN_Buffer_GetBytesLeft(fbuf));
    if (rv==-1) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "write(%s): %s",
                AH_Medium_GetMediumName(m),
		strerror(errno));
      GWEN_Buffer_free(fbuf);
      return -1;
    }
    else if (rv==0)
      break;
    GWEN_Buffer_IncrementPos(fbuf, rv);
  } /* while */

  /* get times */
  if (fstat(fd, &st)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
	      "stat(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    GWEN_Buffer_free(fbuf);
    return -1;
  }

  if (st.st_mode & 0077) {
    DBG_WARN(AQHBCI_LOGDOMAIN,
             "WARNING: Your keyfile \"%s\" is group/world accessible!\n"
	     "Event though I told the system to make this file accessible\n"
	     "only to the owner your system did not respect this.\n"
             "This means you have to be very carefull with that file!",
	     AH_Medium_GetMediumName(m));
  }
  mrdh->mtime=st.st_mtime;
  mrdh->ctime=st.st_ctime;

  GWEN_Buffer_free(fbuf);

  return 0;
}



int AH_MediumRDHKeyFile_Unmount(AH_MEDIUM *m, int force){
  AH_MEDIUMRDHKEYFILE *mrdh;
  int rv;
  unsigned int i;
  int fd;
  const char *s;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* sample vars */
  s=AH_Medium_GetSecurityIdPtr(m);
  free(mrdh->systemId);
  if (s) mrdh->systemId=strdup(s);
  else mrdh->systemId=0;

  /* open file */
  fd=AH_MediumRDHKeyFile__OpenFile(m, 1);
  if (fd==-1) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not open keyfile for writing");
    return -1;
  }

  /* write file */
  rv=AH_MediumRDHKeyFile__WriteFile(m, fd);
  if (rv) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing keyfile");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return rv;
  }

  /* close file */
  if (AH_MediumRDHKeyFile__CloseFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return -1;
  }

  /* overwrite password */
  mrdh->passWordIsSet=0;
  for (i=0; i<sizeof(mrdh->password); i++)
    mrdh->password[i]=0;

  /* free all keys */
  GWEN_CryptKey_free(mrdh->localSignKey);
  mrdh->localSignKey=0;
  GWEN_CryptKey_free(mrdh->localCryptKey);
  mrdh->localCryptKey=0;
  GWEN_CryptKey_free(mrdh->remoteSignKey);
  mrdh->remoteSignKey=0;
  GWEN_CryptKey_free(mrdh->remoteCryptKey);
  mrdh->remoteCryptKey=0;
  GWEN_CryptKey_free(mrdh->localSignKeyBak);
  mrdh->localSignKeyBak=0;
  GWEN_CryptKey_free(mrdh->localCryptKeyBak);
  mrdh->localCryptKeyBak=0;

  AH_Medium_SetLocalSignKeySpec(m, 0);
  AH_Medium_SetLocalCryptKeySpec(m, 0);
  AH_Medium_SetRemoteSignKeySpec(m, 0);
  AH_Medium_SetRemoteCryptKeySpec(m, 0);

  mrdh->mtime=0;
  mrdh->ctime=0;

  mrdh->selected=0;
  return 0;
}



AH_MEDIUM_RESULT AH_MediumRDHKeyFile_Sign(AH_MEDIUM *m,
                                          GWEN_BUFFER *msgbuf,
                                          GWEN_BUFFER *signbuf){
  AH_MEDIUMRDHKEYFILE *mrdh;
  GWEN_BUFFER *hashbuf;
  GWEN_ERRORCODE err;
  char buffer[20];
  unsigned int bsize;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* open and lock file (open for writing) */
  fd=AH_MediumRDHKeyFile__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumRDHKeyFile__ReloadIfNeeded(m, fd)) {
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    return AH_MediumResultGenericError;
  }

  if (mrdh->localSignKey==0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No local sign key");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultNoKey;
  }

  /* hash data */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Hash data");
  bsize=sizeof(buffer);
  if (GWEN_MD_Hash("RMD160",
                   GWEN_Buffer_GetStart(msgbuf),
                   GWEN_Buffer_GetUsedBytes(msgbuf),
                   buffer,
                   &bsize)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Error hashing message");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Hashing done");

  hashbuf=GWEN_Buffer_new(0, bsize, 0, 1);
  GWEN_Buffer_AppendBytes(hashbuf, buffer, bsize);

  /* padd */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Padding hash using ISO 9796");
  if (GWEN_Padd_PaddWithISO9796(hashbuf)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    GWEN_Buffer_free(hashbuf);
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  /* sign hash */
  GWEN_Buffer_Rewind(hashbuf);
  err=GWEN_CryptKey_Sign(mrdh->localSignKey,
                         hashbuf,
                         signbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_Buffer_free(hashbuf);
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }
  GWEN_Buffer_free(hashbuf);

  /* increment signature counter */
  mrdh->localSignSeq++;

  /* write file (just to be sure that the last change in on disc) */
  if (AH_MediumRDHKeyFile__WriteFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (AH_MediumRDHKeyFile__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Signing done");
  return AH_MediumResultOk;
}



int AH_MediumRDHKeyFile_GetNextSignSeq(AH_MEDIUM *m){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  return mrdh->localSignSeq;
}



int AH_MediumRDHKeyFile_SetLocalSignSeq(AH_MEDIUM *m, int i) {
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (i<mrdh->localSignSeq) {
    DBG_WARN(AQHBCI_LOGDOMAIN,
             "New localSignSeq is lesser than current one, "
             "this is not a good idea");
  }
  mrdh->localSignSeq=i;
  return 0;
}



AH_MEDIUM_RESULT AH_MediumRDHKeyFile_Verify(AH_MEDIUM *m,
                                            GWEN_BUFFER *msgbuf,
                                            GWEN_BUFFER *signbuf,
                                            int signseq){
  AH_MEDIUMRDHKEYFILE *mrdh;
  GWEN_BUFFER *hashbuf;
  GWEN_ERRORCODE err;
  char buffer[20];
  unsigned int bsize;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* open and lock file (open for writing) */
  fd=AH_MediumRDHKeyFile__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumRDHKeyFile__ReloadIfNeeded(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (mrdh->remoteSignKey==0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No remote sign key");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultNoKey;
  }

  /* check signature counter */
  if (signseq) {
    if (mrdh->remoteSignSeq>=signseq){
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "Double use of signature detected (%d>=%d)",
                mrdh->remoteSignSeq, signseq);
      AH_MediumRDHKeyFile__CloseFile(m, fd);
      return AH_MediumResultSignSeq;
    }
    mrdh->remoteSignSeq=signseq;

    /* write file (just to be sure that the last change in on disc) */
    if (AH_MediumRDHKeyFile__WriteFile(m, fd)) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
      AH_MediumRDHKeyFile__CloseFile(m, fd);
      return AH_MediumResultGenericError;
    }
  }

  /* hash data */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Hashing data");
  bsize=sizeof(buffer);
  if (GWEN_MD_Hash("RMD160",
                   GWEN_Buffer_GetStart(msgbuf),
                   GWEN_Buffer_GetUsedBytes(msgbuf),
                   buffer,
                   &bsize)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Error hashing message");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Hashing done");

  hashbuf=GWEN_Buffer_new(0, bsize, 0, 1);
  GWEN_Buffer_AppendBytes(hashbuf, buffer, bsize);

  /* padd */
  DBG_DEBUG(0, "Padding hash using ISO 9796");
  if (GWEN_Padd_PaddWithISO9796(hashbuf)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    GWEN_Buffer_free(hashbuf);
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  /* verify hash */
  GWEN_Buffer_Rewind(hashbuf);
  GWEN_Buffer_Rewind(signbuf);
  err=GWEN_CryptKey_Verify(mrdh->remoteSignKey,
                           hashbuf,
                           signbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    DBG_WARN(AQHBCI_LOGDOMAIN, "Invalid signature");
    GWEN_Buffer_free(hashbuf);
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultInvalidSignature;
  }
  DBG_INFO(AQHBCI_LOGDOMAIN, "Signature is valid");

  if (AH_MediumRDHKeyFile__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  GWEN_Buffer_free(hashbuf);
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Verification done");
  return AH_MediumResultOk;
}



AH_MEDIUM_RESULT AH_MediumRDHKeyFile_EncryptKey(AH_MEDIUM *m,
                                                GWEN_BUFFER *srckey,
                                                GWEN_BUFFER *enckey){
  AH_MEDIUMRDHKEYFILE *mrdh;
  int fd;
  AH_MEDIUM_RESULT rv;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* open and lock file (open for reading) */
  fd=AH_MediumRDHKeyFile__OpenFile(m, 0);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumRDHKeyFile__ReloadIfNeeded(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  rv=AH_MediumRDH_EncryptKey(m, mrdh->remoteCryptKey, srckey, enckey);

  if (AH_MediumRDHKeyFile__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  return rv;
}



AH_MEDIUM_RESULT AH_MediumRDHKeyFile_DecryptKey(AH_MEDIUM *m,
                                                GWEN_BUFFER *srckey,
                                                GWEN_BUFFER *deckey){
  AH_MEDIUMRDHKEYFILE *mrdh;
  int fd;
  AH_MEDIUM_RESULT rv;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* open and lock file (open for reading) */
  fd=AH_MediumRDHKeyFile__OpenFile(m, 0);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumRDHKeyFile__ReloadIfNeeded(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  rv=AH_MediumRDH_DecryptKey(m, mrdh->localCryptKey, srckey, deckey);

  if (AH_MediumRDHKeyFile__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  return rv;
}



GWEN_BUFFER *AH_MediumRDHKeyFile_GenerateMsgKey(AH_MEDIUM *m){
  AH_MEDIUMRDHKEYFILE *mrdh;
  GWEN_ERRORCODE err;
  GWEN_CRYPTKEY *sessionKey;
  GWEN_BUFFER *kbuf;
  unsigned char skbuffer[16];
  unsigned int sksize;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  /* open and lock file (open for reading) */
  fd=AH_MediumRDHKeyFile__OpenFile(m, 0);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return 0;
  }

  /* reload keyfile if necessary */
  if (AH_MediumRDHKeyFile__ReloadIfNeeded(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return 0;
  }

  /* generate session key */
  sessionKey=GWEN_CryptKey_Factory("DES");
  if (!sessionKey) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create DES session key");
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return 0;
  }
  err=GWEN_CryptKey_Generate(sessionKey, 0);
  if (!GWEN_Error_IsOk(err)) {
    DBG_ERROR_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_CryptKey_free(sessionKey);
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return 0;
  }

  sksize=sizeof(skbuffer);
  err=GWEN_CryptKey_GetData(sessionKey,
                            skbuffer,
                            &sksize);
  if (!GWEN_Error_IsOk(err)) {
    DBG_ERROR_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_CryptKey_free(sessionKey);
    AH_MediumRDHKeyFile__CloseFile(m, fd);
    return 0;
  }
  GWEN_CryptKey_free(sessionKey);

  if (AH_MediumRDHKeyFile__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return 0;
  }

  kbuf=GWEN_Buffer_new(0, sksize, 0, 1);
  GWEN_Buffer_AppendBytes(kbuf, skbuffer, sksize);
  return kbuf;
}



int AH_MediumRDHKeyFile_ChangePin(AH_MEDIUM *m){
  char password[64];
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }

  /* create key from password */
  memset(password, 0, sizeof(password));
  if (AH_Medium_InputPin(m,
                         password,
                         AH_MEDIUMRDHKEYFILE_PINMINLENGTH,
                         sizeof(password),
                         AB_BANKING_INPUT_FLAGS_CONFIRM)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not get PIN");
    return -1;
  }
  if (strlen(password)<AH_MEDIUMRDHKEYFILE_PINMINLENGTH) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Your program returned a shorter PIN than instructed!");
    return -1;
  }

  if (GWEN_CryptKey_FromPassword(password,
				 mrdh->password, sizeof(mrdh->password))) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
    return -1;
  }
  /* overwrite password as soon as possible */
  memset(password, 0, sizeof(password));
  mrdh->passWordIsSet=1;

  return 0;
}



int AH_MediumRDHKeyFile_SelectContext(AH_MEDIUM *m, int idx){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mrdh->selected=0;
    return -1;
  }

  if (idx!=0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Invalid index %d (only 0 is accepted)", idx);
    return -1;
  }

  if (mrdh->localSignKey)
    AH_Medium_SetLocalSignKeySpec(m,
                                  GWEN_CryptKey_GetKeySpec(mrdh->localSignKey));

  if (mrdh->localCryptKey)
    AH_Medium_SetLocalCryptKeySpec(m,
                                   GWEN_CryptKey_GetKeySpec(mrdh->localCryptKey));

  if (mrdh->remoteSignKey)
    AH_Medium_SetRemoteSignKeySpec(m,
                                   GWEN_CryptKey_GetKeySpec(mrdh->remoteSignKey));

  if (mrdh->remoteCryptKey)
    AH_Medium_SetRemoteCryptKeySpec(m,
                                    GWEN_CryptKey_GetKeySpec(mrdh->remoteCryptKey));

  mrdh->selected=1;
  return 0;
}



int AH_MediumRDHKeyFile_CreateContext(AH_MEDIUM *m,
                                      int country,
                                      const char *bankId,
                                      const char *userId){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  assert(bankId);
  assert(userId);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }

  if (mrdh->selected ||
      mrdh->country ||
      mrdh->bankId ||
      mrdh->userId) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium already contains a user");
    return -1;
  }

  mrdh->country=country;
  mrdh->bankId=strdup(bankId);
  mrdh->userId=strdup(userId);

  return 0;
}



int AH_MediumRDHKeyFile_RemoveContext(AH_MEDIUM *m, int idx) {
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mrdh->selected=0;
    return -1;
  }

  if (idx!=0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Invalid index %d (only 0 is accepted)", idx);
    return -1;
  }

  mrdh->country=0;
  free(mrdh->bankId); mrdh->bankId=0;
  free(mrdh->userId); mrdh->userId=0;

  mrdh->selected=0;
  return 0;
}



int AH_MediumRDHKeyFile_ReadContext(AH_MEDIUM *m,
                                    int idx,
                                    int *country,
                                    GWEN_BUFFER *bankId,
                                    GWEN_BUFFER *userId,
                                    GWEN_BUFFER *server,
                                    int *port){
  AH_MEDIUMRDHKEYFILE *mrdh;
  const char *s;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mrdh->selected=0;
    return -1;
  }

  if (idx>0 || mrdh->country==0) {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Context %d not found", idx);
    return -1;
  }
  if (country)
    *country=mrdh->country;
  if (bankId && mrdh->bankId)
    GWEN_Buffer_AppendString(bankId, mrdh->bankId);
  if (userId && mrdh->userId)
    GWEN_Buffer_AppendString(userId, mrdh->userId);
  s=AH_Medium_GetPeerAddr(m);
  if (server && s)
    GWEN_Buffer_AppendString(server, s);
  if (port)
    *port=AH_Medium_GetPeerPort(m);
  return 0;
}




int AH_MediumRDHKeyFile_WriteContext(AH_MEDIUM *m,
                                     int idx,
                                     int country,
                                     const char *bankId,
                                     const char *userId,
                                     const char *server,
                                     int port){
  AH_MEDIUMRDHKEYFILE *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUMRDHKEYFILE, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mrdh->selected=0;
    return -1;
  }

  if (idx>0) {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Context %d not found", idx);
    return -1;
  }

  if (country)
    mrdh->country=country;
  if (bankId) {
    free(mrdh->bankId);
    mrdh->bankId=strdup(bankId);
  }

  if (userId) {
    free(mrdh->userId);
    mrdh->userId=strdup(userId);
  }

  if (server)
    AH_Medium_SetPeerAddr(m, server);

  if (port)
    AH_Medium_SetPeerPort(m, port);

  return 0;
}


















AH_MEDIUMPROVIDER *rdhkeyfile_MediumProvider_new(AH_HBCI *hbci){
  AH_MEDIUMPROVIDER *mp;

  mp=AH_MediumProvider_new(hbci, AH_MEDIUMRDHKEYFILE_NAME);
  AH_MediumProvider_SetDeviceType(mp, AH_MediumDeviceFile);
  AH_MediumProvider_SetFactoryFn(mp, AH_MediumRDHKeyFileProvider_Factory);
  AH_MediumProvider_SetCheckFn(mp, AH_MediumRDHKeyFileProvider_Check);
  return mp;
}



AH_MEDIUM*
AH_MediumRDHKeyFileProvider_Factory(AH_MEDIUMPROVIDER *mp,
                                    AH_HBCI *hbci,
                                    const char *mediumName) {
  return AH_MediumRDHKeyFile_new(hbci,
                                 mediumName);
}



AH_MEDIUM_CHECKRESULT
AH_MediumRDHKeyFileProvider_Check(AH_MEDIUMPROVIDER *mp,
                                  AH_HBCI *hbci,
                                  GWEN_BUFFER *mediumName) {
  int fd;
  AH_MEDIUM_CHECKRESULT res;
  AH_MEDIUM *m;

  m=AH_MediumRDHKeyFile_new(hbci, GWEN_Buffer_GetStart(mediumName));
  assert(m);

  fd=AH_MediumRDHKeyFile__OpenFile(m, 0);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open file");
    AH_Medium_free(m);
    return AH_MediumCheckResultGenericError;
  }

  res=AH_MediumRDHKeyFile__CheckFile(m, fd);

  if (AH_MediumRDHKeyFile__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close file");
    AH_Medium_free(m);
    return AH_MediumCheckResultGenericError;
  }

  return res;
}















