/***************************************************************************
 $RCSfile: medium.c,v $
                             -------------------
    cvs         : $Id: medium.c,v 1.28 2005/03/28 02:04:32 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <gwenhywfar/misc.h>

#include "medium_p.h"
#include "aqhbci_l.h"
#include <gwenhywfar/debug.h>
#include <gwenhywfar/padd.h>
#include <gwenhywfar/md.h>

#include <stdlib.h>
#include <assert.h>
#include <string.h>


GWEN_LIST_FUNCTIONS(AH_MEDIUM, AH_Medium);
GWEN_INHERIT_FUNCTIONS(AH_MEDIUM);



AH_MEDIUM *AH_Medium_new(AH_HBCI *hbci,
                         AH_MEDIUMTYPE mt,
                         const char *mtn,
                         const char *mediumName) {
  AH_MEDIUM *m;

  assert(hbci);
  assert(mtn);
  GWEN_NEW_OBJECT(AH_MEDIUM, m);
  m->usage=1;
  GWEN_LIST_INIT(AH_MEDIUM, m);
  GWEN_INHERIT_INIT(AH_MEDIUM, m);
  m->hbci=hbci;
  m->mediumType=mt;
  m->mediumTypeName=strdup(mtn);
  if (mediumName)
    m->mediumName=strdup(mediumName);
  m->selectedContext=-1;

  m->isDirty=0;
  return m;
}



void AH_Medium_Attach(AH_MEDIUM *m){
  assert(m);
  m->usage++;
}



void AH_Medium_free(AH_MEDIUM *m){
  if (m) {
    assert(m->usage);
    m->usage--;
    if (m->usage==0) {
      DBG_DEBUG(AQHBCI_LOGDOMAIN, "Destroying AH_MEDIUM");
      GWEN_LIST_FINI(AH_MEDIUM, m);
      GWEN_INHERIT_FINI(AH_MEDIUM, m);
      free(m->mediumTypeName);
      free(m->mediumName);
      free(m->pSecurityId);
      free(m);
    }
  }
}



const char *AH_Medium_GetDescriptiveName(const AH_MEDIUM *m) {
  assert(m);
  if (m->descriptiveName)
    return m->descriptiveName;
  else
    return m->mediumName;
}



void AH_Medium_SetDescriptiveName(AH_MEDIUM *m, const char *s){
  assert(m);
  free(m->descriptiveName);
  if (s)
    m->descriptiveName=strdup(s);
  else
    m->descriptiveName=0;
  m->isDirty=1;
}



int AH_Medium_IsDirty(const AH_MEDIUM *m) {
  assert(m);
  return m->isDirty;
}



void AH_Medium_SetIsDirty(AH_MEDIUM *m, int i) {
  assert(m);
  m->isDirty=i;
}



void AH_Medium_SetDeviceType(AH_MEDIUM *m, AH_MEDIUMDEVICE md){
  assert(m);
  m->deviceType=md;
  m->isDirty=1;
}



AH_MEDIUMDEVICE AH_Medium_GetDeviceType(const AH_MEDIUM *m){
  assert(m);
  return m->deviceType;
}



const char *AH_Medium_GetMediumName(const AH_MEDIUM *m){
  assert(m);
  return m->mediumName;
}



void AH_Medium_SetMediumName(AH_MEDIUM *m, const char *s){
  assert(m);
  free(m->mediumName);
  if (s) m->mediumName=strdup(s);
  else m->mediumName=0;
  m->isDirty=1;
}



GWEN_TYPE_UINT32 AH_Medium_GetFlags(const AH_MEDIUM *m){
  assert(m);
  return m->flags;
}



void AH_Medium_SetFlags(AH_MEDIUM *m, GWEN_TYPE_UINT32 fl){
  assert(m);
  m->flags=fl;
}



void AH_Medium_AddFlags(AH_MEDIUM *m, GWEN_TYPE_UINT32 fl){
  assert(m);
  m->flags|=fl;
}



void AH_Medium_SubFlags(AH_MEDIUM *m, GWEN_TYPE_UINT32 fl){
  assert(m);
  m->flags&=~fl;
}



int AH_Medium_Mount(AH_MEDIUM *m){
  assert(m);
  if (m->mountFn) {
    int rv;

    if (m->mountCount==0) {
      DBG_DEBUG(AQHBCI_LOGDOMAIN, "Really mounting medium");
      m->selectedContext=-1;
      rv=m->mountFn(m);
      if (rv) {
        DBG_INFO(AQHBCI_LOGDOMAIN, "Error mounting medium (%d)", rv);
        return rv;
      }
    }
    else
      rv=0;
    m->mountCount++;
    return rv;
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No mount function set");
    return -1;
  }
}



int AH_Medium_Create(AH_MEDIUM *m){
  int rv;

  assert(m);
  if (AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is mounted");
    return -1;
  }
  m->selectedContext=-1;
  if (m->createFn) {
    rv=m->createFn(m);
    if (rv) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "here");
      return rv;
    }
    m->mountCount++;
    return 0;
  }
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No create function set");
    return -1;
  }
}



int AH_Medium_Unmount(AH_MEDIUM *m, int force){
  assert(m);
  if (m->unmountFn) {
    int rv;

    if (m->mountCount) {
      if (m->mountCount==1 || force) {
        DBG_DEBUG(AQHBCI_LOGDOMAIN, "Really unmounting medium");
        rv=m->unmountFn(m, force);
        m->selectedContext=-1;
      }
      else
        rv=0;
      if (rv) {
        DBG_INFO(AQHBCI_LOGDOMAIN, "Error unmounting medium (%d)", rv);
        return rv;
      }
      m->mountCount--;
      if (force)
        m->mountCount=0;
      return rv;
    }
    else {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
      return -1;
    }

  }
  else {
    m->selectedContext=-1;
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No unmount function set");
    return -1;
  }
}



int AH_Medium_ChangePin(AH_MEDIUM *m){
  assert(m);
  if (m->changePinFn)
    return m->changePinFn(m);
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No changePin function set");
    return -1;
  }
}



int AH_Medium_IsMounted(AH_MEDIUM *m){
  assert(m);
  return m->mountCount!=0;
}



AH_MEDIUM_RESULT AH_Medium_Sign(AH_MEDIUM *m,
                                GWEN_BUFFER *msgbuf,
                                GWEN_BUFFER *signbuf){
  assert(m);
  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }
  if (m->signFn)
    return m->signFn(m, msgbuf, signbuf);
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No sign function set");
    return -1;
  }
}



int AH_Medium_GetNextSignSeq(AH_MEDIUM *m){
  assert(m);
  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }
  if (m->getNextSignSeqFn)
    return m->getNextSignSeqFn(m);
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No getNextSignSeqFn function set");
    return -1;
  }
}



int AH_Medium_SetLocalSignSeq(AH_MEDIUM *m, int i){
  assert(m);
  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }
  if (m->setLocalSignSeqFn)
    return m->setLocalSignSeqFn(m, i);
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No setLocalSignSeqFn function set");
    return -1;
  }
}



AH_MEDIUM_RESULT AH_Medium_Decrypt(AH_MEDIUM *m,
                                   GWEN_BUFFER *msgbuf,
                                   GWEN_BUFFER *decryptbuf,
                                   GWEN_BUFFER *msgKeyBuffer){
  GWEN_ERRORCODE err;
  GWEN_BUFFER *kbuf;
  GWEN_CRYPTKEY *sessionKey;
  AH_MEDIUM_RESULT res;

  assert(m);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return AH_MediumResultGenericError;
  }

  /* create session key */
  GWEN_Buffer_Rewind(msgKeyBuffer);

  /* decrypt session key */
  kbuf=GWEN_Buffer_new(0, 256, 0, 1);
  res=AH_Medium_DecryptKey(m, msgKeyBuffer, kbuf);
  if (res!=AH_MediumResultOk) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Error decrypting message key");
    GWEN_Buffer_free(kbuf);
    return res;
  }

  if (GWEN_Buffer_GetUsedBytes(kbuf)<16) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "Bad size of session key (%d bytes)",
              GWEN_Buffer_GetUsedBytes(kbuf));
    GWEN_Buffer_free(kbuf);
    return AH_MediumResultBadKey;
  }
  GWEN_Buffer_Crop(kbuf, GWEN_Buffer_GetUsedBytes(kbuf)-16, 16);

  sessionKey=GWEN_CryptKey_Factory("DES");
  assert(sessionKey);
  GWEN_CryptKey_SetData(sessionKey,
                        GWEN_Buffer_GetStart(kbuf),
                        16);
  GWEN_Buffer_free(kbuf);

  /* now decrypt the message using the new session key */
  err=GWEN_CryptKey_Decrypt(sessionKey,
			    msgbuf,
			    decryptbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    GWEN_CryptKey_free(sessionKey);
    return AH_MediumResultBadKey;
  }
  GWEN_CryptKey_free(sessionKey);

  /* unpadd message */
  if (GWEN_Padd_UnpaddWithANSIX9_23(decryptbuf)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    return AH_MediumResultBadKey;
  }

  return AH_MediumResultOk;
}



AH_MEDIUM_RESULT AH_Medium_Verify(AH_MEDIUM *m,
                                  GWEN_BUFFER *msgbuf,
                                  GWEN_BUFFER *signbuf,
                                  int signseq){
  assert(m);
  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }
  if (m->verifyFn)
    return m->verifyFn(m, msgbuf, signbuf, signseq);
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No verify function set");
    return -1;
  }
}



AH_MEDIUM_RESULT AH_Medium_EncryptKey(AH_MEDIUM *m,
                                      GWEN_BUFFER *srckey,
                                      GWEN_BUFFER *encKey){
  assert(m);
  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return AH_MediumResultGenericError;
  }
  if (m->encryptKeyFn)
    return m->encryptKeyFn(m, srckey, encKey);
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No encryptKey function set");
    return AH_MediumResultGenericError;
  }
}



AH_MEDIUM_RESULT AH_Medium_DecryptKey(AH_MEDIUM *m,
                                      GWEN_BUFFER *srckey,
                                      GWEN_BUFFER *decKey){
  assert(m);
  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return AH_MediumResultGenericError;
  }
  if (m->decryptKeyFn)
    return m->decryptKeyFn(m, srckey, decKey);
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No decryptKey function set");
    return AH_MediumResultGenericError;
  }
}



GWEN_BUFFER *AH_Medium_GenerateMsgKey(AH_MEDIUM *m) {
  assert(m);
  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return 0;
  }
  if (m->generateMsgKeyFn)
    return m->generateMsgKeyFn(m);
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No generateMsgKey function set");
    return 0;
  }
}




AH_MEDIUM_RESULT AH_Medium_Encrypt(AH_MEDIUM *m,
                                   GWEN_BUFFER *msgbuf,
                                   GWEN_BUFFER *encryptbuf,
                                   GWEN_BUFFER *msgKeyBuffer){
  GWEN_CRYPTKEY *sessionKey;
  GWEN_BUFFER *kbuf;
  GWEN_ERRORCODE err;
  AH_MEDIUM_RESULT res;

  assert(m);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return AH_MediumResultGenericError;
  }

  sessionKey=GWEN_CryptKey_Factory("DES");
  if (!sessionKey) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create DES session key");
    return AH_MediumResultNoKey;
  }

  kbuf=AH_Medium_GenerateMsgKey(m);
  if (!kbuf) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not generate message key");
    return AH_MediumResultNoKey;
  }

  err=GWEN_CryptKey_SetData(sessionKey,
                            GWEN_Buffer_GetStart(kbuf),
                            GWEN_Buffer_GetUsedBytes(kbuf));
  if (!GWEN_Error_IsOk(err)) {
    DBG_ERROR_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_Buffer_free(kbuf);
    GWEN_CryptKey_free(sessionKey);
    return AH_MediumResultNoKey;
  }

  /* padd message */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Padding with ANSI X9.23");
  if (GWEN_Padd_PaddWithANSIX9_23(msgbuf)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    GWEN_Buffer_free(kbuf);
    GWEN_CryptKey_free(sessionKey);
    return AH_MediumResultGenericError;
  }
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Padding with ANSI X9.23: done");

  /* encrypt message */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Encrypting with session key");
  err=GWEN_CryptKey_Encrypt(sessionKey,
                            msgbuf,
                            encryptbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    GWEN_Buffer_free(kbuf);
    GWEN_CryptKey_free(sessionKey);
    return AH_MediumResultGenericError;
  }
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Encrypting with session key: done");
  GWEN_CryptKey_free(sessionKey);

  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Encrypting key");
  res=AH_Medium_EncryptKey(m,
                           kbuf,
                           msgKeyBuffer);
  if (res!=AH_MediumResultOk) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not encrypt message key");
    GWEN_Buffer_free(kbuf);
    return res;
  }
  GWEN_Buffer_free(kbuf);
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Encrypting key: done");

  return AH_MediumResultOk;
}



void AH_Medium_SetChangePinFn(AH_MEDIUM *m, AH_MEDIUM_CHANGEPINFN f){
  assert(m);
  m->changePinFn=f;
}



void AH_Medium_SetMountFn(AH_MEDIUM *m, AH_MEDIUM_MOUNTFN f){
  assert(m);
  m->mountFn=f;
}



void AH_Medium_SetCreateFn(AH_MEDIUM *m, AH_MEDIUM_CREATEFN f){
  assert(m);
  m->createFn=f;
}



void AH_Medium_SetUnmountFn(AH_MEDIUM *m, AH_MEDIUM_UNMOUNTFN f){
  assert(m);
  m->unmountFn=f;
}



void AH_Medium_SetFromDbFn(AH_MEDIUM *m, AH_MEDIUM_FROMDBFN f){
  assert(m);
  m->fromDbFn=f;
}



void AH_Medium_SetToDbFn(AH_MEDIUM *m, AH_MEDIUM_TODBFN f){
  assert(m);
  m->toDbFn=f;
}



void AH_Medium_SetSignFn(AH_MEDIUM *m, AH_MEDIUM_SIGNFN f){
  assert(m);
  m->signFn=f;
}



void AH_Medium_SetGetNextSignSeqFn(AH_MEDIUM *m,
                                   AH_MEDIUM_GETNEXTSIGNSEQFN f){
  assert(m);
  m->getNextSignSeqFn=f;
}



void AH_Medium_SetSetLocalSignSeqFn(AH_MEDIUM *m,
                                    AH_MEDIUM_SETLOCALSIGNSEQFN f){
  assert(m);
  m->setLocalSignSeqFn=f;
}



void AH_Medium_SetEncryptKeyFn(AH_MEDIUM *m,
                               AH_MEDIUM_ENCRYPTKEYFN f){
  assert(m);
  m->encryptKeyFn=f;
}



void AH_Medium_SetDecryptKeyFn(AH_MEDIUM *m,
                               AH_MEDIUM_DECRYPTKEYFN f){
  assert(m);
  m->decryptKeyFn=f;
}



void AH_Medium_SetGenerateMsgKeyFn(AH_MEDIUM *m,
                                   AH_MEDIUM_GENERATEMSGKEYFN f){
  assert(m);
  m->generateMsgKeyFn=f;
}



void AH_Medium_SetVerifyFn(AH_MEDIUM *m,
                               AH_MEDIUM_VERIFYFN f){
  assert(m);
  m->verifyFn=f;
}



AH_MEDIUMTYPE AH_Medium_GetMediumType(const AH_MEDIUM *m){
  assert(m);
  return m->mediumType;
}



const char *AH_Medium_GetMediumTypeName(const AH_MEDIUM *m){
  assert(m);
  return m->mediumTypeName;
}



const GWEN_KEYSPEC*
AH_Medium_GetLocalSignKeySpec(AH_MEDIUM *m){
  assert(m);
  return m->localSignKeySpec;
}



void AH_Medium_SetLocalSignKeySpec(AH_MEDIUM *m, const GWEN_KEYSPEC *ks){
  assert(m);
  GWEN_KeySpec_free(m->localSignKeySpec);
  if (ks) m->localSignKeySpec=GWEN_KeySpec_dup(ks);
  else m->localSignKeySpec=0;
}




const GWEN_KEYSPEC*
AH_Medium_GetLocalCryptKeySpec(AH_MEDIUM *m){
  assert(m);
  return m->localCryptKeySpec;
}



void AH_Medium_SetLocalCryptKeySpec(AH_MEDIUM *m,
                                    const GWEN_KEYSPEC *ks){
  assert(m);
  GWEN_KeySpec_free(m->localCryptKeySpec);
  if (ks) m->localCryptKeySpec=GWEN_KeySpec_dup(ks);
  else m->localCryptKeySpec=0;
}



const GWEN_KEYSPEC*
AH_Medium_GetRemoteSignKeySpec(AH_MEDIUM *m){
  assert(m);
  return m->remoteSignKeySpec;
}



void AH_Medium_SetRemoteSignKeySpec(AH_MEDIUM *m,
                                        const GWEN_KEYSPEC *ks){
  assert(m);
  GWEN_KeySpec_free(m->remoteSignKeySpec);
  if (ks) m->remoteSignKeySpec=GWEN_KeySpec_dup(ks);
  else m->remoteSignKeySpec=0;
}



const GWEN_KEYSPEC*
AH_Medium_GetRemoteCryptKeySpec(AH_MEDIUM *m){
  assert(m);
  return m->remoteCryptKeySpec;
}



void AH_Medium_SetRemoteCryptKeySpec(AH_MEDIUM *m,
                                         const GWEN_KEYSPEC *ks){
  assert(m);
  GWEN_KeySpec_free(m->remoteCryptKeySpec);
  if (ks) m->remoteCryptKeySpec=GWEN_KeySpec_dup(ks);
  else m->remoteCryptKeySpec=0;
}



int AH_Medium_ToDB(const AH_MEDIUM *m, GWEN_DB_NODE *db){
  GWEN_DB_NODE *gr;
  const char *s;

  GWEN_DB_SetCharValue(db,
                       GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "mediumTypeName", m->mediumTypeName);
  GWEN_DB_SetCharValue(db,
                       GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "mediumName", m->mediumName);
  GWEN_DB_SetIntValue(db,
		      GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
		      "flags", m->flags);
  switch(m->mediumType) {
  case AH_MediumTypeDDV:    s="ddv"; break;
  case AH_MediumTypeRDH:    s="rdh"; break;
  case AH_MediumTypePINTAN: s="pintan"; break;
  default:
    DBG_ERROR(AQHBCI_LOGDOMAIN, "unknown medium type %d", m->mediumType);
    return -1;
  }
  GWEN_DB_SetCharValue(db,
                       GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "mediumtype", s);
  if (m->descriptiveName)
    GWEN_DB_SetCharValue(db,
                         GWEN_DB_FLAGS_DEFAULT | GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "descriptiveName", m->descriptiveName);

  gr=GWEN_DB_GetGroup(db,
                      GWEN_DB_FLAGS_DEFAULT |
                      GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                      "data");
  assert(gr);
  if (m->toDbFn)
    return m->toDbFn(m, gr);

  return 0;
}



int AH_Medium_FromDB(AH_MEDIUM *m, GWEN_DB_NODE *db){
  const char *p;
  GWEN_DB_NODE *gr;

  /* mediumName */
  p=GWEN_DB_GetCharValue(db, "mediumName", 0, 0);
  if (p) {
    free(m->mediumName);
    m->mediumName=strdup(p);
  }

  /* descriptiveName */
  p=GWEN_DB_GetCharValue(db, "descriptiveName", 0, 0);
  if (p) {
    free(m->descriptiveName);
    m->descriptiveName=strdup(p);
  }

  /* flags */
  m->flags=GWEN_DB_GetIntValue(db, "flags", 0, 0);

  /* call implementation */
  gr=GWEN_DB_GetGroup(db,
                      GWEN_PATH_FLAGS_PATHMUSTEXIST,
                      "data");
  if (gr && m->toDbFn) {
    return m->fromDbFn(m, gr);
  }
  return 0;
}



AH_HBCI *AH_Medium_GetHBCI(const AH_MEDIUM *m){
  assert(m);
  return m->hbci;
}



const char *AH_Medium_GetSecurityIdPtr(const AH_MEDIUM *m){
  assert(m);
  return m->pSecurityId;
}



unsigned int AH_Medium_GetSecurityIdSize(const AH_MEDIUM *m){
  assert(m);
  return m->lSecurityId;
}



void AH_Medium_SetSecurityId(AH_MEDIUM *m,
                             const void *p, unsigned int l){
  assert(m);
  if (m->pSecurityId)
    free(m->pSecurityId);
  m->pSecurityId=0;
  if (l) {
    /* malloc just a single byte more for the trailing zero */
    m->pSecurityId=(char*)malloc(l+1);
    m->lSecurityId=l;
    memmove(m->pSecurityId, p, l);
    /* append a zero for nice C-strings */
    m->pSecurityId[l]=0;
  }
}



const char *AH_Medium_GetPeerAddr(const AH_MEDIUM *m){
  assert(m);
  return m->peerAddr;
}



void AH_Medium_SetPeerAddr(AH_MEDIUM *m,
                                const char *s){
  assert(m);
  free(m->peerAddr);
  if (s) m->peerAddr=strdup(s);
  else m->peerAddr=0;
}



int AH_Medium_GetPeerPort(const AH_MEDIUM *m){
  assert(m);
  return m->peerPort;
}



void AH_Medium_SetPeerPort(AH_MEDIUM *m, int i){
  assert(m);
  m->peerPort=i;
}



void AH_Medium_Link(AH_MEDIUM *m){
  assert(m);
  m->linkCount++;
}



void AH_Medium_Unlink(AH_MEDIUM *m){
  assert(m);
  assert(m->linkCount);
  m->linkCount--;
}



GWEN_TYPE_UINT32 AH_Medium_GetLinkCount(const AH_MEDIUM *m){
  assert(m);
  return m->linkCount;
}



int AH_Medium_SelectContext(AH_MEDIUM *m, int idx){
  int rv;

  assert(m);
  if (m->selectedContext==idx) {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Context already selected");
    return 0;
  }

  m->selectedContext=-1;
  if (m->selectContextFn) {
    rv=m->selectContextFn(m, idx);
    if (rv) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Could not select context");
      return rv;
    }
    m->selectedContext=idx;
    return 0;
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No selectContext function set");
    return -1;
  }
}



int AH_Medium_FindContext(AH_MEDIUM *m,
                          int country,
                          const char *bankId,
                          const char *userId){
  int idx;
  GWEN_BUFFER *cBankId;
  GWEN_BUFFER *cUserId;

  assert(m);
  cBankId=GWEN_Buffer_new(0, 32, 0, 1);
  cUserId=GWEN_Buffer_new(0, 32, 0, 1);

  for (idx=0; ; idx++) {
    int cCountry;
    int rv;

    rv=AH_Medium_ReadContext(m, idx, &cCountry, cBankId, cUserId, 0, 0);
    if (rv) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "No more contexts (returned %d at %d)",
               rv, idx);
      GWEN_Buffer_free(cUserId);
      GWEN_Buffer_free(cBankId);
      return -1;
    }
    if (cCountry==country &&
        strcasecmp(GWEN_Buffer_GetStart(cBankId), bankId)==0 &&
        strcasecmp(GWEN_Buffer_GetStart(cUserId), userId)==0) {
      /* found context */
      DBG_DEBUG(AQHBCI_LOGDOMAIN,
               "Found context %d/%s/%s: %d",
               cCountry,
               GWEN_Buffer_GetStart(cBankId),
               GWEN_Buffer_GetStart(cUserId),
               idx);
      GWEN_Buffer_free(cUserId);
      GWEN_Buffer_free(cBankId);
      return idx;
    }
    GWEN_Buffer_Reset(cBankId);
    GWEN_Buffer_Reset(cUserId);
  } /* for */
}



int AH_Medium_CreateContext(AH_MEDIUM *m,
                            int country,
                            const char *bankId,
                            const char *userId){
  int rv;

  assert(m);
  if (m->createContextFn) {
    rv=m->createContextFn(m, country, bankId, userId);
    if (rv<0) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Could not create context");
    }
    return rv;
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No createContext function set");
    return -1;
  }
}



int AH_Medium_RemoveContext(AH_MEDIUM *m, int idx) {
  int rv;

  assert(m);
  if (m->removeContextFn) {
    rv=m->removeContextFn(m, idx);
    if (rv) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Could not remove context");
    }
    return rv;
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No removeContext function set");
    return -1;
  }
}




int AH_Medium_ReadContext(AH_MEDIUM *m,
                          int idx,
                          int *country,
                          GWEN_BUFFER *bankId,
                          GWEN_BUFFER *userId,
                          GWEN_BUFFER *server,
                          int *port){
  int rv;

  assert(m);
  if (m->readContextFn) {
    rv=m->readContextFn(m, idx, country, bankId, userId, server, port);
    if (rv<0) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Could not read context");
    }
    return rv;
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No readContext function set");
    return -1;
  }
}



int AH_Medium_WriteContext(AH_MEDIUM *m,
                           int idx,
                           int country,
                           const char *bankId,
                           const char *userId,
                           const char *server,
                           int port){
  int rv;

  assert(m);
  if (m->writeContextFn) {
    rv=m->writeContextFn(m, idx, country, bankId, userId, server, port);
    if (rv<0) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Could not write context");
    }
    return rv;
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No writeContext function set");
    return -1;
  }
}





void AH_Medium_SetReadContextFn(AH_MEDIUM *m,
                               AH_MEDIUM_READCONTEXTFN f){
  assert(m);
  m->readContextFn=f;
}



void AH_Medium_SetWriteContextFn(AH_MEDIUM *m,
                                 AH_MEDIUM_WRITECONTEXTFN f){
  assert(m);
  m->writeContextFn=f;
}



void AH_Medium_SetSelectContextFn(AH_MEDIUM *m, AH_MEDIUM_SELECTCONTEXTFN f){
  assert(m);
  m->selectContextFn=f;
}



void AH_Medium_SetCreateContextFn(AH_MEDIUM *m, AH_MEDIUM_CREATECONTEXTFN f){
  assert(m);
  m->createContextFn=f;
}



void AH_Medium_SetRemoveContextFn(AH_MEDIUM *m, AH_MEDIUM_REMOVECONTEXTFN f){
  assert(m);
  m->removeContextFn=f;
}



const char *AH_Medium_TypeToName(AH_MEDIUMTYPE t) {
  switch(t) {
  case AH_MediumTypeDDV:    return "DDV";
  case AH_MediumTypeRDH:    return "RDH";
  case AH_MediumTypePINTAN: return "PINTAN";
  default:                  return "unknown";
  } /* switch */
}



















GWEN_LIST_FUNCTIONS(AH_MEDIUMPROVIDER, AH_MediumProvider);
GWEN_INHERIT_FUNCTIONS(AH_MEDIUMPROVIDER);



AH_MEDIUMPROVIDER *AH_MediumProvider_new(AH_HBCI *hbci,
                                         const char *typeName){
  AH_MEDIUMPROVIDER *mp;

  assert(typeName);
  GWEN_NEW_OBJECT(AH_MEDIUMPROVIDER, mp);
  GWEN_LIST_INIT(AH_MEDIUMPROVIDER, mp);
  GWEN_INHERIT_INIT(AH_MEDIUMPROVIDER, mp);
  mp->mediumTypeName=strdup(typeName);
  mp->hbci=hbci;
  return mp;
}



void AH_MediumProvider_free(AH_MEDIUMPROVIDER *mp){
  if (mp) {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Destroying AH_MEDIUMPROVIDER");
    GWEN_INHERIT_FINI(AH_MEDIUMPROVIDER, mp);
    free(mp->mediumTypeName);
    free(mp->description);
    if (mp->libLoader) {
      GWEN_LibLoader_CloseLibrary(mp->libLoader);
      GWEN_LibLoader_free(mp->libLoader);
    }
    GWEN_LIST_FINI(AH_MEDIUMPROVIDER, mp);
    free(mp);
  }
}



AH_HBCI *AH_MediumProvider_GetHBCI(const AH_MEDIUMPROVIDER *mp){
  assert(mp);
  return mp->hbci;
}



GWEN_LIBLOADER*
AH_MediumProvider_GetLibLoader(const AH_MEDIUMPROVIDER *mp){
  assert(mp);
  return mp->libLoader;
}



void AH_MediumProvider_SetLibLoader(AH_MEDIUMPROVIDER *mp,
                                    GWEN_LIBLOADER *ll){
  assert(mp);
  mp->libLoader=ll;
}




const char*
AH_MediumProvider_GetMediumTypeName(const AH_MEDIUMPROVIDER *mp){
  assert(mp);
  return mp->mediumTypeName;
}



const char*
AH_MediumProvider_GetDescription(const AH_MEDIUMPROVIDER *mp){
  assert(mp);
  return mp->description;
}




void AH_MediumProvider_SetDescription(AH_MEDIUMPROVIDER *mp,
                                      const char *s){
  assert(mp);
  assert(s);
  free(mp->description);
  mp->description=strdup(s);
}



AH_MEDIUM *AH_MediumProvider_Factory(AH_MEDIUMPROVIDER *mp,
                                     AH_HBCI *hbci,
                                     const char *mediumName) {
  assert(mp);
  assert(mp->factoryFn);
  return mp->factoryFn(mp, hbci, mediumName);
}



void AH_MediumProvider_SetFactoryFn(AH_MEDIUMPROVIDER *mp,
                                         AH_MEDIUMPROVIDER_FACTORYFN f){
  assert(mp);
  mp->factoryFn=f;
}



void AH_MediumProvider_SetCheckFn(AH_MEDIUMPROVIDER *mp,
                                  AH_MEDIUMPROVIDER_CHECKFN f){
  assert(mp);
  mp->checkFn=f;
}



void AH_MediumProvider_SetDeviceType(AH_MEDIUMPROVIDER *mp,
                                     AH_MEDIUMDEVICE md){
  assert(mp);
  mp->deviceType=md;
}



AH_MEDIUMDEVICE AH_MediumProvider_GetDeviceType(const AH_MEDIUMPROVIDER *mp){
  assert(mp);
  return mp->deviceType;
}



AH_MEDIUM_CHECKRESULT AH_MediumProvider_Check(AH_MEDIUMPROVIDER *mp,
                                              AH_HBCI *hbci,
                                              GWEN_BUFFER *mediumName) {
  assert(mp);
  if (mp->checkFn)
    return mp->checkFn(mp, hbci, mediumName);
  return AH_MediumCheckResultNotSupported;
}



int AH_Medium_InputPin(AH_MEDIUM *m,
                       char *pwbuffer,
                       int minLen,
                       int maxLen,
                       int flags){
  int rv;
  const char *name;
  const char *numeric_warning = "";
  char buffer[512];

  assert(m);

  buffer[0]=0;
  buffer[sizeof(buffer)-1]=0;
  if (flags & AB_BANKING_INPUT_FLAGS_NUMERIC) {
    numeric_warning = I18N(" You must only enter numbers, not letters.");
  }
  if (flags & AB_BANKING_INPUT_FLAGS_CONFIRM) {
    snprintf(buffer, sizeof(buffer)-1,
	     I18N("Please enter a new password for \n"
		  "%s\n"
		  "The password must be at least %d characters long.%s"
		  "<html>"
		  "Please enter a new password for <i>%s</i>. "
		  "The password must be at least %d characters long.%s"
		  "</html>"),
	     AH_Medium_GetDescriptiveName(m),
	     minLen,
	     numeric_warning,
	     AH_Medium_GetDescriptiveName(m),
	     minLen,
	     numeric_warning);
  }
  else {
    snprintf(buffer, sizeof(buffer)-1,
	     I18N("Please enter the password for \n"
		  "%s\n"
		  "%s<html>"
		  "Please enter the password for <i>%s</i>.%s"
		  "</html>"),
	     AH_Medium_GetDescriptiveName(m),
	     numeric_warning,
	     AH_Medium_GetDescriptiveName(m),
	     numeric_warning);
  }

  name=AH_Medium_GetMediumName(m);
  if (name) {
    GWEN_BUFFER *nbuf;

    nbuf=GWEN_Buffer_new(0, 256 ,0 ,1);
    GWEN_Buffer_AppendString(nbuf, "PASSWORD::");
    GWEN_Buffer_AppendString(nbuf, name);
    rv=AB_Banking_GetPin(AH_HBCI_GetBankingApi(m->hbci),
                         flags,
                         GWEN_Buffer_GetStart(nbuf),
			 I18N("Enter Password"),
			 buffer,
			 pwbuffer,
                         minLen,
                         maxLen);
    GWEN_Buffer_free(nbuf);
  }
  else {
    rv=AB_Banking_InputBox(AH_HBCI_GetBankingApi(m->hbci),
                           flags,
			   I18N("Enter Password"),
                           buffer,
			   pwbuffer,
                           minLen,
                           maxLen);
  }
  return rv;
}



int AH_Medium_SetPinStatus(AH_MEDIUM *m,
                           const char *pin,
                           AB_BANKING_PINSTATUS status){
  const char *name;

  name=AH_Medium_GetMediumName(m);
  if (name) {
    GWEN_BUFFER *nbuf;
    int rv;

    nbuf=GWEN_Buffer_new(0, 256 ,0 ,1);
    GWEN_Buffer_AppendString(nbuf, "PASSWORD::");
    GWEN_Buffer_AppendString(nbuf, name);
    rv=AB_Banking_SetPinStatus(AH_HBCI_GetBankingApi(m->hbci),
                               GWEN_Buffer_GetStart(nbuf),
                               pin,
                               status);
    GWEN_Buffer_free(nbuf);
    return rv;
  }

  return 0;
}



int AH_Medium_InputTan(AH_MEDIUM *m,
		       char *pwbuffer,
                       int minLen,
                       int maxLen){
  int rv;
  const char *name;
  char buffer[512];

  assert(m);

  buffer[0]=0;
  buffer[sizeof(buffer)-1]=0;
  snprintf(buffer, sizeof(buffer)-1,
	   I18N("Please enter the next TAN\n"
		"for %s"
		"<html>"
		"Please enter the next TAN for <i>%s</i>"
		"</html>"),
	   AH_Medium_GetDescriptiveName(m),
	   AH_Medium_GetDescriptiveName(m));

  name=AH_Medium_GetMediumName(m);
  if (name) {
    rv=AB_Banking_GetTan(AH_HBCI_GetBankingApi(m->hbci),
			 name,
			 I18N("Enter TAN"),
			 buffer,
			 pwbuffer,
			 minLen,
			 maxLen);
  }
  else {
    rv=AB_Banking_InputBox(AH_HBCI_GetBankingApi(m->hbci),
			   AB_BANKING_INPUT_FLAGS_NUMERIC|
			   AB_BANKING_INPUT_FLAGS_SHOW,
			   I18N("Enter TAN"),
			   buffer,
			   pwbuffer,
			   minLen,
			   maxLen);
  }
  return rv;
}



int AH_Medium_SetTanStatus(AH_MEDIUM *m,
                           const char *tan,
                           AB_BANKING_TANSTATUS status){
  const char *name;

  name=AH_Medium_GetMediumName(m);
  if (name) {
    return AB_Banking_SetTanStatus(AH_HBCI_GetBankingApi(m->hbci),
                                   name,
                                   tan,
                                   status);
  }

  return 0;
}



int AH_Medium_UI(AH_MEDIUM *m, AH_MEDIUM_UIACTION act){
  int rv;
  char buffer[512];

  assert(m);

  buffer[0]=0;
  buffer[sizeof(buffer)-1]=0;
  switch(act) {
  case AH_Medium_UIActionInsertMedium:
  case AH_Medium_UIActionInsertCorrectMedium:
    if (act==AH_Medium_UIActionInsertMedium) {
      if (m->deviceType==AH_MediumDeviceFile)
	snprintf(buffer, sizeof(buffer)-1,
		 I18N("Please insert the security disc\nfor %s"
		      "<html>"
		      "Please insert the security disc for <i>%s</i>"
		      "</html>"),
		 AH_Medium_GetDescriptiveName(m),
		 AH_Medium_GetDescriptiveName(m));
      else
	snprintf(buffer, sizeof(buffer)-1,
		 I18N("Please insert the chip card\nfor %s"
		      "<html>"
		      "Please insert the chip card for <i>%s</i>"
		      "</html>"),
		 AH_Medium_GetDescriptiveName(m),
		 AH_Medium_GetDescriptiveName(m));
    }
    else {
      if (m->deviceType==AH_MediumDeviceFile)
	snprintf(buffer, sizeof(buffer)-1,
		 I18N("Please insert the correct security disc\nfor %s"
		      "<html>"
		      "Please insert the correct security disc for <i>%s</i>"
		      "</html>"),
		 AH_Medium_GetDescriptiveName(m),
		 AH_Medium_GetDescriptiveName(m));
      else
	snprintf(buffer, sizeof(buffer)-1,
		 I18N("Please insert the correct chip card\nfor %s"
		      "<html>"
		      "Please insert the correct chip card for <i>%s</i>"
		      "</html>"),
		 AH_Medium_GetDescriptiveName(m),
		 AH_Medium_GetDescriptiveName(m));
    }
    rv=AB_Banking_MessageBox(AH_HBCI_GetBankingApi(m->hbci),
			     AB_BANKING_MSG_FLAGS_TYPE_WARN |
                             AB_BANKING_MSG_FLAGS_SEVERITY_DANGEROUS |
                             AB_BANKING_MSG_FLAGS_CONFIRM_B1,
			     I18N("Insert Medium"),
			     buffer,
			     I18N("OK"), I18N("Abort"), 0);
    if (rv==2) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "User aborted");
      AB_Banking_MessageBox(AH_HBCI_GetBankingApi(m->hbci),
                            AB_BANKING_MSG_FLAGS_TYPE_ERROR |
                            AB_BANKING_MSG_FLAGS_SEVERITY_DANGEROUS |
                            AB_BANKING_MSG_FLAGS_CONFIRM_B1,
			    I18N("User Abort"),
			    I18N("Action aborted by user."),
			    I18N("Dismiss"), 0, 0);
      return -1;
    }
    else if (rv!=1) {
      AB_Banking_MessageBox(AH_HBCI_GetBankingApi(m->hbci),
                            AB_BANKING_MSG_FLAGS_TYPE_ERROR |
                            AB_BANKING_MSG_FLAGS_SEVERITY_DANGEROUS |
                            AB_BANKING_MSG_FLAGS_CONFIRM_B1,
                            I18N("Error"),
			    I18N("An internal error occurred."),
			    I18N("Dismiss"), 0, 0);
      return -1;
    }
    return 0;

  case AH_Medium_UIActionStartInputPinViaKeypad:
    snprintf(buffer, sizeof(buffer)-1,
	     I18N("Please enter your PIN into the card reader."
		  "<html>"
		  "Please enter your PIN into the card reader."
		  "</html>"));
    m->showBoxId=AB_Banking_ShowBox(AH_HBCI_GetBankingApi(m->hbci),
				    AB_BANKING_SHOWBOX_FLAGS_BEEP,
				    I18N("Secure PIN Input"),
				    buffer);
    return 0;

  case AH_Medium_UIActionFinishedInputPinViaKeypad:
    AB_Banking_HideBox(AH_HBCI_GetBankingApi(m->hbci), m->showBoxId),
    m->showBoxId=0;
    return 0;
  default:
    return -1;
  }
}



AB_BANKING *AH_Medium_GetBankingApi(const AH_MEDIUM *m){
  assert(m);
  return AH_HBCI_GetBankingApi(m->hbci);
}














