/***************************************************************************
 $RCSfile: jobsingletransfer.c,v $
                             -------------------
    cvs         : $Id: jobsingletransfer.c,v 1.16 2005/04/07 21:38:58 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "jobsingletransfer_p.h"
#include "aqhbci_l.h"
#include "accountjob_l.h"
#include <gwenhywfar/debug.h>
#include <gwenhywfar/misc.h>
#include <gwenhywfar/inherit.h>
#include <gwenhywfar/dbio.h>
#include <gwenhywfar/text.h>

#include <aqbanking/jobsingletransfer.h>
#include <aqbanking/jobsingletransfer_be.h>
#include <aqbanking/jobsingledebitnote_be.h>
#include <aqbanking/job_be.h>

#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>


GWEN_INHERIT(AH_JOB, AH_JOB_SINGLETRANSFER);



/* --------------------------------------------------------------- FUNCTION */
AH_JOB *AH_Job_SingleTransfer_new(AH_CUSTOMER *cu,
                                  AH_ACCOUNT *account) {
  return AH_Job_SingleTransferBase_new(cu, account, 1);
}



/* --------------------------------------------------------------- FUNCTION */
AH_JOB *AH_Job_SingleDebitNote_new(AH_CUSTOMER *cu,
                                   AH_ACCOUNT *account) {
  return AH_Job_SingleTransferBase_new(cu, account, 0);
}



/* --------------------------------------------------------------- FUNCTION */
AH_JOB *AH_Job_SingleTransferBase_new(AH_CUSTOMER *cu,
                                      AH_ACCOUNT *account,
                                      int isTransfer) {
  AH_JOB *j;
  AH_JOB_SINGLETRANSFER *aj;
  GWEN_DB_NODE *dbArgs;

  j=AH_AccountJob_new(isTransfer?"JobSingleTransfer":"JobSingleDebitNote",
                      cu, account);
  if (!j)
    return 0;

  GWEN_NEW_OBJECT(AH_JOB_SINGLETRANSFER, aj);
  GWEN_INHERIT_SETDATA(AH_JOB, AH_JOB_SINGLETRANSFER, j, aj,
                       AH_Job_SingleTransfer_FreeData);
  aj->isTransfer=isTransfer;
  /* overwrite some virtual functions */
  AH_Job_SetProcessFn(j, AH_Job_SingleTransfer_Process);
  AH_Job_SetExchangeFn(j, AH_Job_SingleTransfer_Exchange);

  /* set some known arguments */
  dbArgs=AH_Job_GetArguments(j);
  assert(dbArgs);


  return j;
}



/* --------------------------------------------------------------- FUNCTION */
void AH_Job_SingleTransfer_FreeData(void *bp, void *p){
  AH_JOB_SINGLETRANSFER *aj;

  aj=(AH_JOB_SINGLETRANSFER*)p;

  GWEN_FREE_OBJECT(aj);
}



/* --------------------------------------------------------------- FUNCTION */
int AH_Job_SingleTransfer_Process(AH_JOB *j){
  AH_JOB_SINGLETRANSFER *aj;

  assert(j);
  aj=GWEN_INHERIT_GETDATA(AH_JOB, AH_JOB_SINGLETRANSFER, j);
  assert(aj);
  DBG_INFO(AQHBCI_LOGDOMAIN, "Processing %s",
           (aj->isTransfer)?"JobSingleTransfer":"JobSingleDebitNote");

  return 0;
}



/* --------------------------------------------------------------- FUNCTION */
int AH_Job_SingleTransfer__ValidateTransfer(AB_JOB *bj,
					    AH_JOB *mj,
					    AB_TRANSACTION *t) {
  const GWEN_STRINGLIST *sl;
  int maxn;
  int maxs;
  int n;
  const char *s;
  AH_JOB_SINGLETRANSFER *aj;
  const AB_TRANSACTION_LIMITS *lim;

  assert(mj);
  aj=GWEN_INHERIT_GETDATA(AH_JOB, AH_JOB_SINGLETRANSFER, mj);
  assert(aj);

  if (aj->isTransfer)
    lim=AB_JobSingleTransfer_GetFieldLimits(bj);
  else
    lim=AB_JobSingleDebitNote_GetFieldLimits(bj);

  /* collapse splits if any */
  n=AB_Split_List_GetCount(AB_Transaction_GetSplits(t));
  if (n) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
	      "Transfer contains splits, unable to handle");
    return -1;
  }

  /* check purpose */
  if (lim) {
    maxn=AB_TransactionLimits_GetMaxLinesPurpose(lim);
    maxs=AB_TransactionLimits_GetMaxLenPurpose(lim);
  }
  else {
    maxn=0;
    maxs=0;
  }
  sl=AB_Transaction_GetPurpose(t);
  n=0;
  if (sl) {
    GWEN_STRINGLISTENTRY *se;
    GWEN_STRINGLIST *nsl;
    const char *p;

    nsl=GWEN_StringList_new();
    se=GWEN_StringList_FirstEntry(sl);
    while(se) {
      p=GWEN_StringListEntry_Data(se);
      if (p && *p) {
	char *np;
	int l;
	GWEN_BUFFER *tbuf;

	n++;
	if (maxn && n>maxn) {
	  DBG_ERROR(AQHBCI_LOGDOMAIN,
		    "Too many purpose lines (%d>%d)", n, maxn);
	  GWEN_StringList_free(nsl);
	  return AB_ERROR_INVALID;
	}
	tbuf=GWEN_Buffer_new(0, maxs, 0, 1);
	AB_ImExporter_Utf8ToDta(p, -1, tbuf);
	l=GWEN_Buffer_GetUsedBytes(tbuf);
	if (l>maxs) {
	  DBG_ERROR(AQHBCI_LOGDOMAIN,
		    "Too many chars in purpose line %d (%d>%d)", n, l, maxs);
	  GWEN_StringList_free(nsl);
	  return AB_ERROR_INVALID;
	}
	np=(char*)malloc(l+1);
	memmove(np, GWEN_Buffer_GetStart(tbuf), l+1);
	GWEN_Buffer_free(tbuf);
	/* let string list take the newly alllocated string */
	GWEN_StringList_AppendString(nsl, np, 1, 0);
      }
      se=GWEN_StringListEntry_Next(se);
    } /* while */
    AB_Transaction_SetPurpose(t, nsl);
  }
  if (!n) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No purpose lines");
    return AB_ERROR_INVALID;
  }

  /* check remote name */
  if (lim) {
    maxn=AB_TransactionLimits_GetMaxLinesRemoteName(lim);
    maxs=AB_TransactionLimits_GetMaxLenRemoteName(lim);
  }
  else {
    maxn=0;
    maxs=0;
  }
  sl=AB_Transaction_GetRemoteName(t);
  n=0;
  if (sl) {
    GWEN_STRINGLISTENTRY *se;
    GWEN_STRINGLIST *nsl;
    const char *p;

    nsl=GWEN_StringList_new();
    se=GWEN_StringList_FirstEntry(sl);
    while(se) {
      p=GWEN_StringListEntry_Data(se);
      if (p && *p) {
	char *np;
	int l;
        GWEN_BUFFER *tbuf;

	n++;
	if (maxn && n>maxn) {
	  DBG_ERROR(AQHBCI_LOGDOMAIN,
		    "Too many remote name lines (%d>%d)",
		    n, maxn);
          GWEN_StringList_free(nsl);
	  return AB_ERROR_INVALID;
	}
	tbuf=GWEN_Buffer_new(0, maxs, 0, 1);
        AB_ImExporter_Utf8ToDta(p, -1, tbuf);
	l=GWEN_Buffer_GetUsedBytes(tbuf);
	if (l>maxs) {
	  DBG_WARN(AQHBCI_LOGDOMAIN,
		   "Too many chars in remote name line %d (%d>%d)",
		   n, l, maxs);
          GWEN_StringList_free(nsl);
	  return AB_ERROR_INVALID;
	}
	np=(char*)malloc(l+1);
	memmove(np, GWEN_Buffer_GetStart(tbuf), l+1);
	GWEN_Buffer_free(tbuf);
	/* let string list take the newly alllocated string */
	GWEN_StringList_AppendString(nsl, np, 1, 0);
      }
      se=GWEN_StringListEntry_Next(se);
    } /* while */
    AB_Transaction_SetRemoteName(t, nsl);
  }
  if (!n) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No remote name lines");
    return AB_ERROR_INVALID;
  }

  /* check local name */
  s=AB_Transaction_GetLocalName(t);
  if (!s) {
    AB_ACCOUNT *a;

    DBG_WARN(AQHBCI_LOGDOMAIN,
	     "No local name, filling in");
    a=AB_Job_GetAccount(bj);
    assert(a);
    s=AB_Account_GetOwnerName(a);
    if (!s) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
		"No owner name in account, giving up");
      return AB_ERROR_INVALID;
    }
    AB_Transaction_SetLocalName(t, s);
  }

  /* check local bank code */
  s=AB_Transaction_GetLocalBankCode(t);
  if (!s) {
    AH_ACCOUNT *a;

    DBG_WARN(AQHBCI_LOGDOMAIN,
	     "No local bank code, filling in");
    a=AH_AccountJob_GetAccount(mj);
    assert(a);
    s=AH_Account_GetBankId(a);
    assert(s);
    AB_Transaction_SetLocalBankCode(t, s);
  }

  /* check local account number */
  s=AB_Transaction_GetLocalAccountNumber(t);
  if (!s) {
    AH_ACCOUNT *a;

    DBG_WARN(AQHBCI_LOGDOMAIN,
	     "No local account number, filling in");
    a=AH_AccountJob_GetAccount(mj);
    assert(a);
    s=AH_Account_GetAccountId(a);
    assert(s);
    AB_Transaction_SetLocalAccountNumber(t, s);
  }

  /* check local account suffix */
  s=AB_Transaction_GetLocalSuffix(t);
  if (!s) {
    AH_ACCOUNT *a;

    DBG_INFO(AQHBCI_LOGDOMAIN,
	     "No local suffix, filling in (if possible)");
    a=AH_AccountJob_GetAccount(mj);
    assert(a);
    s=AH_Account_GetSuffix(a);
    if (s)
      AB_Transaction_SetLocalSuffix(t, s);
  }

  /* check text key */
  if (lim) {
    if (GWEN_StringList_Count(AB_TransactionLimits_GetValuesTextKey(lim))){
      char numbuf[32];

      n=AB_Transaction_GetTextKey(t);
      if (n==0) {
	if (aj->isTransfer)
	  n=51; /* "Ueberweisung" */
	else
	  n=5;  /* "Lastschrift" */
	AB_Transaction_SetTextKey(t, n);
      }

      snprintf(numbuf, sizeof(numbuf), "%d", n);
      if (!AB_TransactionLimits_HasValuesTextKey(lim, numbuf)) {
	DBG_ERROR(AQHBCI_LOGDOMAIN, "Text key \"%s\" not supported by bank",
		  numbuf);
	return AB_ERROR_INVALID;
      }
    }
  }

  return 0;
}



/* --------------------------------------------------------------- FUNCTION */
int AH_Job_SingleTransfer_Exchange(AH_JOB *j, AB_JOB *bj,
				   AH_JOB_EXCHANGE_MODE m){
  AH_JOB_SINGLETRANSFER *aj;

  DBG_INFO(AQHBCI_LOGDOMAIN, "Exchanging (%d)", m);

  assert(j);
  aj=GWEN_INHERIT_GETDATA(AH_JOB, AH_JOB_SINGLETRANSFER, j);
  assert(aj);

  if ((aj->isTransfer && AB_Job_GetType(bj)!=AB_Job_TypeTransfer) ||
      (!aj->isTransfer && AB_Job_GetType(bj)!=AB_Job_TypeDebitNote)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "Not a %s job job",
              (aj->isTransfer)?"SingleTransfer":"SingleDebitNote");
    return AB_ERROR_INVALID;
  }

  switch(m) {
  case AH_Job_ExchangeModeParams: {
    GWEN_DB_NODE *dbParams;
    GWEN_DB_NODE *dbTk;
    AB_TRANSACTION_LIMITS *lim;
    int i;

    dbParams=AH_Job_GetParams(j);
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Have this parameters to exchange:");
    if (GWEN_Logger_GetLevel(AQHBCI_LOGDOMAIN)>=GWEN_LoggerLevelDebug)
        GWEN_DB_Dump(dbParams, stderr, 2);
    /* read limits */
    lim=AB_TransactionLimits_new();
    AB_TransactionLimits_SetMaxLenPurpose(lim, 27);
    AB_TransactionLimits_SetMaxLenRemoteName(lim, 27);
    AB_TransactionLimits_SetMaxLinesRemoteName(lim, 2);

    i=GWEN_DB_GetIntValue(dbParams, "maxpurposeLines", 0, 0);
    AB_TransactionLimits_SetMaxLinesPurpose(lim, i);

    /* read text keys */
    dbTk=GWEN_DB_GetGroup(dbParams, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "textkey");
    if (dbTk) {
      for (i=0; ; i++) {
	int k;
	char numbuf[16];

	k=GWEN_DB_GetIntValue(dbTk, "key", i, -1);
	if (k==-1)
	  break;
	snprintf(numbuf, sizeof(numbuf), "%d", k);
	AB_TransactionLimits_AddValuesTextKey(lim, numbuf, 1);
      }
    }

    if (aj->isTransfer)
      AB_JobSingleTransfer_SetFieldLimits(bj, lim);
    else
      AB_JobSingleDebitNote_SetFieldLimits(bj, lim);

    return 0;
  }

  case AH_Job_ExchangeModeArgs: {
    GWEN_DB_NODE *dbArgs;
    const AB_TRANSACTION *ot;
    const AB_VALUE *v;

    dbArgs=AH_Job_GetArguments(j);
    assert(dbArgs);
    if (aj->isTransfer)
      ot=AB_JobSingleTransfer_GetTransaction(bj);
    else
      ot=AB_JobSingleDebitNote_GetTransaction(bj);
    if (ot) {
      GWEN_DB_NODE *dbT;
      const char *p;
      const GWEN_STRINGLIST *sl;
      AB_TRANSACTION *t;

      t=AB_Transaction_dup(ot);
      assert(t);
      if (AH_Job_SingleTransfer__ValidateTransfer(bj, j, t)) {
	DBG_ERROR(AQHBCI_LOGDOMAIN,
		  "Invalid transaction");
	AB_Job_SetStatus(bj, AB_Job_StatusError);
	return AB_ERROR_INVALID;
      }
      /* store the validated transaction back into application job,
       * to allow the application to recognize answers to this job later */
      if (aj->isTransfer)
        AB_JobSingleTransfer_SetTransaction(bj, t);
      else
        AB_JobSingleDebitNote_SetTransaction(bj, t);

      dbT=GWEN_DB_GetGroup(dbArgs, GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                           "transaction");
      assert(dbT);

      /* store transaction */
      GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
			  "ourAccount/country", 280);
      GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
			   "ourAccount/bankCode",
			   AB_Transaction_GetLocalBankCode(t));
      GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                           "ourAccount/accountId",
                           AB_Transaction_GetLocalAccountNumber(t));

      p=AB_Transaction_GetLocalSuffix(t);
      if (p)
	GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
			     "ourAccount/accountsubid", p);
      GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
			   "name",
			   AB_Transaction_GetLocalName(t));

      GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
			  "otherAccount/country",
                          280);
      GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
			   "otherAccount/bankCode",
			   AB_Transaction_GetRemoteBankCode(t));
      GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
                           "otherAccount/accountId",
                           AB_Transaction_GetRemoteAccountNumber(t));

      sl=AB_Transaction_GetRemoteName(t);
      if (sl) {
	GWEN_STRINGLISTENTRY *se;

	se=GWEN_StringList_FirstEntry(sl);
	GWEN_DB_DeleteVar(dbT, "otherName");
	while(se) {
	  p=GWEN_StringListEntry_Data(se);
	  if (p)
	    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_DEFAULT,
				 "otherName", p);
	  se=GWEN_StringListEntry_Next(se);
	} /* while */
      }

      v=AB_Transaction_GetValue(t);
      if (v) {
	GWEN_DB_NODE *dbV;
        GWEN_BUFFER *nbuf;
        char *p;
        const char *s;
        int l;

	dbV=GWEN_DB_GetGroup(dbT, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "value");
        assert(dbV);

	nbuf=GWEN_Buffer_new(0, 32, 0, 1);
	if (GWEN_Text_DoubleToBuffer(AB_Value_GetValue(v),
				     nbuf)) {
	  DBG_ERROR(AQHBCI_LOGDOMAIN, "Buffer overflow");
          GWEN_Buffer_free(nbuf);
	  abort();
	}

	l=GWEN_Buffer_GetUsedBytes(nbuf);
	if (!l) {
	  DBG_ERROR(AQHBCI_LOGDOMAIN, "Error in conversion");
	  GWEN_Buffer_free(nbuf);
	  abort();
        }

        /* replace "C" comma with "DE" comma, remove thousand's comma */
        p=GWEN_Buffer_GetStart(nbuf);
        s=p;
        while(*s) {
          if (*s=='.') {
            *p=',';
            p++;
          }
          else if (*s!=',') {
            *p=*s;
            p++;
          }
          s++;
        } /* while */
        *p=0;

	if (strchr(GWEN_Buffer_GetStart(nbuf), ',')) {
	  /* kill all trailing '0' behind the comma */
	  p=GWEN_Buffer_GetStart(nbuf)+l;
	  while(l--) {
	    --p;
            if (*p=='0')
              *p=0;
            else
              break;
          }
	}
	else
	  GWEN_Buffer_AppendString(nbuf, ",");

	/* store value */
	GWEN_DB_SetCharValue(dbV, GWEN_DB_FLAGS_OVERWRITE_VARS,
                             "value",
			     GWEN_Buffer_GetStart(nbuf));
	GWEN_Buffer_free(nbuf);

	s=AB_Value_GetCurrency(v);
        if (!s)
          s="EUR";
        GWEN_DB_SetCharValue(dbV, GWEN_DB_FLAGS_OVERWRITE_VARS,
                             "currency", s);
      } /* if value */

      GWEN_DB_SetIntValue(dbT, GWEN_DB_FLAGS_OVERWRITE_VARS,
			  "textKey",
			  AB_Transaction_GetTextKey(t));

      sl=AB_Transaction_GetPurpose(t);
      if (sl) {
	GWEN_STRINGLISTENTRY *se;

	se=GWEN_StringList_FirstEntry(sl);
	GWEN_DB_DeleteVar(dbT, "purpose");
	while(se) {
	  p=GWEN_StringListEntry_Data(se);
	  if (p)
	    GWEN_DB_SetCharValue(dbT, GWEN_DB_FLAGS_DEFAULT,
				 "purpose", p);
	  se=GWEN_StringListEntry_Next(se);
	} /* while */
      }
    }
    else {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "No transaction");
      AB_Job_SetStatus(bj, AB_Job_StatusError);
      return AB_ERROR_NO_DATA;
    }
    return 0;
  }

  case AH_Job_ExchangeModeResults: {
    AH_RESULT_LIST *rl;
    AH_RESULT *r;
    int has10;
    int has20;

    rl=AH_Job_GetSegResults(j);
    assert(rl);

    r=AH_Result_List_First(rl);
    if (!r) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "No segment results");
      AB_Job_SetStatus(bj, AB_Job_StatusError);
      return AB_ERROR_NO_DATA;
    }
    has10=0;
    has20=0;
    while(r) {
      if (AH_Result_GetCode(r)==10)
        has10=1;
      else if (AH_Result_GetCode(r)==20)
        has20=1;
      r=AH_Result_List_Next(r);
    }

    if (has20) {
      AB_Job_SetStatus(bj, AB_Job_StatusFinished);
      DBG_INFO(AQHBCI_LOGDOMAIN, "Job finished");
    }
    else if (has10) {
      AB_Job_SetStatus(bj, AB_Job_StatusPending);
      DBG_INFO(AQHBCI_LOGDOMAIN, "Job pending");
    }
    else {
      DBG_INFO(AQHBCI_LOGDOMAIN,
               "Can't determine the status (neither 0010 nor 0020)");
      AB_Job_SetStatus(bj, AB_Job_StatusError);
      return AB_ERROR_NO_DATA;
    }
    return 0;
  }

  default:
    DBG_NOTICE(AQHBCI_LOGDOMAIN, "Unsupported exchange mode");
    return AB_ERROR_NOT_SUPPORTED;
  } /* switch */
}










