/*
 *  playlist.c
 *  mod_musicindex
 *
 *  $Id: playlist.c 662 2006-07-09 21:05:07Z varenet $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *  Copyright (c) 2003-2005 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * Playlist management system.
 *
 * @author Regis Boudin
 * @author Thibaut Varene
 * @version $Revision: 662 $
 * @date 2003-2006
 *
 * This is where the magic takes place. Here are the functions in charge of
 * analyzing the content of the folders requested by the client, and of
 * generating the structures and lists that will be used by the HTML
 * subsystem to generate the pretty output.
 *
 * @todo document the code path as best as possible (especiallly make_music_entry)
 */

#include "playlist.h"
#include "sort.h"

#include "playlist-flac.h"
#include "playlist-vorbis.h"
#include "playlist-mp3.h"
#include "playlist-mp4.h"

#include <stdio.h>	/* fops */
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* access */
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_DIRENT_H
#include <dirent.h>	/* opendir */
#endif

/**
 * Array of common strings widely used.
 *
 * @warning Must be kept in sync with the enum in playlist.h
 *
 * http://filext.com/detaillist.php?extdetail=M3U
 * http://filext.com/detaillist.php?extdetail=MP3
 * http://filext.com/detaillist.php?extdetail=OGG
 * http://filext.com/detaillist.php?extdetail=FLAC
 * http://filext.com/detaillist.php?extdetail=MP4
 */
const struct ftype const filetype[] = {
	{ "MP3", "audio/x-mp3" },
	{ "Vorbis", "audio/x-ogg" },
	{ "FLAC", "audio/flac" },
	{ "MP4/AAC", "audio/mp4" }
};


/**
 * Fallback function if the file could not be recognised.
 *
 * @param pool Pool
 * @param head Head
 * @param in file to parse (closed on normal exit)
 * @param conf MusicIndex configuration paramaters struct
 * @param names Names
 * @param r Apache request_rec struct
 *
 * @return head, whatever happens.
 */
static mu_ent *make_no_entry(request_rec *r, apr_pool_t *pool, FILE *const in,
	mu_ent *const head, const mu_config *const conf, mu_ent_names *const names)
{
	fclose(in);
	return head;
}

/**
 * Wrapper function to call the actuall cache backend.
 *
 * @param r Apache request_rec struct
 * @param pool Apache memory pool
 * @param in file to parse (closed on normal exit)
 * @param head head list entry
 * @param conf MusicIndex configuration parameters
 * @param names Names struct associated with the current in file
 *
 * @return the cache data on success, head otherwise
 */
static mu_ent *make_cache_entry(request_rec *r, apr_pool_t *pool, FILE *const in,
	mu_ent *const head, const mu_config *const conf, mu_ent_names *const names)
{
	if (conf->cache != NULL) {
		if (conf->cache->make_entry != NULL) {
			mu_ent *result = conf->cache->make_entry(r, pool, in, head, conf, names);
			if (result != NULL) {
				result->flags |= EF_INCACHE;
				return result;
			}
		}
	}
	
	return head;
}

/**
 * Function pointers array.
 *
 * Of course, as we defined function pointers, we will use them in a nice
 * array. They will be called in a sequential manner.<br>
 * make_cache_entry() <b>MUST</b> be called first.<br>
 * <i>Note</i> : a make_dir_entry() function to handle directories has
 * already been tried, unfortunately, it implies some bad side effects.
 * Plus, it probably does not make things any quicker and increases the
 * number of function calls. (that's a personnal NB to the developpers,
 * to avoid repeating twice the same havoc as time goes ;o)
 */
static make_entry_ptr make_entry[] = {
	make_cache_entry,
#ifdef	ENABLE_CODEC_VORBIS
	make_ogg_entry,
#endif
#ifdef	ENABLE_CODEC_MP3
	make_mp3_entry,
#endif
#ifdef	ENABLE_CODEC_FLAC
	make_flac_entry,
#endif
#ifdef	ENABLE_CODEC_MP4
	make_mp4_entry,
#endif
	make_no_entry,	/* MUST be last before NULL */
	NULL
};

typedef struct mu_dir {
	void *dir;
	unsigned char handled;
} mu_dir;

/**
 * Custom directory opener.
 * 
 * This is a first shot at the custom opendir function.
 * At the end, This wraps around the various caches for directory
 * management. Depending on what the cache back-end can do, it may add things
 * to the entries list, and generate an internal list of files left to deal
 * with. These files names will be retrieved with musicindex_readdir().
 * We mimic the opendir()/readdir()/closedir() scheme.
 *
 * These functions stay here as static, so th compiler will happily inline them
 * for us.
 *
 * @param r the Apache request
 * @param dir Abstract pointer to our custom directory handler
 * @param entries list pointer (double indirection)
 * @param conf Configuration of the current request
 * @param names structure containing the name of the directory
 * @param soptions the current extended options
 *
 * @return handler for the directory.
 */
static void 
musicindex_opendir(request_rec *r, mu_dir *dir, mu_ent **entries, const mu_config *const conf,
			const mu_ent_names * const names, unsigned long soptions)
{
	/* It's all we need to initialise. Don't add a call to memset */
	dir->dir = NULL;

	/* A cache is configured, try to read directly data from it */
	if (conf->cache != NULL) {
		if (conf->cache->opendir != NULL)
			dir->dir = conf->cache->opendir(r, entries, conf, names, soptions);
	}
	
	/* Cache data could not be used, fall back to the old opendir way */
	if (dir->dir == NULL) {
		dir->dir = opendir(names->filename);
		dir->handled = 0;
	}
	else {
		dir->handled = 1;
	}
}

/**
 * Get the string of the next filename to open.
 *
 * @param dir Abstract pointer to our custom directory handler
 * @param conf Module configuration (to get the current cache backend)
 *
 * @return pointer to the string of the filename, or NULL if there is nothing
 *         left to open.
 */
static const char *musicindex_readdir(mu_dir *dir, const mu_config *const conf)
{
	if (dir->handled == 0 ) {
		struct dirent *dstruct = NULL;
	
		dstruct = readdir((DIR *)(dir->dir));

		if ( NULL == dstruct)
			return NULL;
		else
			return dstruct->d_name;

	}
	else {
		if (conf->cache != NULL) {
			if (conf->cache->readdir != NULL)
				return conf->cache->readdir(dir->dir, conf);
		}
	}
	return NULL;
}

/**
 * Close the open directory
 *
 * @param dir Abstract pointer to our custom directory handler
 * @param conf Module configuration (to get the current cache backend)
 *
 * @return nothing.
 */
static void musicindex_closedir(mu_dir *dir, const mu_config *const conf)
{
	/* A cache is configured, try to read directly data from it */
	if (dir->handled == 0)
		closedir((DIR *)(dir->dir));
	else
		conf->cache->closedir(dir->dir, conf);
}

/**
 * Check lots of details that could mean we shouldn't go through the directory.
 *
 * @param r Apache current request
 * @param conf MusicIndex configuration paramaters struct
 * @param names Names
 *
 * @return 0 if .
 */
static int go_through_directory(request_rec *r, const mu_config *const conf, mu_ent_names *names)
{
	request_rec *sub_req = NULL;
	unsigned short local_options;

	/* When going into subdir, get their local options first */
	sub_req = ap_sub_req_lookup_uri(names->uri, r, NULL);
	if (sub_req == NULL)
		return -1;

	local_options = ((mu_config*)ap_get_module_config(sub_req->per_dir_config, &musicindex_module))->options;
	ap_destroy_sub_req(sub_req);

	/* before dealing with a directory, sanity checks */
	/* First, is the module enabled ? */
	if (!(local_options & MI_ACTIVE))
		return -1;

	/* playall... is stream allowed ? */
	if (((conf->options & MI_STREAMALL) == MI_STREAMALL)
		&& !(local_options & MI_ALLOWSTREAM))
		return -1;

	/* Searching... Is searching allowed ? */
	if ((conf->search) && !(local_options & MI_ALLOWSEARCH))
		return -1;

	/* Everything is fine, go on */
	return 0;
}

/**
 * Add a file (and the content of the directory if requested) to the chain.
 *
 * This function creates a new entry from a file.
 * If the file is a directory and the recursive option is set, all its content
 * is also added.
 * Otherwise, if the file is "recognized", it is simply added.
 *
 * @warning If the filename is too long, it will be ignored.
 * @todo Find a way to not delegate file closing to subsystems (essentially hack vorbis)
 * @bug I'm afraid if we're asked to recurse into too many subdirs, we might
 *	end up with too many open fdesc.
 *
 * @param pool Pool
 * @param r Apache request_rec struct to handle log writings (debugging)
 * @param head Head
 * @param conf MusicIndex configuration paramaters struct
 * @param names Names
 * @param soptions bitwise special options: these are made of the extended options flag set
 *	and are used to fiddle with the algorithms (regis please docu...)
 *
 * @return When possible, struct mu_ent correctly set up
 */
mu_ent *make_music_entry(request_rec *r, apr_pool_t *pool,
	mu_ent *const head, const mu_config *const conf, mu_ent_names *names, unsigned long soptions)
{
	mu_ent			*p = head;
	register unsigned short	i;
	char			*uri;

	if (!names) {
		/* DEV NOTE: filename is the path to the file on the filesystem,
		while parsed_uri.path is the path to the file on the webserver,
		so no assertion can be made on their respective length */
		if ((strlen(r->filename) >= MAX_STRING) || (strlen(r->parsed_uri.path) >= MAX_STRING))
			return head;
		names = ap_palloc(pool, sizeof(mu_ent_names));
		strcpy(names->filename, r->filename);
		strcpy(names->uri, r->parsed_uri.path);
	}

	/* place us at the end of the string for concatenation */
	uri = names->uri + strlen(names->uri);

	while (*(--uri) != '/')
		continue;

	if (*(++uri) == '.')	/* we don't want "invisible" files to show */
		return head;

	if (access(names->filename, R_OK) != 0)
		return head;

	if ((ap_is_directory(pool, names->filename))) {

		request_rec *subreq;
		char *fn = names->filename + strlen(names->filename) - 1;

		if (*fn++ != '/')
			*fn++ = '/';
		*fn = '\0';

		uri = names->uri + strlen(names->uri) - 1;
		if (*uri++ != '/')
			*uri++ = '/';
		*uri = '\0';

		subreq = ap_sub_req_lookup_uri(names->uri, r, NULL);

		soptions &= ~(MI_ALLOWFETCH|MI_ALLOWRSS);
		
		if (subreq != NULL) {
			int local_options = ((mu_config *)ap_get_module_config(subreq->per_dir_config, &musicindex_module))->options;
			int local_rss = ((mu_config *)ap_get_module_config(subreq->per_dir_config, &musicindex_module))->rss_items;
			ap_destroy_sub_req(subreq);
			
			/* XXX pkoi pas soptions = local_options ? */
			soptions |= (local_options & MI_ALLOWFETCH);
			
			if (local_rss > 0)
				soptions |= MI_ALLOWRSS;
		}

		if (soptions & MI_RECURSIVE) {
			/* We were asked to recurse, let's dig in */
			mu_dir			dir_handler;
			const char		*filename;
			unsigned short fn_max = MAX_STRING;
			unsigned short uri_max = MAX_STRING;

			/* We remove the recursive bit at this point.
			   XXX je comprends pas bien pkoi on | conf->options plutot que soptions */
			soptions &= ((~(MI_RECURSIVE)) | (conf->options));

			if ( 0 != go_through_directory(r, conf, names) )
				return head;

			/* Try opening the directory */
			musicindex_opendir(r, &dir_handler, &p, conf, names, soptions);
			if (NULL == dir_handler.dir)
				return head;

			/* DEV NOTE: keep in mind that uri and fn both points to
			the end of their "names->*" counterpart */
			fn_max -= strlen(names->filename);
			uri_max -= strlen(names->uri);

			while ((filename = musicindex_readdir(&dir_handler, conf))) {
				/* Before doing some strcpy, check there is
				   enough space... */
				if ((strlen(filename) >= fn_max) || (strlen(filename) >= uri_max))
					continue;
				strcpy(fn, filename);
				strcpy(uri, filename);
				p = make_music_entry(r, pool, p, conf, names, soptions);
			}
			musicindex_closedir(&dir_handler, conf);
			return p;
		}
		else if (!(conf->options & MI_STREAM) && !(conf->options & MI_RSS) && !(conf->options & MI_TARBALL)) {
			if (!(p = NEW_ENT(pool)))
				return head;
			p->filetype = -1;
		}
	}
	else { /* The file was not a directory */
		FILE *in = fopen(names->filename, "r");

		if (in == NULL)
			return head;

		/* This is the end of adventures for the array of function pointers.
		   We call them sequentially, until a function has recognised
		   it (p!=head), or until we reach the end of the array. In either
		   case, the file will be closed (this is the magic loop) */
		for (i=0; (p == head) && make_entry[i]; i++)
			p = make_entry[i](r, pool, in, head, conf, names);

	}

	/* The file was not recognised, don't go any further */
	if (p == head) {
		return head;
	}

	/* OK, now we have an entry, fill in the missing bits */
	
	p->next = head; /* Add it to the linked list */
	
	p->uri = ap_pstrdup(pool, names->uri); /* The absolute URI will be useful at some point */
	p->file = p->uri; /* Add the relative filename */

	/* convert options into flags - We overwrite whatever was in cache,
	   as it is an abuse to use these EF_ALLOW flags anyway: nothing like that
	   will be cached, as the setting may change while the cache info may still
	   be valid. */
	if (soptions & MI_ALLOWSTREAM)
		p->flags |= EF_ALLOWSTREAM;
	if (soptions & MI_ALLOWDWNLD)
		p->flags |= EF_ALLOWDWNLD;
	if (soptions & MI_ALLOWTARBALL)
		p->flags |= EF_ALLOWTARBALL;	
	if (soptions & MI_ALLOWRSS)
		p->flags |= EF_ALLOWRSS;

	if ((soptions & MI_CUSTOM) == 0)
		p->file += strlen(r->parsed_uri.path);	/* remove the path before the filename except for custom playlists */

	if (conf->options & MI_TARBALL)
		p->filename = ap_pstrdup(pool, names->filename);

	/* Entry is a directory, don't go further */
	if (p->filetype < 0)
		return p;

	if (!(p->title)) {
#ifdef NO_TITLE_STRIP
		p->title = p->file;
#else
		/* Copy the name removing file extension (if any) and changing '_' to ' ' */
		char *const stitle = ap_pstrndup(pool, p->file,
					strrchr(p->file, '.') ?
					(strrchr(p->file, '.') - p->file) : /* <=> strlen(p->file) - strlen(strrchr(p->file, '.')) */
					strlen(p->file));
		
		for (i=0; stitle[i]; i++)
			if (stitle[i] == '_')
				stitle[i] = ' ';
			
		p->title = stitle; 
#endif	/* NO_TITLE_STRIP */
		/* We remove the path prefix if any. */
		if (strrchr(p->title, '/')) {
			p->title = strrchr(p->title, '/');
			p->title++;
		}

	}

	/* We put that here so that we do not create cache files for unhandled file types,
	 * and it's faster to retrieve p->title than to process it again. */
	if ((p->flags & EF_INCACHE) == 0) {
		if (conf->cache != NULL) {
			if (conf->cache->write != NULL)
				conf->cache->write(r, p, conf, names);
		}
	}

	/* Here we go if we have an ongoing search request, and the file we're
	 * looking at wasn't found in the cache. We then fall back to a basic strcasestr()
	 * check, returning only the entries matching the search criteria */
	if ((conf->search) && ((soptions & MI_CUSTOM) == 0)) {
		const char *const file = names->uri + strlen(r->parsed_uri.path);
		if ((ap_strcasestr(file, conf->search))				||
			(p->artist && (ap_strcasestr(p->artist, conf->search)))	||
			(p->album && (ap_strcasestr(p->album, conf->search)))	||
			(p->title && (ap_strcasestr(p->title, conf->search)))		);
				/* do nothing if there's a match */
		else	
			return head;	/* skip if there isn't */
	}

	return p;
}

/**
 * Creates a link list based on the custom list.
 *
 * This function generates a list containing all the files referenced in the
 * conf->custom_list string. It handles parameters in a way so that it can
 * handle indifferently strings for cookies or playlists.
 *
 * @param r Apache request_rec struct
 * @param conf MusicIndex configuration paramaters struct
 *
 * @return 0 on success, error code otherwise.
 */
mu_ent *build_custom_list(request_rec *r, const mu_config *const conf)
{
	request_rec	*subreq;
	mu_ent_names	names;
	const char	*args;
	mu_ent		*mobile_ent = NULL, *custom = NULL, *result = NULL;
	register unsigned short	i;
	char		*p;

	if (conf->custom_list == NULL)
		return NULL;

	args = conf->custom_list;

	if (strncmp(args, "playlist=", 9) == 0)
		args += 9;

	while ((*args != '\0') && (*args != ';')) {
		int local_options;
		p = ap_getword(r->pool, &args, '&');

		if (strncmp("file=", p, 5) == 0)
			p += 5;
		else if (strchr(p, '='))
			continue;

		for (i=0; p[i]; i++) {
			if (p[i] == '+')
				p[i] = ' ';
		}

		ap_unescape_url(p);

		subreq = ap_sub_req_lookup_uri(p, r, NULL);

		if (subreq == NULL)
			continue;

		strcpy(names.uri, subreq->parsed_uri.path);
		strcpy(names.filename, subreq->filename);

		local_options = ((mu_config *)ap_get_module_config(subreq->per_dir_config, &musicindex_module))->options;

		ap_destroy_sub_req(subreq);
		
		custom = make_music_entry(r, r->pool, custom, conf, &names,
			(MI_CUSTOM | (local_options & MI_ALLOWFETCH)));
	}

	for (result = NULL; custom; custom = mobile_ent) {
		mobile_ent = custom->next;
		custom->next = result;
		result = custom;
	}

	return result;
}
