/*
 *  html.c
 *  mod_musicindex
 *
 *  $Id: html.c 662 2006-07-09 21:05:07Z varenet $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2006 Regis BOUDIN
 *  Copyright (c) 2003-2006 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * HTML (and assimilated) code generation
 *
 * @author Regis Boudin
 * @author Thibaut Varene
 * @version $Revision: 662 $
 * @date 2003-2006
 *
 * That file groups all functions related to HTML and assimilated
 * generation, that is to say functions that are used to
 * send data to the client. It is mainly (X)HTML, but also
 * playlist (in M3U format) and RSS.
 *
 * @todo still some work on the custom table
 * @todo Replace text with eye-candy icons wherever possible
 * @todo reduce the size of this file & optimize functions
 * @todo Fix search box collide on long path names
 * @todo Fix translation of form buttons
 */

#include "html.h"
#include "playlist.h"

#include <http_core.h>

#include <time.h> /* For podcast informations */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_DIRENT_H
#include <dirent.h>	/* opendir */
#endif

static const char const Gfavicon[] = "sound.png";
static const char const Gcd_icon[] = "general.png";
static const char const Gfetch_icon[] = "fetch.png";
static const char const Gsound_icon[] = "sound.png";
static const char const Gshuffle_icon[] = "shuffle.png";
static const char const Grss_icon [] = "rss.png";
static const char const Garrow[] = "right_arrow.gif";
static const char const Gsmall_cd_icon[] = "directory.png";
static const char const Gorigcss[] = "musicindex.css";
/** Cover icon names to look for in a directory */
static const char *const Gcovericns[] = {
	"cover.jpg",
	"cover.png",
	"cover.gif",
	"folder.jpg",
	"folder.png",
	"folder.gif",
	NULL
};

/**
 * Formats the list of songs to be sent.
 *
 * This function formats a list of tracks (if any) to be sent
 * to the client, as found in the p list passed in argument.
 * Depending on config options, it allows streaming/shuffle etc.
 *
 * @todo Sort option for recursive search?
 * @todo Keep sort conf on userside (using cookie(s)).
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 * @param customlist Whether the list of entries p is a custom list or not
 *
 * @return The total number of fields in a row.
 */
static unsigned short list_songs(request_rec *r, const mu_ent *const p, const mu_config *const conf, const int customlist)
{
	const mu_ent *q = p;
	register unsigned short fld = 0, j = 0;
	const char *current = NULL, *new = NULL;

	ap_rputs("  <tr class=\"title\">\n", r);

	if ( (customlist != 0) || (conf->options & (MI_ALLOWDWNLD | MI_ALLOWSTREAM)) ||
		((conf->search) && (conf->options & MI_RECURSIVE)))
		ap_rvputs(r, "   <th class=\"Select\">", _("Select"), "</th>\n", NULL);

	/* The sort option is disabled for search result, and this
	   implementation handles consecutive sort requests. TODO: keep
	   memory of sort configuration by the user (either on a per-dir basis,
	   or globally), probably using cookie(s) */

	for (fld = 0; conf->fields[fld]; fld++) {	/* Display title line */
		switch (conf->fields[fld]) {
			case SB_TITLE:
				current = "Title";	/* the style class */
				new = _("Title");	/* the field name */
				break;
			case SB_TRACK:
				current = "Track";
				new = _("Track");
				break;
			case SB_POSN:
				current = "Disc";
				new = _("Disc");
				break;
			case SB_ARTIST:
				current = "Artist";
				new = _("Artist");
				break;
			case SB_LENGTH:
				current = "Length";
				new = _("Length");
				break;
			case SB_BITRATE:
				current = "Bitrate";
				new = _("Bitrate");
				break;
			case SB_FREQ:
				current = "Freq";
				new = _("Freq");
				break;
			case SB_ALBUM:
				current = "Album";
				new = _("Album");
				break;
			case SB_DATE:
				current = "Date";
				new = _("Date");
				break;
			case SB_FILETYPE:
				current = "Filetype";
				new = _("Filetype");
				break;
			case SB_GENRE:
				current = "Genre";
				new = _("Genre");
				break;
			case SB_SIZE:
				current = "Size";
				new = _("Size");
				break;
			default:
				continue;
		}

		/* This test should not be necessary. But, just to be sure... */
		if ((current == NULL) || (new == NULL))
			continue;

		/* With the quickplay option, do not show Length, freq and bitrate */
		if ((conf->options & MI_QUICKPL) && ((conf->fields[fld] == SB_LENGTH) ||
			(conf->fields[fld] == SB_BITRATE) || (conf->fields[fld] == SB_FREQ)))
			continue;

		/* XXX sorting is disabled for search results */
		ap_rvputs(r, "   <th class=\"", current, "\">", NULL);
		if ((customlist == 0) && !(conf->search))
			ap_rprintf(r, "<a href=\"?sort=%c\">%s</a>", conf->fields[fld], new);
		else
			ap_rputs(new, r);
		ap_rputs("</th>\n", r);
	}
	/* at the end of the loop, 'fld' contains the number of fields */

	ap_rputs("  </tr>\n", r);

	/* init value. No chance to see the first strcmp true with that one */
	current = "";
	new = current;

	for (q=p; q!= NULL; q=q->next) {	/* XXX doc */
		if (q->filetype < 0)
			continue;

		if ((conf->search != NULL) && (customlist == 0) &&
			(conf->options & MI_RECURSIVE) == MI_RECURSIVE) {
			new = ap_make_dirstr_parent(r->pool, q->file);

			if (strcmp(current, new)) {
				current = new;
				ap_rputs("  <tr class=\"title\">\n", r);
				ap_rprintf(r, "   <th align=\"left\" colspan=\"%d\">", fld+1);	/* "Select" => +1 */
				if ((current[0] == '\0') || (current[1] == '\0'))	/* current dir is either "" or "/" */
					ap_rputs(_("In Current Directory"), r);
				else
					ap_rvputs(r, _("In "), "<a href=\"", ap_escape_html(r->pool, current), "\">", ap_escape_html(r->pool, current), "</a>", NULL);

				ap_rputs("</th>\n"
					"  </tr>\n", r);
				j = 0;
			}
		}

		if ((j++) & 1)
			ap_rputs("  <tr class=\"odd\">\n", r);
		else
			ap_rputs("  <tr class=\"even\">\n", r);

		/* prepare the "Select" panel, if any */
		if ((customlist != 0) || (conf->options & (MI_ALLOWDWNLD | MI_ALLOWSTREAM)) ||
			((conf->search) && (conf->options & MI_RECURSIVE))) {
			ap_rputs("   <td class=\"Select\">\n", r);

			if (q->flags & EF_ALLOWSTREAM)	/* Display checkbox */
				ap_rvputs(r, "    <input type=\"checkbox\" name=\"file\" value=\"",
					ap_escape_html(r->pool, (customlist != 0) ? q->uri : q->file), "\" />\n", NULL);

			if (customlist == 0) {
				if (q->flags & EF_ALLOWDWNLD)	/* Display [download] */
					ap_rvputs(r, "    <a href=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, q->file)), "\">"
						"<img alt=\"[D]\" title=\"", _("Download"), "\" src=\"", conf->directory, "/",
						Gfetch_icon, "\" /></a>\n", NULL);

				if (q->flags & EF_ALLOWSTREAM)	/* Display [stream] */
					ap_rvputs(r, "    <a href=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, q->file)), "?stream\">"
						"<img alt=\"[S]\" title=\"", _("Stream"), "\" src=\"", conf->directory, "/",
						Gsound_icon, "\" /></a>\n", NULL);
			}

			ap_rputs("   </td>\n", r);
		}

		/* Now fill in the fields */
		for (fld = 0; conf->fields[fld]; fld++) {
			switch (conf->fields[fld]) {
				case SB_TITLE:
					ap_rvputs(r, "   <td class=\"Title\">", ap_escape_html(r->pool, q->title), "</td>\n", NULL);
					break;
				case SB_TRACK:
					if (q->track)
						ap_rprintf(r, "   <td class=\"Track\">%d</td>\n", q->track);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_POSN:
					if (q->posn)
						ap_rprintf(r, "  <td class=\"Disc\">%d</td>\n", q->posn);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_ARTIST:
					ap_rprintf(r, "   <td class=\"Artist\">%s</td>\n", q->artist ? ap_escape_html(r->pool, q->artist) : "");
					break;
				case SB_LENGTH:
					if (conf->options & MI_QUICKPL)
						break;
					if (q->length)
						ap_rprintf(r, "   <td class=\"Length\">%d:%.2d</td>\n", q->length / 60, q->length % 60);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_BITRATE:
					if (conf->options & MI_QUICKPL)
						break;
					if (q->bitrate)
						ap_rprintf(r, "   <td class=\"Bitrate\"><acronym title=\"kbps%s\">%ld</acronym></td>\n", (q->flags & EF_VBR) ? " VBR" : "", q->bitrate >> 10);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_FREQ:
					if (conf->options & MI_QUICKPL)
						break;
					if (q->freq)
						ap_rprintf(r, "   <td class=\"Freq\"><acronym title=\"kHz\">%.1f</acronym></td>\n", (float)q->freq / 1000);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_DATE:
					if (q->date)
						ap_rprintf(r, "   <td class=\"Date\">%d&nbsp;</td>\n", q->date);
					else
						ap_rputs( "   <td></td>\n", r);
					break;
				case SB_ALBUM:
					ap_rprintf(r, "   <td class=\"Album\">%s</td>\n", q->album ? ap_escape_html(r->pool, q->album) : "");
					break;
				case SB_FILETYPE:
					if (q->filetype >= 0)
						ap_rvputs(r, "   <td class=\"Filetype\">", filetype[q->filetype % FT_MAX].nametype, "</td>\n", NULL);
					else
						ap_rputs("   <td></td>\n", r);
					break;
				case SB_GENRE:
					ap_rprintf(r, "   <td class=\"Genre\">%s</td>\n", q->genre ? ap_escape_html(r->pool, q->genre) : "");
					break;
				case SB_SIZE:
					ap_rprintf(r, "   <td class=\"Size\"><acronym title=\"MB\">%.2f</acronym></td>\n", ((float)(q->size>>10))/1024);
					break;
				default:
					break;
			}
		}
		/* at the end of the loop, 'fld' contains the number of fields */
		ap_rputs("  </tr>\n", r);
	}
	return fld;
}

/**
 * Formats a complete url string to be sent to client.
 *
 * This function does all the URL formatting, in the form:
 * http://[user:][passwd\@]hostname[:port]/uri/
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param uri A string (basically a path) to be sent to form a complete url.
 * @param command An eventual command (like ?action=playall).
 * @param conf MusicIndex configuration paramaters struct.
 */
static void send_url(request_rec *r, const char *const uri, const char *const command,
			const mu_config *const conf)
{
	char prefix[MAX_PREFIX];
	char str_port[6];	/* 65536 + '\0' */
	char *bufcoded, *decoded;
	unsigned short l;

	/* If we have an icecast server, it takes precedence, in a 'staticdir' fashion.
	If we're sending an RSS feed, it's up to us to send the ?stream links. */
	if ((conf->iceserver) && !(conf->options & MI_RSS)) {
		/* icecast can only do http, so force that method */
		strcpy(prefix, "http://");

		/* if we only have a port number, we assume icecast is running on the same host */
		if (conf->iceserver[0] == ':')
			strcat(prefix, r->hostname);

		strcat(prefix, conf->iceserver);
	}
	else {
		strcpy(prefix, ap_http_method(r));
		strcat(prefix, "://");

		if (REQUEST_USER(r)) {
			/* grab the auth credentials base64 encoded */
			const char *const auth = ap_table_get(r->headers_in, "Authorization");
			if (auth) {
				bufcoded = strrchr(auth, ' ');
				bufcoded++;
				decoded = (char *)ap_palloc(r->pool, 1 + ap_base64decode_len(bufcoded));
				l = ap_base64decode(decoded, bufcoded);
				strncat(prefix, decoded, l);	/* we have "user:pass" */
			}
			strcat(prefix, "@");
		}

		/* add the hostname */
		strcat(prefix, r->hostname);

		/* add the port number no matter what. This won't hurt, and
		 * checking whether it's really needed or not is just too
		 * much overhead.
		 * ap_get_server_port() should hopefully deal with UseCanonicalName
		 * for us, and thus redirected ports will be properly handled. */
		sprintf(str_port, ":%u", ap_get_server_port(r));
		strcat(prefix, str_port);
	}

	/* add the uri and potential command */
	ap_rvputs(r, prefix, ap_escape_html(r->pool, ap_escape_uri(r->pool, uri)), NULL);
	if (command)
		ap_rputs(command, r);
}

/**
 * Sends HTML page headers and top of the page.
 *
 * This function takes care of the formating of the HTML page headers.
 * It sends the data that is common to all pages generated by the handler.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_head(request_rec *r, const mu_config *const conf)
{
	short i;
	const char *dir;
	char *uri, *u;
	request_rec *subreq = NULL;
#ifdef NEW_HEADER /* New ! :) */
	const mu_config *localconf = NULL;
#endif

	ap_rputs("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" "
		"\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n"
		"<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\">\n"
		"<head>\n"
		" <meta name=\"generator\" content=\"mod_musicindex/" MUSIC_VERSION_STRING "\" />\n", r);

	/* XXX REVISIT: restoring a feature with probably not too shabby code? */
	/* dive into user's customisations, if any */
	if ((subreq = ap_sub_req_lookup_uri(conf->directory, r, NULL)) != NULL) {
	/* XXX enough to catch forbidden access? */
		DIR		*dir;
		struct dirent	*dstruct;
		const char	*ext;

		if ((dir = opendir(subreq->filename))) {
			while ((dstruct = readdir(dir))) {
				if ((dstruct->d_name[0] != '.') &&
					(ext = strrchr(dstruct->d_name, '.')) &&
					(!strncmp(ext+1, "css", 3))) {

					if (!strcmp(dstruct->d_name, Gorigcss))
						continue;	/* this will be dealt with later */
					if (!strcmp(dstruct->d_name, conf->css))
						ap_rputs(" <link rel=\"stylesheet\" title=\"default\"", r);
					else
						ap_rvputs(r, " <link rel=\"alternate stylesheet\" title=\"", dstruct->d_name, "\"", NULL);

					ap_rvputs(r, " type=\"text/css\" href=\"", conf->directory, "/", dstruct->d_name, "\" />\n", NULL);
				}
			}
			closedir(dir);
		}
		ap_destroy_sub_req(subreq);
	}
		
	/* the original CSS */
	if (!strcmp(Gorigcss, conf->css))
		ap_rputs(" <link rel=\"stylesheet\" title=\"default\"", r);
	else
		ap_rputs(" <link rel=\"alternate stylesheet\" title=\"Original\"", r);
	ap_rvputs(r, " type=\"text/css\" href=\"", conf->directory, "/", Gorigcss, "\" />\n", NULL);

	/* RSS link */
	if (conf->rss_items > 0) {
		ap_rputs(" <link rel=\"alternate\" type=\"application/rss+xml\" "
			"title=\"Latest titles\" href=\"?action=RSS\" />\n", r);
		ap_rputs(" <link rel=\"alternate\" type=\"application/rss+xml\" "
			"title=\"Podcast\" href=\"?action=podcast\" />\n", r);
	}


	/* the original favicon */
	ap_rvputs(r, " <link rel=\"shortcut icon\" href=\"", conf->directory, "/", Gfavicon, "\" />\n"
		" <link rel=\"icon\" href=\"", conf->directory, "/", Gfavicon, "\" type=\"image/ico\" />\n"
		" <title>", _("Musical index of"), " ", ap_escape_html(r->pool, r->uri), "</title>\n"
		"</head>\n\n"
		"<body>\n"
		"<!-- begin header -->\n", NULL);

	ap_rputs("<div id=\"header\">\n"
		" <div id=\"mainicon\">\n"
		"  <img alt=\"Dir\" src=\"", r);

	for (i = 0; Gcovericns[i]; i++) {
		if (access(ap_pstrcat(r->pool, r->filename, "/", Gcovericns[i], NULL), R_OK) == 0) {
			ap_rputs(Gcovericns[i], r);
			i = -1;
			break;
		}
		else if (access(ap_pstrcat(r->pool, r->filename, "/.", Gcovericns[i], NULL), R_OK) == 0) {
			ap_rvputs(r, ".", Gcovericns[i], NULL);
			i = -1;
			break;
		}
	}
	
	if (i > 0)
		ap_rvputs(r, conf->directory, "/", Gcd_icon, NULL);

	ap_rputs("\" />\n"
		" </div>\n", r);

	/* Display the box containing the path and the stream links */
	ap_rputs(" <div id=\"maintitle\">\n"
		"  <h1>\n", r);

	uri = ap_pstrdup(r->pool, r->uri);

#ifdef NEW_HEADER /* New ! :) */	/* XXX Could use some doc... :P */
	for (u = uri; *u != '\0'; u++) {
		dir = u;
		while ((*u != '/') && (*u != '\0'))
			u++;

		if (conf->title == NULL) {
			char bk = *(++u);
			*u = '\0';
			subreq = ap_sub_req_lookup_uri(uri, r, NULL);
			*(u--) = bk;
			localconf = (mu_config*)ap_get_module_config(subreq->per_dir_config, &musicindex_module);
		}
		else {
			localconf = conf;
			subreq = NULL;
		}

		if ((localconf->options & MI_ACTIVE) || (localconf->title == NULL)) {
			if ((u == uri) && (localconf->title != NULL))
				dir = localconf->title;

			*u = '\0';
			ap_rvputs(r, "   <a href=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, uri)), "/\">", ap_escape_html(r->pool, dir), "</a>\n", NULL);
			*u = '/';

			if (*(u+1) != '\0')
				ap_rvputs(r, "   <img src=\"", conf->directory, "/",
					Garrow, "\" alt=\"=>\" />\n", NULL);
		}

		if (subreq != NULL)
			ap_destroy_sub_req(subreq);
	}
#else /* Here to have a fallback solution if there is a problem with the new code */
	/* XXX Maybe we can trash this? */
	for (u = uri; *u != '\0'; u++) {
		dir = u;
		while ((*u != '/') && (*u != '\0'))
			u++;

		if (u == uri)
			dir = conf->title;
		else
			ap_rvputs(r, "     <img src=\"", conf->directory, "/",
				Garrow, "\" alt=\"=>\" />\n", NULL);

		*u = '\0';
		ap_rvputs(r, "   <a href=\"", uri, "/\">", dir, "</a>\n", NULL);
		*u = '/';
	}
#endif

	ap_rputs("  </h1>\n", r);

	if (conf->options & MI_ALLOWSTREAM) {
		ap_rvputs(r, "  <a class=\"shuffle\" "
			"href=\"?option=recursive&amp;option=shuffle&amp;action=playall\">",
			"<img alt=\"[R]\" title=\"", _("Shuffle All"), "\" src=\"",
			conf->directory, "/", Gshuffle_icon,
			"\" /></a>\n"
			"  <a class=\"stream\" "
			"href=\"?option=recursive&amp;action=playall\">",
			"<img alt=\"[S]\" title=\"", _("Stream All"), "\" src=\"",
			conf->directory, "/", Gsound_icon,
			"\" /></a>\n", NULL);
	}

	if (conf->options & MI_ALLOWTARBALL) {
		ap_rvputs(r, "  <a class=\"tarball\" "
			"href=\"?option=recursive&amp;action=tarball\">",
			"<img alt=\"[D]\" title=\"", _("Download All"), "\" src=\"",
			conf->directory, "/", Gfetch_icon,
			"\" /></a>\n", NULL);
	}

	if (conf->rss_items > 0) {
		ap_rvputs(r, "    <a class=\"rss\" "
			"href=\"?action=RSS\">",
			"<img alt=\"[RSS]\" title=\"", _("RSS"), "\" src=\"",
			conf->directory, "/", Grss_icon,
			"\" /></a>\n", NULL);
	}

	/* XXX to integrate in a better way */
	ap_rvputs(r, "    <br /><a class=\"rss\" "
			"href=\"?action=randomdir\">[",
			_("Random subdirectory..."), "]</a>\n", NULL);

	ap_rputs(" </div>\n", r);

	/* displays a search box if option is activated */
	/* XXX collision avec le header, a regler */
 	if (conf->options & MI_ALLOWSEARCH) {
		ap_rvputs(r,
			" <form method=\"post\" action=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, r->uri)), "\""
				" enctype=\"application/x-www-form-urlencoded\""
				" id=\"searching\">\n"
			"  <p>\n"
			"   <input type=\"text\" name=\"search\" />\n"
			"   <br />\n"
			"   <input type=\"submit\" name=\"action\" value=\"Search\" />\n"
			"   <input type=\"submit\" name=\"action\" value=\"Recursive Search\" />\n"	/* XXX youpi on peut pas i18n-iser... */
			"   <input type=\"hidden\" name=\"action\" value=\"Search\" />\n"
			"  </p>\n"
			" </form>\n",
			NULL);
	}

	ap_rputs("</div>\n"
		"<hr />\n"
		"<!-- end header -->\n\n", r);

}

/**
 * Sends directory listing for the current folder.
 *
 * This function takes care of preparing and sending to the client the
 * list (if any) of the available directories in the current folder.
 * Depending on config options, it allows streaming/shuffle etc.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_directories(request_rec *r, const mu_ent *const p, const mu_config *const conf)
{
	const mu_ent *q;
	unsigned short dircnt = 0, nb = 0;
	unsigned register short i;
	char temp[MAX_STRING];

	/* We count the number of directories for later use */
	for (q = p; q && (q->filetype < 0); q = q->next)
		nb++;

	if (nb == 0)
		return;

	ap_rputs("<!-- begin subdirs -->\n<h2>",r);
	ap_rprintf(r, _("Music Directories (%d)"), nb);
	ap_rputs("</h2>\n\n<table id=\"directories\">\n", r);

	for (q=p; q && (q->filetype < 0); q = q->next) {
		char *tend = ap_cpystrn(temp, q->file, MAX_STRING-1);
		/* Copies at most numbytes of str to buf. Differs from strncpy()
		in that buf is always null terminated, but is not null filled.
		Therefore, buf should always be at least numbytes + 1 bytes long.
		Returns a pointer to the terminating '\0'. */

		/* remove trailing "/" */
		tend--;
		if (*tend == '/')
			*tend = '\0';

		/* remove underscores from directories display to help formatting (wrapping) */
		for (i=0; temp[i]; i++)
			if (temp[i] == '_')
				temp[i] = ' ';

		/* start a new directories row */
		if (dircnt++ == 0)
			ap_rputs(" <tr>\n", r);

		ap_rvputs(r, "  <td>\n"
			"   <a href=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, q->file)), NULL);

		ap_rputs("\"><img alt=\"\" src=\"", r);
#ifdef SHOW_THUMBNAILS
/* XXX tv: This is just wrong. It doesn't rescale the picture, hence, loading 20 dirs with a 200kB pic each is AWFUL
Not mentionning the fact that letting the browser do the scaling leads often to ugly and unreadable results.
That's IMHO a bad feature */
		/* TODO : optimize to use less memory */
		if (access(ap_pstrcat(r->pool, r->filename, "/", q->file, "cover.png", NULL), R_OK) == 0)
			ap_rvputs(r, q->file, "cover.png", NULL);
		else if (access(ap_pstrcat(r->pool, r->filename, "/", q->file, "cover.jpg", NULL), R_OK) == 0)
			ap_rvputs(r, q->file, "cover.jpg", NULL);
		else if (access(ap_pstrcat(r->pool, r->filename, "/", q->file, "cover.gif", NULL), R_OK) == 0)
			ap_rvputs(r, q->file, "cover.gif", NULL);
		else
#endif
			ap_rvputs(r, conf->directory, "/", Gsmall_cd_icon, NULL);

		ap_rputs("\" /></a>\n", r);

		ap_rvputs(r, "   <div>\n"
			"    <a href=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, q->file)), "\">",
	    		temp, "</a><br />\n", NULL);

		/* show various useful links when needed */
		if (q->flags & EF_ALLOWSTREAM) {
			ap_rvputs(r, "    <a class=\"shuffle\" href=\"",
				ap_escape_uri(r->pool, q->file),
				"?option=recursive&amp;option=shuffle&amp;action=playall\">",
				"<img alt=\"[R]\" title=\"", _("Shuffle"), "\" src=\"",
				conf->directory, "/", Gshuffle_icon,
				"\" /></a>\n", NULL);

			ap_rvputs(r, "    <a class=\"stream\" href=\"",
				ap_escape_uri(r->pool, q->file),
				"?option=recursive&amp;action=playall\">",
				"<img alt=\"[S]\" title=\"", _("Stream"), "\" src=\"",
				conf->directory, "/", Gsound_icon,
				"\" /></a>\n", NULL);
		}

		if (q->flags & EF_ALLOWTARBALL) {
			ap_rvputs(r, "    <a class=\"tarball\" href=\"",
				ap_escape_uri(r->pool, q->file),
				"?option=recursive&amp;action=tarball\">",
				"<img alt=\"[D]\" title=\"", _("Download"), "\" src=\"",
				conf->directory, "/", Gfetch_icon,
				"\" /></a>\n", NULL);
		}

		if (q->flags & EF_ALLOWRSS) {
			ap_rvputs(r, "    <a class=\"rss\" href=\"",
				ap_escape_uri(r->pool, q->file),
				"?action=RSS\">",
				"<img alt=\"[RSS]\" title=\"", _("RSS"), "\" src=\"",
				conf->directory, "/", Grss_icon,
				"\" /></a>\n", NULL);
		}

		ap_rputs("   </div>\n"
			"  </td>\n",r);

		/* end the directories row */
		if (dircnt == conf->dir_per_line) {
			dircnt = 0;
			ap_rputs(" </tr>\n", r);
		}
	}

	/* if we haven't ended on a dir_per_line row, close it here */
	if (dircnt != 0)
		ap_rputs( "</tr>\n", r);

	ap_rputs("</table>\n<hr />\n<!-- end subdirs -->\n\n", r);
}

/**
 * Sends track listing for the current folder.
 *
 * This function takes care of preparing and sending to the client the
 * list (if any) of the available songs in the current p list.
 * Depending on config options, it allows streaming/shuffle etc.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_tracks(request_rec *r, const mu_ent *const p, const mu_config *const conf)
{
	const mu_ent *q;
	register unsigned short nb = 0;
	unsigned short fld;

	/* We count the number of songs for later use */
	for (q = p; q; q = q->next) {
		if (q->filetype >= 0)	/* XXX try to get rid of this test: on peut avoir des 'dirs' ou des fichiers non reconnus dans cette liste de mu_ents? */
			nb++;
	}

	if (nb == 0)
		return;

	ap_rputs("<!-- begin tracks -->\n<h2>", r);

	if (conf->search)
		ap_rprintf(r, _("Result List (%d)"), nb);
	else
		ap_rprintf(r, _("Song List (%d)"), nb);

	ap_rputs("</h2>\n\n", r);

	ap_rvputs(r, "<form method=\"post\" action=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, r->uri)), "\" "
		"enctype=\"application/x-www-form-urlencoded\" id=\"tracks\">\n", NULL);

	ap_rputs(" <table>\n", r);

	fld = list_songs(r, p, conf, 0);

	/* Do NOT show 'select all' if there's nothing to select... */
	if ((conf->search) && (conf->options & MI_ALLOWSTREAM)) {
		ap_rprintf(r, "  <tr class=\"title\"><th align=\"left\" colspan=\"%d\">\n", fld+1); /* Select => +1 */
		ap_rputs("   <input type=\"checkbox\" name=\"all\" onClick=\"for(var i=0;i<this.form.elements.length;i++){var inpt=this.form.elements[i];"
			"var m=inpt.name.match(/-/g);if((inpt.name.substr(0,4)=='file') && (m<1)) inpt.checked=this.form.all.checked}\" />\n"
			"Select All</th>\n"
			"</tr>\n", r);
	}

	ap_rputs(" </table>\n", r);

	/* Do NOT show buttons if we do not allow anything but listing or downloading! */
	if (conf->options & (MI_ALLOWSTREAM|MI_ALLOWTARBALL)) {
		ap_rvputs(r, " <div>\n"
			"  <input type=\"hidden\" name=\"sort\" value=\"", conf->order,"\" />\n"
			"  <input type=\"submit\" name=\"action\" value=\"Add To Playlist\" class=\"playlist\" />\n", NULL); /* XXX i18n DTC */

		if (conf->search == NULL) {
			ap_rputs("  <input type=\"submit\" name=\"action\" value=\"Add All To Playlist\" class=\"playlist\" />\n", r);
			if (conf->options & MI_ALLOWSTREAM)
				ap_rputs("  <input type=\"submit\" name=\"action\" value=\"Shuffle All\" />\n"
				"  <input type=\"submit\" name=\"action\" value=\"Play All\" />\n", r);	/* XXX idem */
		}

		if (conf->options & MI_ALLOWSTREAM)
			ap_rputs("  <input type=\"submit\" name=\"action\" value=\"Play Selected\" />\n", r);

		if (conf->options & MI_ALLOWTARBALL)
			ap_rputs("  <input type=\"submit\" name=\"action\" value=\"Download Selected\" />\n"
				"  <input type=\"submit\" name=\"action\" value=\"Download All\" />\n", r);

		ap_rputs(" </div>\n", r);	/* XXX et encore */
	}

	ap_rputs("</form>\n"
		"<hr />\n"
		"<!-- end tracks -->\n\n", r);
}

/**
 * Sends track listing for the current custom playlist.
 *
 * This function takes care of preparing and sending to the client the
 * list (if any) of the available songs in the current custom p list,
 * retrieved from a cookie.
 * Depending on config options, it allows streaming/shuffle etc.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_customlist(request_rec *r, const mu_ent *const p, const mu_config *const conf)
{
	const mu_ent *q;
	register unsigned short nb = 0;

	if (p == NULL)
		return;

	/* We count the number of songs for later use */
	for (q = p; q; q = q->next)
		nb++;

	ap_rputs("<!-- begin custom -->\n"
		"<h2>", r);
	ap_rprintf(r, _("Custom Playlist (%d)"), nb);
	ap_rputs("</h2>\n\n", r);

	ap_rvputs(r, " <form method=\"post\" action=\"", ap_escape_html(r->pool, ap_escape_uri(r->pool, r->uri)), "\" "
		"enctype=\"application/x-www-form-urlencoded\" id=\"custom\">\n", NULL);

	ap_rputs("  <table>\n", r);

	list_songs(r, p, conf, 1);

	ap_rputs("  </table>\n"
		"  <div>\n", r);

	ap_rputs("   <input type=\"submit\" name=\"action\" value=\"Remove from Playlist\" class=\"playlist\" />\n"
		"   <input type=\"submit\" name=\"action\" value=\"Clear Playlist\" class=\"playlist\" />\n", r);

	if ((conf->options & MI_ALLOWSTREAM) != 0)
	ap_rputs("   <input type=\"submit\" name=\"action\" value=\"Stream Playlist\" class=\"playlist\" />\n", r);

	if ((conf->options & MI_ALLOWTARBALL) != 0)
	ap_rputs("   <input type=\"submit\" name=\"action\" value=\"Download Playlist\" class=\"playlist\" />\n", r);

	ap_rputs("  </div>\n"
		" </form>\n"
		"<hr />\n"
		"<!-- end custom -->\n\n", r);
}

/**
 * Sends playlist with url and EXTM3U info to the client.
 *
 * This function prepares an m3u playlist (with extra information where
 * available: length, artist, title, album) and sends it to the client.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param p struct mu_ent.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_playlist(request_rec *r, const mu_ent *const p, const mu_config *const conf)
{
	const mu_ent *q = p;

	if (!p)
		return;

	ap_rputs("#EXTM3U\n", r);

	for (q = p; q; q = q->next) {
		ap_rprintf(r, "#EXTINF:%i,", q->length);
		if (q->artist)
			ap_rvputs(r, q->artist, " - ", NULL);
		ap_rvputs(r, q->title, NULL);
		if (q->album)
			ap_rvputs(r, " (", q->album, ")", NULL);
		ap_rputc('\n', r);
		send_url(r, q->uri, NULL, conf);
		ap_rputc('\n', r);
	}
}

/**
 * Sends RSS feed.
 *
 * see http://blogs.law.harvard.edu/tech/rss
 * and http://feedvalidator.org/docs/rss2.html
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param p A list of music entries.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_rss(request_rec *r, const mu_ent *const p, const mu_config *const conf)
{
	const mu_ent *q;
	char date_buf[32];
	struct tm time_buf;
	unsigned short limit = conf->rss_items;

	if (!p)
		return;

	ap_rputs("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		"<rss ", r);

	if (conf->options & MI_PODCAST)
		ap_rputs("xmlns:itunes=\"http://www.itunes.com/dtds/podcast-1.0.dtd\" ", r);

	ap_rvputs(r, "version=\"2.0\">\n"
		" <channel>\n"
		"  <title>RSS Feed for ", ap_escape_html(r->pool, r->uri), "</title>\n"
		"  <link>", NULL);
	send_url(r, r->uri, NULL, conf);
	ap_rputs("</link>\n", r);
	ap_rprintf(r, "  <description>%d most recent songs from %s</description>\n", conf->rss_items, ap_escape_html(r->pool, r->uri));

	if ((conf->options & MI_PODCAST) != 0)
		ap_rprintf(r, "  <itunes:summary>%d most recent songs from %s</itunes:summary>\n", conf->rss_items, ap_escape_html(r->pool, r->uri));

	ap_rputs("  <generator>mod_musicindex/" MUSIC_VERSION_STRING "</generator>\n"
		"  <docs>http://backend.userland.com/rss</docs>\n"
		"  <ttl>60</ttl>\n",
		r);

	for (q = p; limit && q; q = q->next) {
		if (q->filetype < 0)	/* XXX try to get rid of this test */
			continue;

		ap_rvputs(r, "  <item>\n"
			"   <title>", ap_escape_html(r->pool, q->title), "</title>\n", NULL);

		if (conf->options & MI_ALLOWSTREAM) {
			ap_rputs("   <link>", r);
			send_url(r, q->uri, "?stream", conf);
			ap_rputs("</link>\n", r);
		}

		if (conf->options & MI_ALLOWDWNLD) {
			ap_rputs("   <enclosure url=\"", r);
			send_url(r, q->uri, NULL, conf);
			ap_rprintf(r, "\" length=\"%lu\" type=\"%s\" />\n",
				q->size, filetype[q->filetype % FT_MAX].mimetype);

			if (conf->options & MI_PODCAST) {
				ap_rputs("   <guild>", r);
				send_url(r, q->uri, NULL, conf);
				ap_rputs("</guild>\n", r);
				localtime_r(&q->mtime, &time_buf);
				strftime(date_buf, 32, "%a, %e %b %Y %H:%M:%S %z", &time_buf);
				ap_rprintf(r, "   <pubDate>%s</pubDate>\n", date_buf);
				if (q->length)
					ap_rprintf(r, "<itunes:duration>%u:%.2u</itunes:duration>",
							q->length / 60, q->length % 60);
			}
		}
		ap_rputs("   <description>\n", r);
		ap_rputs("    Artist | Album | Track | Disc | Length | Genre | Bitrate | Freq | Type | Size\n", r);
		ap_rprintf(r, "    %s | ", q->artist ? ap_escape_html(r->pool, q->artist) : "");
		ap_rprintf(r, "%s | ", q->album ? ap_escape_html(r->pool, q->album) : "");
		if (q->track)
			ap_rprintf(r, "%u", q->track);
		ap_rputs(" | ", r);
		if (q->posn)
			ap_rprintf(r, "%u", q->posn);
		ap_rputs(" | ", r);
		if (q->length)
			ap_rprintf(r, "%u:%.2u", q->length / 60, q->length % 60);
		ap_rputs(" | ", r);
		ap_rprintf(r, "%s | ", q->genre ? ap_escape_html(r->pool, q->genre) : "");
		if (q->bitrate)
			ap_rprintf(r, "%lu %s", q->bitrate >> 10, (q->flags & EF_VBR) ? "VBR" : "");
		ap_rputs(" | ", r);
		if (q->freq)
			ap_rprintf(r, "%u", q->freq);
		ap_rputs(" | ", r);
		ap_rprintf(r, "%s", q->filetype ? filetype[q->filetype % FT_MAX].nametype : "");
		ap_rputs(" | ", r);
		ap_rprintf(r, "%lu\n", q->size);
		ap_rputs("   </description>\n"
			"  </item>\n", r);
		
		limit--;
	}
	ap_rputs(" </channel>\n"
		"</rss>\n", r);

}

/**
 * Sends a random directory location.
 *
 * This function constructs a new URL relative to the current directory,
 * corresponding to a random subdirectory. It might stop at an arbitrary
 * sublevel, even at the current dir.
 *
 * XXX A better (or at least different) implementation would only accounts for
 * actual folders, and setup a random threshold level at which the function stops
 * recursing.
 *
 * @bug A forseable bug is that this function can send you to a directory which
 * access may be denied by .htaccess
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_randomdir(request_rec *r, const mu_config *const conf)
{
	const char *filename, *uri, *nextfile, *nexturi;
	unsigned short nb, test, i;
	unsigned int seed = time(NULL);
	DIR *dir;
	struct dirent *dirent;

	filename = ap_pstrdup(r->pool, r->filename);
	uri = ap_pstrdup(r->pool, r->uri);
	
	while (1) {
		/* XXX this is ugly */
		filename = ap_pstrcat(r->pool, filename, "/", NULL);
	
		/* open the current directory for scanning */
		dir = opendir(filename);
		
		/* XXX TODO optimize the loops */
		/* count the number of *valid* dir entries */
		nb = 0;
		while ((dirent = readdir(dir)))
			if ((dirent->d_name[0] != '.') && (access(r->filename, R_OK|X_OK) == 0))
				nb++;
		
		/* once it's done, go back to the begining of the dir stream */
		rewinddir(dir);
		
		/* generate a random position, using high-order bits. */
		test = (int)(1.0*nb*rand_r(&seed)/(RAND_MAX+1.0));

		/* cycle through the current dir until we've reached EOF or random position */
		i = 0;
		while((i <= test) && (dirent = readdir(dir)))
			if ((dirent->d_name[0] != '.') && (access(r->filename, R_OK|X_OK) == 0))
				i++;
			
		/* construct a new filename with the current dirent */
		nextfile = ap_pstrcat(r->pool, filename, dirent->d_name, NULL);
		nexturi = ap_pstrcat(r->pool, uri, dirent->d_name, NULL);

		/* done with the dir listing, close it */
		closedir(dir);
		
		/* if the new filename is a directory, start the loop again, else, break */
		if (!ap_is_directory(r->pool, nextfile)) {
			/* XXX this is probably not good either */
			ap_table_setn(r->headers_out, "Location", ap_construct_url(r->pool, ap_escape_uri(r->pool, uri), r));
			return;
		}
		else {
			/* XXX gack! */
			filename = nextfile;
			uri = ap_pstrcat(r->pool, nexturi, "/", NULL);
		}
	}
}

/**
 * Sends HTML page footers.
 *
 * This function takes care of the formating of the HTML page footers.
 * It sends the data that is common to all pages generated by the handler.
 * Currently it relies on apache_mp3 original stylesheets.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param conf MusicIndex configuration paramaters struct.
 */
void send_foot(request_rec *r, const mu_config *const conf)
{
	ap_rputs("<!-- begin footer -->\n"
		"<!-- mod_musicindex v." MUSIC_VERSION_STRING " -->\n"
		"<!-- Authors: " MUSIC_AUTHORS_STRING " -->\n", r);
	ap_rputs("<div id=\"footer\">\n"
		" <div id=\"valid\">\n", r);
	/* Should be possible to use on-site validation icons */
	ap_rputs("  <a href=\"http://validator.w3.org/check?uri=referer\">\n"
		"   <img src=\"http://www.w3.org/Icons/valid-xhtml11\"\n"
        	"    alt=\"Valid XHTML 1.1!\" height=\"31\" width=\"88\" />\n"
		"  </a>\n", r);
	ap_rputs("  <a href=\"http://jigsaw.w3.org/css-validator/check/referer\">\n"
		"   <img src=\"http://jigsaw.w3.org/css-validator/images/vcss\"\n"
		"    alt=\"Valid CSS!\" height=\"31\" width=\"88\" />\n"
		"  </a>\n", r);
	ap_rputs("  <a href=\"http://rss.scripting.com/\">\n"
		"   <img src=\"http://cyber.law.harvard.edu/blogs/gems/tech/loveRss.gif\"\n"
		"    width=\"65\" height=\"59\" alt=\"Valid RSS feed.\" />\n"
		"  </a>\n", r);
	ap_rputs(" </div>\n"
		" <div id=\"name\">"
		"<a href=\"http://freshmeat.net/projects/musicindex/\">"
		"MusicIndex v." MUSIC_VERSION_STRING "</a></div>\n"
		"</div>\n", r);
	ap_rputs("<!-- end footer -->\n\n"
		"</body>\n"
		"</html>", r);
}
