/*
 *  playlist.c
 *  mod_musicindex
 *
 *  $Id: playlist.c,v 1.81 2004/05/29 11:26:54 varenet Exp $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *  Copyright (c) 2003-2004 Thibaut VARENE
 *   
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 */

/**
 * @file 
 * Playlist management system.
 *
 * @author Regis Boudin
 * @author Thibaut Varene
 * @version $Revision: 1.81 $
 * @date 2003-2004
 *
 * This is where the magic takes place. Here are the functions in charge of
 * analyzing the content of the folders requested by the client, and of
 * generating the structures and lists that will be used by the HTML
 * subsystem to generate the pretty output.
 */

#include "playlist.h"
#include "html.h"
#include "inf.h"

#include "cache.h"
#include "playlist-ogg.h"
#include "playlist-mp3.h"

/** mime types handled */
const char *handlers[] = {
	"audio/mpeg",
	"application/ogg",
	"audio/x-ogg", /* At some point, we should be able to remove this one */
	NULL
};

/**
 * Function pointer type, as we need it.
 *
 * Here comes the declaration of the funny part... We will (again) use
 * function pointers which will allocate a new entry  from the apache pool,
 * fill it, and close the "in" file if they recognise the file.
 */
typedef mu_ent* (*make_entry_ptr)(apr_pool_t*, mu_ent *, FILE *,
				mu_config *, mu_ent_names *, request_rec *);

/**
 * Function pointers array.
 *
 * Of course, as we defined function pointers, we will use them in a nice
 * array. They will be called in a sequential manner.<br>
 * make_cache_entry() <b>MUST</b> be called first.<br>
 * <i>Note</i> : a make_dir_entry() function to handle directories has 
 * already been tried, unfortunately, it implies some bad side effects.
 * Plus, it probably does not make things any quicker and increases the
 * number of function calls. (that's a personnal NB to the developpers,
 * to avoid repeating twice the same havoc as time goes ;o)
 */
static make_entry_ptr make_entry[] = {
#ifndef NO_CACHE
	make_cache_entry,
#endif
	make_ogg_entry,
	make_mp3_entry,
	NULL
};

/**
 * Creates a new musical entry (mp3, ogg or directory).
 *
 * @param pool Apache pool linked to the current request
 * @param head music entry structure which will follow this one in the list
 *
 * @return A new mu_ent
 */
mu_ent *new_ent(apr_pool_t *pool, mu_ent *head)
{
	mu_ent *p = (mu_ent *)ap_pcalloc(pool, sizeof(mu_ent));
	p->next = head;
	return p;
}

/**
 * Add a file (and the content of the directory if requested) to the chain.
 *
 * This function creates a new entry from a file.
 * If the file is a directory and the recursive option is set, all its content
 * is also added.
 * If the file is an ogg vorbis or an mp3, it is simply added.
 * 
 * @param pool Pool
 * @param r Apache request_rec struct to handle log writings (debugging)
 * @param head Head
 * @param conf MusicIndex configuration paramaters struct
 * @param names Names
 *
 * @return When possible, struct mu_ent correctly set up
 */
static mu_ent *make_music_entry(apr_pool_t *pool, request_rec *r,
	mu_ent *head, mu_config *conf, mu_ent_names *names)
{
	DIR 			*dir;
	struct dirent		*dstruct;
	mu_ent			*p = head;
	FILE			*in = NULL;
	unsigned short		i;
	char			*fn, *uri;
	
	if (!names) {
		if (strlen(r->filename) >= MAX_STRING)
			return head;
		if (strlen(r->parsed_uri.path) >= MAX_STRING)
			return head;
		names = ap_palloc(r->pool, sizeof(mu_ent_names));
		strcpy(names->filename, r->filename);
		strcpy(names->uri, r->parsed_uri.path);
	}

	names->create_cache_file = 0;

	uri = names->uri + strlen(names->uri);

	while (*(--uri) != '/');
	if (*(++uri) == '.')	/* we don't want "invisible" files to show */	
		return head;

	in = fopen(names->filename, "r");
	
	if (in == NULL)
		return head;

	if ((ap_is_directory(pool, names->filename))) {
		
		fclose(in);
		
		fn = names->filename + strlen(names->filename) - 1;
		if (*fn++ != '/')
			*fn++ = '/';
		*fn = '\0';
		
		uri = names->uri + strlen(names->uri) - 1;
		if (*uri++ != '/')
			*uri++ = '/';
		*uri = '\0';
		
		if (conf->options & MI_RECURSIVE) {
			request_rec *sub_req = NULL;
			unsigned short local_options;
			unsigned short fn_max = MAX_STRING;
			unsigned short uri_max = MAX_STRING;
			
			conf->options &= (conf->play_recursive);
			
			sub_req = ap_sub_req_lookup_uri(names->uri, r, NULL);
			if (sub_req == NULL)
				return head;
			
			local_options = ((mu_config*)ap_get_module_config(sub_req->per_dir_config, &musicindex_module))->options;
			
			ap_destroy_sub_req(sub_req);
			
			/* before dealing with a directory, sanity checks */
			/* First, is the module enabled ? */
			if ((local_options & MI_ACTIVE) == 0)
				return head;
			
			/* playall... is stream allowed ? */
			if ((conf->options & MI_PLAYALL) && !(local_options & MI_ALLOWSTREAM))
				return head;
			
			/* Searching... Is searching allowed ? */
			if ((conf->search) && !(local_options & MI_ALLOWSEARCH))
				return head;
			
			dir = opendir(names->filename);
			/* If we can't open the dir, don't go any further */
			if (dir == NULL)
				return head;
			
			if ((conf->cache_path) && (cache_check_dir(r, conf, names->filename)))
				ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] DBG: cache_check_dir failed");	/* XXX debug */
			
			fn_max -= strlen(names->filename);
			uri_max -= strlen(names->filename);
			while ((dstruct = readdir(dir))) {
				/* Before doing some strcpy, check there is
				   enough space... */
				if (strlen(dstruct->d_name) >= fn_max)
					continue;
				if (strlen(dstruct->d_name) >= uri_max)
					continue;
				strcpy(fn, dstruct->d_name);
				strcpy(uri, dstruct->d_name);
				p = make_music_entry(pool, r, p, conf, names);	
			}
			closedir(dir);
			return p;
		}
		else if (!(conf->options & (MI_PLAYLIST|MI_PLAYALL))) {
			p = new_ent(pool, head);
			p->filetype = FT_DIR;
			/* We can go throuht here, as long as the magic loop
			   stops if (p != head) */
		}
		else {
			return head;
		}
	}
	
	/* The end of adventures for the array of function pointers.
	   We call them sequentially, until a function has recognised
	   it (p!=head), or until we reach the end of the array. (this is the
	   magic loop) */
	for (i=0; (make_entry[i] != NULL) && (p == head); i++)
		p = make_entry[i](pool, head, in, conf, names, r);
	
	/* File not recognised, so not closed, and nothing interesting to return.
	   Do it now! */
	if (p == head) {
		fclose(in);
		names->create_cache_file = 0;
		return head;
	}

	p->uri = ap_pstrdup(pool, names->uri);
	p->file = p->uri;

	if ((conf->options & MI_COOKIE) == 0)
		p->file += strlen(r->parsed_uri.path);
	
	if (p->filetype == FT_DIR)
		return p;

	if (!(p->title)) {
#ifdef NO_TITLE_STRIP
		p->title = p->file;
#else
		/* Copy the name removing file extension and changing '_' to ' ' */
		p->title = ap_pstrndup(pool, p->file, strlen(p->file) - 4);
		for (i=0; p->title[i] != '\0'; i++)
			if (p->title[i] == '_')
				p->title[i] = ' ';
#endif	/* NO_TITLE_STRIP */
		/* We remove the trailing path if any. basename() ? */
		if (strrchr(p->title, '/')) {
			p->title = strrchr(p->title, '/');
			p->title++;
		}

	}
	
	/* We put that here so that we do not create cache files for unhandled file types,
	 * and p->title is formated as we desire */
	if (names->create_cache_file)
		cache_write_file(r, p, conf, names);

	names->create_cache_file = 0;

	if (!(conf->search) || (conf->options & MI_COOKIE))
		return p;

	/* Here we go if we have an ongoing search request.
	 * We will return only the entries matching the search criteria */
	if (p->file && (ap_strcasestr(p->file, conf->search)))
		return p;
	if (p->artist && (ap_strcasestr(p->artist, conf->search)))
		return p;
	if (p->album && (ap_strcasestr(p->album, conf->search)))
		return p;
	if (p->title && (ap_strcasestr(p->title, conf->search)))
		return p;

	return head;
}

/**
 * Creates and sends a playlist or webpage with the content of a directory.
 *
 * This function generates a list containing all the files in the current
 * directory, either recursively or not. The list is then sent, either as a
 * m3u playlist or a web page.
 * 
 * @todo Compact/split the code.
 * @todo Find better ways to do the MI_COOKIE stuff and to handle filtering when displaying search results.
 * @bug Beware of "big lists"
 * @todo Store custom list locally, and store only a session reference in the cookie.
 *
 * @param r Apache request_rec struct
 * @param conf MusicIndex configuration paramaters struct
 *
 * @return 0 on success, error code otherwise.
 */
short musicindex_list(request_rec *r, mu_config *conf)
{
	apr_dir_t	*dir;
	mu_ent		*head = NULL, *custom = NULL;
	const char	*custom_string = NULL;
	const char	*unescaped_args = NULL;
	char		*p;
	apr_pool_t	*subpool = NULL;

	apr_dir_open(&dir, r->filename, r->pool);

	if (dir == NULL) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
			"Can't open directory for music index: %s",
			r->filename);
		return HTTP_FORBIDDEN;
	}

	ap_pclosedir(r->pool, dir);

	/* Create a subpool we will clear sometimes. This is to save memory, as
	some operations are based on many ap_{palloc,strcat,strdup,...} for 
	playlists */
	apr_pool_create(&subpool, r->pool);
	
	if (conf->options & (MI_PLAYLIST|MI_PLAYALL|MI_COOKIESTREAM)) {
		ap_set_content_type(r, "audio/mpegurl");
		ap_table_setn(r->headers_out, "Content-Disposition",
			"filename = \"playlist.m3u\"");
	}
	else
		ap_set_content_type(r, "text/html; charset=UTF-8");

	/* Set the string containing the unescaped args */
	if (r->args) {
		unsigned short i;
		p = ap_pstrdup(r->pool, r->args);
		for (i=0; p[i]; i++) {
			if (p[i] == '+')
				p[i] = ' ';
		}
		ap_unescape_url(p);
		unescaped_args = p;
	}

	if (conf->options & MI_PLAYLIST)
		custom_string = "";
	else if ((conf->options & MI_PLAYALL) == 0) {
		/* Any operation that might imply sending a cookie ? OK, we will
		   try to get the cookie from the client and reinsert its list
		   to the new one. If the request is to remove tracks, we will
		   filter what we add */
		const char *cookie_in = ap_table_get(r->headers_in, "Cookie");
		const char *args = NULL;
		
		/* standard beginning for our cookie */
		if ((cookie_in) || (conf->options & MI_COOKIEANY))
		custom_string = ap_pstrdup(r->pool, "playlist=");
		
		if ((cookie_in) && !(conf->options & MI_COOKIEPURGE))
			args = strstr(cookie_in, "playlist=");
		
		if (args) {
			
			if ((conf->options & MI_COOKIEDEL) == 0) {
				/* XXX y se passe quoi si on essaye de stocker une playlist de
				mettons 1M chez le client (genre qlq 100aines de reps sur npyu) ?
				Perso j'opterai plutot pour une approche à la Apache::MP3:
				construire la playlist avec un ID attaché au cookie dans /tmp,
				genre /tmp/playlist-x20435JDSV, avec x20435JDSV stocké dans le cookie,
				et l'envoyer quand c'est fini... Paske la c un truc a plomber le
				browser du client sur une liste un peu violente...
				sans parler de la consommation de RAM d'Apache à chaque lecture du
				cookie du client... N'oublions pas qu'on est pas censé se limiter à
				1 ou 2 clients simultanés...
				mais bon je capte meme pas comment ils marchent ces coookies, je les vois
				pas dans mon browser... Tu peux clarifier le principe de fonctionnement? */
				custom_string = ap_getword(r->pool, &args, ';');
			} else {
				const char *escaped_args = ap_escape_uri(subpool, unescaped_args);
				
				args += 9;	/* strlen("playlist=") */
				/* If our token was found, copy from the client string
				to our cookie_header */
				while (*args != '\0') {
					p = ap_getword(subpool, &args, '&');
					
					/* check the incoming track is not in a request to remove */
					if (strstr(escaped_args, p) == NULL)
						custom_string = ap_pstrcat(subpool, custom_string, p, "&", NULL);
				}
				custom_string = ap_pstrdup(r->pool, custom_string);
				ap_clear_pool(subpool);
			}
		}
	}
	
	/* The same code used for requests of local or X-dir custom playlist.
	Both give the list of tracks in the same format :) */
	if (conf->options & (MI_PLAYLIST|MI_COOKIEADD)) {
		const char *args = unescaped_args;
		const char *uri = NULL;
		
		while (*args) {
			p = ap_getword(subpool, &args, '&');
			if (strncmp(p, "file=", 5))
				continue;
			
			uri = ap_escape_uri(subpool, ap_pstrcat(subpool, r->uri, p+5, NULL));
			if (custom_string && (strstr(custom_string, uri) == NULL))
				custom_string = ap_pstrcat(r->pool, custom_string, uri, "&", NULL);
			ap_clear_pool(subpool);
		}
	}

	/* Case where there is a request to add all the files from the current
	dir */
	if (conf->options & MI_COOKIEALL) {
		const char *uri = NULL;
		head = make_music_entry(r->pool, r, NULL, conf, NULL);
		head = quicksort(head, NULL, conf);
		
		for (custom=head; custom != NULL; custom = custom->next) {
			if (custom->filetype == FT_DIR)
				continue;
			/* la je comprends pas pkoi vu qu'on vire les DIR dans la custom,
			on les retrouve qd meme dans la mu_ent* au moment de l'affichage (suffit de virer
			le test sur FT_DIR dans la boucle de comptage pour s'en rendre compte)? */
			uri = ap_escape_uri(subpool, custom->uri);
			if (strstr(custom_string, uri) == NULL)
				custom_string = ap_pstrcat(r->pool, custom_string, uri, "&", NULL);
			ap_clear_pool(subpool);
		}
	}

	/* Ah ! We have a local string defining a cookie ! */
	if (custom_string && !strncmp(custom_string, "playlist=", 9)) {
		char *end_string = NULL;
		
		end_string = ap_psprintf(subpool, ";Version=1; Max-Age=%d; Path=/", (custom_string[9] == '\0')? 0 : conf->cookie_life);
		custom_string = ap_pstrcat(r->pool, custom_string, end_string, NULL);
		ap_table_setn(r->headers_out, "Set-Cookie", custom_string);
	}
	
	ap_send_http_header(r);

	if (r->header_only)
		return 0;
	
	ap_hard_timeout("send music list", r);

	/* build custom list*/
	if (custom_string) {
		request_rec *sub_rec = NULL;
		mu_ent_names names;
		const char *args = custom_string;
		mu_ent *mobile_ent = NULL;
		
		custom = NULL;
		
		names.create_cache_file = 0;
		conf->options |= MI_COOKIE; /* TODO : find something better (if possible) */
		
		if (!strncmp(custom_string, "playlist=", 9))
			args += 9;
		
		while ((*args != '\0') && (*args != ';')) {
			p = ap_getword(r->pool, &args, '&');
			ap_unescape_url(p);
			sub_rec = ap_sub_req_lookup_uri(p, r, NULL);
			
			if (sub_rec == NULL)
				continue;
			
			/* TODO: find a better method for filtering maybe when displaying search results... */
			strcpy(names.uri, sub_rec->unparsed_uri);
			strcpy(names.filename, sub_rec->filename);
			if (!(custom)) {
				custom = make_music_entry(r->pool, r, NULL, conf, &names);
				mobile_ent = custom;
			} else {
				mobile_ent->next = make_music_entry(r->pool, r, NULL, conf, &names);
				mobile_ent = mobile_ent->next;
			}
			custom->file = custom->uri;
			ap_destroy_sub_req(sub_rec);
		}
		conf->options &= ~MI_COOKIE;
	}

	/* We do not need the subpool any more */
	ap_destroy_pool(subpool);

	/* If we generate a custom playlist, don't build the list of what is in
	the current directory. Otherwise, we generate it only if has not be done
	before (to add all files to the cookie) */
	if (conf->options & (MI_COOKIESTREAM|MI_PLAYLIST)) {
		head = custom;
	} else if (!(head)) {
		head = make_music_entry(r->pool, r, NULL, conf, NULL);
		head = quicksort(head, NULL, conf);
	}

	if (conf->options & (MI_PLAYLIST|MI_PLAYALL|MI_COOKIESTREAM)) {
		send_playlist(r, head, conf);
	} else {
		send_head(r, conf);
		if (!(conf->search))
			send_directories(r, head, conf);
		send_tracks(r, head, conf);
		send_customlist(r, custom, conf);
		send_foot(r, conf);
	}

	ap_kill_timeout(r);
	return 0;
}

/**
 * Generates a playlist for a single file
 *
 * This function takes the requested filename from the request and sends
 * the http link in a "audio/mpegurl" (aka playlist) file.
 * 
 * @param r Apache request_rec struct
 * @param conf MusicIndex configuration paramaters struct
 *
 * @return 0
 */
short playlist_single(request_rec *r, mu_config *conf)
{
	mu_ent			*head = NULL;

	ap_set_content_type(r, "audio/mpegurl");
	ap_table_setn(r->headers_out, "Content-Disposition",
		"filename = \"playlist.m3u\"");

	ap_send_http_header(r);

	if (r->header_only)
		return 0;
	
	ap_hard_timeout("send playlist", r);

	head = make_music_entry(r->pool, r, head, conf, NULL);
	send_playlist(r, head, conf);
	
	ap_kill_timeout(r);
	return 0;
}
