/*
 *  cache.c
 *  mod_musicindex
 *
 *  $Id: cache.c,v 1.46 2004/02/10 17:43:32 varenet Exp $
 *
 *  Created by Thibaut VARENE on Fri Jul 04 2003.
 *  Copyright (c) 2003-2004 Thibaut VARENE
 *   
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 */

/**
 * @file 
 * Cache management subsystem.
 *
 * Any cache implementation (regular files, SQL, or whatever backend)
 * ought to provide the 4 mandatory interfaces:<br>
 * short cache_check_dir(request_rec *, mu_config *, mu_cache_data *)<br>
 * short cache_check_file(request_rec *, mu_config *, mu_cache_data *)<br>
 * mu_ent * cache_read_file(request_rec *, mu_ent *, mu_config *, mu_cache_data *)<br>
 * short cache_write_file(request_rec *, mu_ent *, mu_config *, mu_cache_data *)
 *
 * @author Thibaut Varene
 * @version $Revision: 1.46 $
 * @date 2003-2004
 *
 * @warning Under development, not thoroughly tested!
 * @warning 'errno' seems not to be seen as volatile by the compiler, therefore we
 *	cannot do "if(mkdir(foo) && errno == EEXIST)" for instance.
 * @warning No flat file cache support for Solaris.
 *
 * @bug We do rely on success of chdir() calls for the flat file implementation.
 *
 * @todo Write SQL bits.
 * @todo Solaris support (flat file).
 */

#include "cache.h"
#include "playlist.h"

#ifdef CACHE_SQL
/* A coder */
short cache_check_dir(request_rec *r, mu_config *conf, mu_cache_data *cachedata);
short cache_check_file(request_rec *r, mu_config *conf, mu_cache_data *cachedata);
mu_ent *cache_read_file(request_rec *r, mu_ent *head, mu_config *conf, mu_cache_data *cachedata);
short cache_write_file(request_rec *r, mu_ent *p, mu_config *conf, mu_cache_data *cachedata);
/* http://www.mysql.com/documentation/mysql/bychapter/manual_Clients.html#C */
/* one db, two tables
 * table files: |id|timestamp|full path|
 * table fdata: |id|rid| -fields- |
 * pid being the 'relative id', aka the id of the corresponding file entry.
 * fdata would contain data related to actual music files only.
 * it would probably be more interesting to store only full paths for directories,
 * and to store files separately... Who knows, who cares indeed?
 */
 /*
 MYSQL *mysql_real_connect(MYSQL *mysql, const char *host, const char *user, const char *passwd, const char *db, unsigned int port, const char *unix_socket, unsigned long client_flag) 
 int mysql_query(MYSQL *mysql, const char *query)
 unsigned int mysql_field_count(MYSQL *mysql)
 unsigned long mysql_real_escape_string(MYSQL *mysql, char *to, const char *from, unsigned long length) 
 int mysql_select_db(MYSQL *mysql, const char *db) 
 void mysql_close(MYSQL *mysql) 
 char *mysql_error(MYSQL *mysql) 
 MYSQL_FIELD *mysql_fetch_field(MYSQL_RES *result) 
 unsigned long *mysql_fetch_lengths(MYSQL_RES *result) 
 MYSQL_RES *mysql_store_result(MYSQL *mysql) 
 void mysql_free_result(MYSQL_RES *result) 
 */

#else	/* Default cache system: filesystem hierarchy with regular text files */

#ifndef NO_CACHE
/**
 * Handles error for the flat file cache subsystem.
 *
 * This function handles various errors depending on errno's value.
 * 
 * @param r Apache request_rec struct to handle log writings.
 * @param caller A string (eg. calling function name) used in messages sent.
 *
 * @todo Many things.
 */
static void error_handler(request_rec *r, const char *caller)
{
	switch (errno) {
		case EPERM:
			/* The filesystem containing pathname does not support the creation of directories. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) Can't create/delete directory.", caller);
			break;
#if 0
		case EISDIR:
			/* pathname refers to a directory.  (This is  the  non-POSIX  value returned by Linux since 2.1.132.) */
		case EINVAL:
			/* mode  requested  creation of something other than a normal file, device special file or FIFO */
		case EEXIST:
			/* pathname already exists (not necessarily as a directory). */
		case EFAULT:
			/* pathname points outside your accessible address space. */
#endif
		case EACCES:
			/* The parent directory does not allow write permission to the  process,  or  one  of  the
			directories in pathname did not allow search (execute) permission. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) Permission denied.", caller);
			break;
		case EMFILE:
			/* Too many file descriptors in use by process. */
		case ENFILE:
			/* Too many files are currently open in the system. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) Too many open files!", caller);
			break;
		case ENAMETOOLONG:
			/* pathname was too long. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) Pathname was too long.", caller);
			break;
#if 0
		case ENOENT:
			/* A directory component in pathname does not exist or is a dangling symbolic link. */
		case ENOTDIR:
			/* A component used as a directory in pathname is not, in fact, a directory. */
		case ENOTEMPTY:
			/* pathname contains entries other than . and .. . */
#endif
		case ENOMEM:
			/* Insufficient kernel memory was available. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) Out Of Memory!", caller);
			break;
		case EROFS:
			/* pathname refers to a file on a read-only filesystem. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) Read-Only filesystem!", caller);
			break;
		case ELOOP:
			/* Too many symbolic links were encountered in resolving pathname. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) Too many symbolic links.", caller);
			break;
		case EIO:
			/* An I/O error occured. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) I/O error.", caller);
			break;
		case ENOSPC:
			/* The device containing pathname has no room for the new directory.
			The new directory cannot be created because the user's disk quota is exhausted. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) No space left on device!", caller);
			break;
		default:
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, "[musicindex] (%s) - error_handler! errno=%i", caller, errno);
			break;
	}
	return;
}

/**
 * Creates cache subdirectories.
 *
 * This subroutine takes care of creating all requested directories and
 * subdirectories if they don't already exist and if possible.
 * 
 * @warning If dirpath begins with a "/" the function will return immediately.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param dirpath A string representing a path to create.
 *
 * @return 0 on succes, CA_FATAL otherwise.
 */
static short cache_make_dir(request_rec *r, char *dirpath)
{
	short l = 0, m = 0;
	char *tempdir = NULL;
	
	do {	/* We build the path subdirs by subdirs, in a "mkdir -p" fashion */
		tempdir = ap_pstrndup(r->pool, dirpath, (m + (l = strcspn(dirpath + m, "/"))));
		m += l;
		
		if (!l)
			break;

		/* skipping (potentially multiple) slashes */
		while (dirpath[m] == '/')
			m++;
		
		if (mkdir(tempdir, S_IRWXU)) {
			if (errno == EEXIST);
			else
				goto error_out;
		}
	} while (1);

	return 0;

error_out:
	error_handler(r, __FUNCTION__);
	return CA_FATAL;
}

/**
 * Removes cache subdirectories.
 *
 * This subroutine takes care of removing any given directory and
 * its content (recursively) if any, and if possible.
 * 
 * @param r Apache request_rec struct to handle log writings.
 * @param cachedir A DIR stream corresponding to the directory to remove.
 * @param curdir A string representing the absolute path of the corresponding
 *	parent directory on the "original" filesystem.
 */
static void cache_remove_dir(request_rec *r, DIR *cachedir, char *curdir)
{
	DIR		*subdir = NULL;
	struct dirent	*cachedirent = NULL;
	struct stat	origdirstat;
	char 		*origdir = NULL;
	
	fchdir(dirfd(cachedir));			/* on se place dans le repertoire de cache. XXX Pas de test ici, tout est verifie avant (en principe). */
	while ((cachedirent = readdir(cachedir))) {	/* on parcours le repertoire */
		if (!(strcmp(cachedirent->d_name, ".")) || !(strcmp(cachedirent->d_name, "..")))	/* We'd rather avoid trying to remove the whole filesystem... */
			continue;
		
		if (unlink(cachedirent->d_name)) {	/* We try to remove any entry (actually we will only remove regular files) */
				if ((errno == EISDIR) || (errno == EPERM)) {	/* On BSDs unlink() returns EPERM on non empty directories. This shouldn't lead to infloop because of subsequent tests. */
				/* If it's a directory, we check that the "original" still exists.
				 * If not, we remove it recursively.
				 * Reminder: "errno == (EISDIR || EPERM)" doesn't work */
				origdir = ap_pstrcat(r->pool, curdir, "/", cachedirent->d_name, NULL);
				if (stat(origdir, &origdirstat)) {
					if (rmdir(cachedirent->d_name)) {	/* stat() sets errno. We have to split */
						if (errno == ENOTEMPTY) {			/* il est pas vide, bigre! */
							subdir = opendir(cachedirent->d_name);	/* on ouvre le vilain repertoire pour en supprimer le contenu */
							cache_remove_dir(r, subdir, origdir);	/* en rappelant recursivement la fonction sur son contenu. */
							closedir(subdir);			/* a noter que dans ce cas la il y a un test inutile, celui qui verifie si l'original existe tjrs. Mais bon. */
							fchdir(dirfd(cachedir));		/* on retourne au repertoire precedent */
							rmdir(cachedirent->d_name);		/* maintenant il est vide, et on peut pas avoir d'erreur vu les tests precedants */
						}
						else
							error_handler(r, __FUNCTION__);			/* Oops, on est tombe sur une merde */
					}
				}
			}
			else
				error_handler(r, __FUNCTION__);		/* Oops, on est tombe sur une merde, mais plus tot */
		}
	}
	
	return;
}

/**
 * Initializes flat file cache subsystem.
 *
 * Basically we do nothing more here than creating the root cache folder.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param conf configuration structure in which the path to the cache root can be found.
 *
 * @return 0 on succes, CA_FATAL otherwise.
 */
static short cache_init(request_rec *r, mu_config *conf)
{
	chdir("/");	/* let's pray this will never fail */
	if (cache_make_dir(r, conf->cache_path + 1))	/* since we've chdir'd, we send the path without the leading '/' */
		goto error_out;
	
	return 0;
	
error_out:
	error_handler(r, __FUNCTION__);
	return CA_FATAL;
}

/**
 * Checks if a directory already exists in the cache.
 *
 * This function is called when the caller wants to know whether a given
 * directory (as found in the cachedata struct) exists or not in the cache.
 * If the directory exists, the function returns successfully, otherwise
 * it tries to create it if possible and returns accordingly.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param conf The config structure used to find out cache configuration.
 * @param path A string containing the full directory path.
 *
 * @return 0 on succes, CA_MISSARG if cachedata->name is missing, CA_FATAL otherwise.
 */
short cache_check_dir(request_rec *r, mu_config *conf, char *path)
{
	DIR		*cachedir = NULL;
	struct stat	cachedirstat, dirstat;
	
	if (!path)
		return CA_MISSARG;
	
	/* Making sure the cache has been initialized, initialize it otherwise.
	 * Bear in mind we're chdir'd from now on. */
	if (chdir(conf->cache_path)) {	 		/* on va dans le rep de cache */
		if (errno == ENOENT) {			/* il n'existe pas, il est temps d'initialiser le cache */
			if (cache_init(r, conf))
				return CA_FATAL;	/* l'init du cache a chie, c'est mauvais */
			chdir(conf->cache_path);	/* maintenant qu'on l'a cree on peut enfin y aller. Considere sans echec */
		}
		else
			goto error_out;			/* un autre probleme, on degage */
	}
	
	/* Actually check for the directory in the cache, create it if needed.
	 * "+ 1" offset to suppress leading '/'. */
	if (!(cachedir = opendir(path + 1))) {		/* on essaye d'ouvrir le repertoire concerne dans le cache (on supprime le leading "/" */
		if (errno == ENOENT) {					/* il n'existe pas mais on peut le creer (ca correspond a ENOENT, a verifier) */
			if (cache_make_dir(r, path + 1))	/* on envoie le chemin prive du leading '/' */
				goto error_out;
		}
		else
			goto error_out;					/* un autre probleme, on degage */
	}
	else {	/* Checking for cache sanity. Has it expired for that folder ? If so, delete its content. */
		fstat(dirfd(cachedir), &cachedirstat);			/* recuperons les stats du repertoire. XXX On considere cet appel sans echec vu les tests qu'on a fait avant. */	
		stat(path, &dirstat);			/* recuperons les stats du rep d'origine. XXX pas de test ici, a priori ya pas de raison qu'on puisse pas les recuperer */
		if (cachedirstat.st_mtime < dirstat.st_mtime)		/* si la date de modif du rep de cache est plus vieille que celle du rep original, alors qqc a ete ajoute ou retire ou ecrit */
			cache_remove_dir(r, cachedir, path);	/* alors on le vide proprement */
		closedir(cachedir);					/* On en a fini avec le repertoire, on le referme */
	}

	return 0;

error_out:
	error_handler(r, __FUNCTION__);
	return CA_FATAL;
}

/**
 * Fills in the information fields about a music file from the cache.
 *
 * This function reads the tags from the cache file
 * and fills in the struct mu_ent fields accordingly.
 * 
 * @param pool Pool
 * @param head Head
 * @param in Not used (except for closing).
 * @param conf MusicIndex configuration paramaters struct.
 * @param names struct names to get the current filename.
 * @param r Apache request_rec struct to handle log writings.
 *
 * @return When possible, struct mu_ent correctly set up, file stream closed,
 * 		Head pointer otherwise.
 */
mu_ent *make_cache_entry(apr_pool_t *pool, mu_ent *head,
	FILE *in, mu_config *conf, mu_ent_names *names, request_rec *r)
{
	mu_ent		*p = head;
	short 		result = 0;
	FILE		*cache_file = NULL;

	/* Is cache enabled? */
	if (conf->cache_path == NULL)
		return p;

	/* Making sure the cache has been initialized, initialize it otherwise.
	 * Bear in mind we're chdir'd from now on. */
	if (chdir(conf->cache_path)) {		 	/* on va dans le rep de cache */
		if (cache_init(r, conf))
			goto error_out;		/* l'init du cache a chie, c'est mauvais */
		chdir(conf->cache_path);	/* maintenant qu'on l'a cree on peut enfin y aller. Considere sans echec */
	}

	/* Actually check for the file in the cache, open it if possible.
	 * "+ 1" offset to suppress leading '/'. */
	cache_file = fopen(names->filename + 1, "r");

	
	if (!cache_file) {	/* on essaye d'ouvrir le fichier concerne en ro */
		if (errno == ENOENT) {	/* n'existe pas encore mais on peut a priori le creer */
			names->create_cache_file = 1;
			return p;			/* la suite du programme s'occupera du reste */
		}
		else
			goto error_out;		/* game over */
	}

	/* We acquire a shared advisory lock on the file to be (almost) certain of its integrity.
	 * This will prevent reading from incomplete cache files */
	if (flock(fileno(cache_file), LOCK_SH|LOCK_NB)) {
		fclose(cache_file);
		return p;
	}

	p = new_ent(r->pool, head);
	p->title = ap_pcalloc(r->pool, MAX_STRING);	/* ugh..ly! */
	p->album = ap_pcalloc(r->pool, MAX_STRING);
	p->artist = ap_pcalloc(r->pool, MAX_STRING);
	p->genre = ap_pcalloc(r->pool, 64);	/* no very_long_and_explicit_genre I hope */
	
	result = fscanf(cache_file, "album: %[^\n]\nartist: %[^\n]\n"
		"title: %[^\n]\ndate: %hu\ntrack: %hu\nposn: %hu\n"
		"length: %lu\nbitrate: %lu\nsize: %lu\nfiletype: %c\n"
		"genre: %[^\n]\n",
		p->album, p->artist, p->title, &p->date, &p->track, &p->posn, &p->length,
		&p->bitrate, &p->size, &p->filetype, p->genre);

	/* Explicitely releasing the lock, and closing the file */
	flock(fileno(cache_file), LOCK_UN);
	fclose(cache_file);
	
	/* XXX ameliorer le test ici. if (result != NB_FIELDS) ? */
	if (!result)	/* fscanf() returns the number of input items assigned */
		return head;
	
	if (!strcmp(p->album, "(null)"))
		p->album[0] = '\0';
	if (!strcmp(p->artist, "(null)"))
		p->artist[0] = '\0';
	if (!strcmp(p->genre, "(null)"))
		p->genre[0] = '\0';
	
	fclose(in);
	
	return p;
	
error_out:
	error_handler(r, __FUNCTION__);
	return p;
}

/**
 * Creates and writes cache file information.
 *
 * This function creates a new cache file (using cachedata->name), and
 * fills it with the data contained in the mu_ent p structure.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param p A mu_ent struct containing the data to store.
 * @param conf The config structure used to find out cache configuration.
 * @param names A names structure in which the file name can be found.
 *
 * @return 0 on success, CA_LOCKED if the file is locked already, CA_FATAL otherwise.
 */
short cache_write_file(request_rec *r, mu_ent *p, mu_config *conf, mu_ent_names *names)
{
	FILE	*cache_file = NULL;

	chdir(conf->cache_path);	/* Par securite on se re-chdir(). XXX Considere sans echec */
	
	cache_file = fopen(names->filename + 1, "w"); /* on ouvre le fichier en ecriture et en "truncate" (pour eviter les fichiers pourris) */

	/* let's check if something bad happened */
	if (!cache_file)
		goto error_out;

	/* We acquire an exclusive advisory lock on the file to avoid corruption by another process.
	 * This will also prevent reading from incomplete cache */
	if (flock(fileno(cache_file), LOCK_EX|LOCK_NB)) {
		fclose(cache_file);
		if (errno == EWOULDBLOCK)
			return CA_LOCKED;
		else
			goto error_out;
	}
	
	fprintf(cache_file, "album: %s\nartist: %s\ntitle: %s\ndate: %hu\n"
		"track: %hu\nposn: %hu\nlength: %lu\nbitrate: %lu\n"
		"size: %lu\nfiletype: %s\ngenre: %s\n",
		p->album, p->artist, p->title, p->date, p->track, p->posn, p->length,
		p->bitrate, p->size, &p->filetype, p->genre);

	/* Explicitely releasing the lock, and closing the file */
	flock(fileno(cache_file), LOCK_UN);
	fclose(cache_file);

	return 0;

error_out:
	error_handler(r, __FUNCTION__);
	return CA_FATAL;
}
#endif	/* NO_CACHE */
#endif	/* Cache type */
