/* -------------------------------------------------------------------------- *\
 * mod_choke.c                                                                *
 * Nathan Shafer <nate-mod_choke@seekio.com>                                  *
 *                                                                            *
 * Limit bandwidth usage per apache child.                                    *
 *                                                                            *
 * Copyright (C) 2004 Cyberheat, Inc                                          *
 *                                                                            *
 * This program is free software; you can redistribute it and/or              *
 * modify it under the terms of the GNU General Public License                *
 * as published by the Free Software Foundation; either version 2             *
 * of the License, or (at your option) any later version.                     *
 *                                                                            *
 * This program is distributed in the hope that it will be useful,            *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 * GNU General Public License for more details.                               *
 *                                                                            *
 * You should have received a copy of the GNU General Public License          *
 * along with this program; if not, write to the Free Software                *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.*
\* -------------------------------------------------------------------------- */

#include "httpd.h"  
#include "http_config.h"  
#include "http_request.h"  
#include "http_protocol.h"  
#include "http_core.h"  
#include "http_main.h" 
#include "http_log.h"  
#include "scoreboard.h"  

#include <math.h>

#ifdef HAVE_SHMGET
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

module choke_module;

#define CHOKE_VERSION "0.06"
#undef CHOKE_DEBUG

/* -------------------------------------------------------------------------- *\
 * Some config options.  Should not have to change these at all               *
\* -------------------------------------------------------------------------- */
#define SHM_KEY                IPC_PRIVATE

/* rate settings */
#define MIN_SEND_LENGTH        1024
#define MAX_SEND_LENGTH        1048576
#define MAX_SEND_PERCENT       .25
#define MAX_TARGET             .5

/* -------------------------------------------------------------------------- *\
 * Constants                                                                  *
\* -------------------------------------------------------------------------- */
#define UNSET                  -45922
#define CHOKE_ENABLED          1
#define CHOKE_DISABLED         0
#define CHOKE_SUMMARY_ENABLED  1
#define CHOKE_SUMMARY_DISABLED 0

/* -------------------------------------------------------------------------- *\
 * Data structures                                                            *
\* -------------------------------------------------------------------------- */
typedef struct {
  int glb_max_ip;
  int glb_max_user;
  int max_ip;
  int max_user;
} choke_server_config;

typedef struct {
  int   state;
  long  rate;
  char *rate_env;
  long  burst;
  char *burst_env;
  int   summary;
} choke_dir_config;

typedef struct {
  long offset;
  long length;
} range;

#ifdef HAVE_SHMGET
/* our own personal scoreboard */
typedef struct {
  char ip[32];
  char fwd[32];
  char user[50];
  char server_name[100];
  char request[100];
} choke_score_rec;

typedef struct {
  choke_score_rec servers[HARD_SERVER_LIMIT];
} choke_scoreboard;
#endif

/* -------------------------------------------------------------------------- *\
 * Globals                                                                    *
\* -------------------------------------------------------------------------- */
#ifdef HAVE_SHMGET
static char status_flags[SERVER_NUM_STATUS];
static choke_scoreboard *csb = NULL;
static int choke_shmid;
#endif

/* -------------------------------------------------------------------------- *\
 * Miscellaneous functions                                                    *
\* -------------------------------------------------------------------------- */

#ifdef HAVE_SHMGET
static void choke_detach_shm(void *data) {
  shmdt(data);
}
#endif

static void choke_init(server_rec *s, pool *p) {
#ifdef CHOKE_DEBUG
  ap_log_printf(s, "Initializing mod_choke");
#endif

  /* Add our banner to the ServerBanner */
  ap_add_version_component("mod_choke/" CHOKE_VERSION);

#ifdef HAVE_SHMGET
  /* Set status flags for the choke-status display.  Taken from mod_status */
  status_flags[SERVER_DEAD] = '.';    /* We don't want to assume these are in */
  status_flags[SERVER_READY] = '_';   /* any particular order in scoreboard.h */
  status_flags[SERVER_STARTING] = 'S';
  status_flags[SERVER_BUSY_READ] = 'R';
  status_flags[SERVER_BUSY_WRITE] = 'W';
  status_flags[SERVER_BUSY_KEEPALIVE] = 'K';
  status_flags[SERVER_BUSY_LOG] = 'L';
  status_flags[SERVER_BUSY_DNS] = 'D';
  status_flags[SERVER_GRACEFUL] = 'G';

  if((choke_shmid = shmget(SHM_KEY, sizeof(choke_scoreboard), IPC_CREAT | 0662)) == -1) {
    ap_log_error(APLOG_MARK, APLOG_EMERG, s, 
        "could not initialize shared memory");
    exit(1);
  }

#ifdef CHOKE_DEBUG
  ap_log_printf(s, "Initialized shared memory for mod_choke (%d)", choke_shmid);
#endif

  if( (csb = (choke_scoreboard *) shmat(choke_shmid, NULL, 0)) == (choke_scoreboard *) -1) {
    ap_log_error(APLOG_MARK, APLOG_EMERG, s,
        "shmat error");
    /* exit later after marking the segment to remove itself */
  } else {
    ap_register_cleanup(p, (void *)csb, choke_detach_shm, ap_null_cleanup);
  }

  /* Mark the segment for deletion once all attachments are detached */
  if(shmctl(choke_shmid, IPC_RMID, NULL) == -1) {
    ap_log_error(APLOG_MARK, APLOG_EMERG, s,
        "Could not mark shared segment for deletion, you must manually clean it up");
  }

  /* exit if we didn't attach successfully */
  if(csb == (choke_scoreboard *) -1) {
    exit(1);
  }
#else
  ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, s,
      "Shared memory is not available, MaxConnectionsPerIP and "
      "MaxConnectionsPerUser will have no effect");
#endif
}

static double choke_gettime() {
  struct timeval tv;

  gettimeofday(&tv, (struct timezone *) 0);
  return(tv.tv_sec + (tv.tv_usec * .0000001));
}

int choke_is_number(const char *str) {
  int len = strlen(str);
  int i;
  
  for(i = 0; i < len; i++) {
    if(!isdigit(str[i])) {
      return(0);
    }
  }
  return(1);
}

long choke_parse_bytes(const char *arg) {
  long ret;
  int  arglen = strlen(arg);
  char last_char = arg[arglen - 1];

  if(last_char == 'K' || last_char == 'k') {
    char number[128];

    /* cut off the last char */
    snprintf(number, arglen, "%s", arg);

    if(choke_is_number(number))
      return(atoll(number) * 1000);
    else
      return(-1);

  } else if(last_char == 'M' || last_char == 'm') {
    char number[128];

    /* cut off the last char */
    snprintf(number, arglen, "%s", arg);

    if(choke_is_number(number))
      return(atoll(number) * 1000000);
    else
      return(-1);

  } else {
    if(choke_is_number(arg))
      return(atoll(arg));
    else
      return(-1);
  }
}

/* -------------------------------------------------------------------------- *\
 * Configuration functions                                                    *
\* -------------------------------------------------------------------------- */

/* create_config functions */
static void *choke_create_server_config(pool *p, server_rec *s) {
  choke_server_config *cfg = (choke_server_config *)ap_palloc(p, sizeof(choke_server_config));

  cfg->glb_max_ip   = UNSET;
  cfg->glb_max_user = UNSET;
  cfg->max_ip       = UNSET;
  cfg->max_user     = UNSET;

  return( (void *)cfg );
}

static void *choke_create_dir_config(pool *p, char *path) {
  choke_dir_config *cfg = (choke_dir_config *)ap_palloc(p, sizeof(choke_dir_config));

  cfg->state      = UNSET;
  cfg->rate       = UNSET;
  cfg->rate_env   = NULL;
  cfg->burst      = UNSET;
  cfg->burst_env  = NULL;
  cfg->summary    = UNSET;

  return( (void *)cfg );
}

/* merge functions */
static void *choke_merge_server_config(pool *p, void *base, void *new) {
  choke_server_config *merged = (choke_server_config *)ap_palloc(p, sizeof(choke_server_config));
  choke_server_config *parent = (choke_server_config *)base;
  choke_server_config *child  = (choke_server_config *)new;

  merged->glb_max_ip   = child->glb_max_ip   == UNSET ? parent->glb_max_ip   : child->glb_max_ip;
  merged->glb_max_user = child->glb_max_user == UNSET ? parent->glb_max_user : child->glb_max_user;
  merged->max_ip       = child->max_ip       == UNSET ? parent->max_ip       : child->max_ip;
  merged->max_user     = child->max_user     == UNSET ? parent->max_user     : child->max_user;

  return( (void *)merged );
}

static void *choke_merge_dir_config(pool *p, void *base, void *new) {
  choke_dir_config *merged = (choke_dir_config *)ap_palloc(p, sizeof(choke_dir_config));
  choke_dir_config *parent = (choke_dir_config *)base;
  choke_dir_config *child  = (choke_dir_config *)new;

  merged->state     = child->state      == UNSET ? parent->state      : child->state;
  merged->rate      = child->rate       == UNSET ? parent->rate       : child->rate;
  merged->rate_env  = child->rate_env   == NULL  ? parent->rate_env   : child->rate_env;
  merged->burst     = child->burst      == UNSET ? parent->burst      : child->burst;
  merged->burst_env = child->burst_env  == NULL  ? parent->burst_env  : child->burst_env;
  merged->summary   = child->summary    == UNSET ? parent->summary    : child->summary;

  return( (void *)merged );
}

/* set functions */
static const char *choke_cmd_glb_max_ip(cmd_parms *parms, void *mconfig, const char *arg) {
  choke_server_config *cfg = (choke_server_config *)
    ap_get_module_config(parms->server->module_config, &choke_module);

  if(choke_is_number(arg)) {
    cfg->glb_max_ip = atoi(arg);
  } else {
    cfg->glb_max_ip = UNSET;
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, parms->server,
        "Argument %s to GlobalMaxConnectionsPerIP is not a valid number", arg);
  }

  return(NULL);
}

static const char *choke_cmd_glb_max_user(cmd_parms *parms, void *mconfig, const char *arg) {
  choke_server_config *cfg = (choke_server_config *)
    ap_get_module_config(parms->server->module_config, &choke_module);

  if(choke_is_number(arg)) {
    cfg->glb_max_user = atoi(arg);
  } else {
    cfg->glb_max_user = UNSET;
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, parms->server,
        "Argument %s to GlobalMaxConnectionsPerUser is not a valid number", arg);
  }

  return(NULL);
}

static const char *choke_cmd_max_ip(cmd_parms *parms, void *mconfig, const char *arg) {
  choke_server_config *cfg = (choke_server_config *)
    ap_get_module_config(parms->server->module_config, &choke_module);

  if(choke_is_number(arg)) {
    cfg->max_ip = atoi(arg);
  } else {
    cfg->max_ip = UNSET;
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, parms->server,
        "Argument %s to MaxConnectionsPerIP is not a valid number", arg);
  }

  return(NULL);
}

static const char *choke_cmd_max_user(cmd_parms *parms, void *mconfig, const char *arg) {
  choke_server_config *cfg = (choke_server_config *)
    ap_get_module_config(parms->server->module_config, &choke_module);

  if(choke_is_number(arg)) {
    cfg->max_user = atoi(arg);
  } else {
    cfg->max_user = UNSET;
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, parms->server,
        "Argument %s to MaxConnectionsPerUser is not a valid number", arg);
  }

  return(NULL);
}

static const char *choke_cmd_state(cmd_parms *parms, void *mconfig, int flag) {
  choke_dir_config *cfg = (choke_dir_config *)mconfig;
  cfg->state = (flag ? CHOKE_ENABLED : CHOKE_DISABLED);
  return(NULL);
}

static const char *choke_cmd_rate(cmd_parms *parms, void *mconfig, const char *arg) {
  choke_dir_config *cfg = (choke_dir_config *)mconfig;
  long bytes = choke_parse_bytes(arg);

  if(bytes != -1) {
    cfg->rate = bytes;
  } else {
    cfg->rate = UNSET;
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, parms->server,
        "Could not parse argument %s to ChokeRate", arg);
  }

  return(NULL);
}

static const char *choke_cmd_rate_env(cmd_parms *parms, void *mconfig, const char *arg) {
  choke_dir_config *cfg = (choke_dir_config *)mconfig;

  cfg->rate_env = ap_pstrdup(parms->pool, arg);

  return(NULL);
}

static const char *choke_cmd_burst(cmd_parms *parms, void *mconfig, const char *arg) {
  choke_dir_config *cfg = (choke_dir_config *)mconfig;
  long bytes = choke_parse_bytes(arg);

  if(bytes != -1) {
    cfg->burst = bytes;
  } else {
    cfg->burst = UNSET;
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, parms->server,
        "Could not parse argument %s to ChokeBurst", arg);
  }

  return(NULL);
}

static const char *choke_cmd_burst_env(cmd_parms *parms, void *mconfig, const char *arg) {
  choke_dir_config *cfg = (choke_dir_config *)mconfig;

  cfg->burst_env = ap_pstrdup(parms->pool, arg);

  return(NULL);
}

static const char *choke_cmd_summary(cmd_parms *parms, void *mconfig, int flag) {
  choke_dir_config *cfg = (choke_dir_config *)mconfig;
  cfg->summary = (flag ? CHOKE_SUMMARY_ENABLED : CHOKE_SUMMARY_DISABLED);
  return(NULL);
}

/* the array of command_recs */
static const command_rec choke_cmds[] = {
  { "GlobalMaxConnectionsPerIP",   choke_cmd_glb_max_ip,   NULL, RSRC_CONF,  TAKE1,
    "maximum number of connections to allow from a single ip over the whole server" },
  { "GlobalMaxConnectionsPerUser", choke_cmd_glb_max_user, NULL, RSRC_CONF,  TAKE1,
    "maximum number of connections to allow from a single user over the whole server" },
  { "MaxConnectionsPerIP",         choke_cmd_max_ip,       NULL, RSRC_CONF,  TAKE1,
    "maximum number of connections to allow from a single ip for each [virtual]host" },
  { "MaxConnectionsPerUser",       choke_cmd_max_user,     NULL, RSRC_CONF,  TAKE1,
    "maximum number of connections to allow from a single user for each [virtual]host" },
  { "Choke",                       choke_cmd_state,        NULL, OR_OPTIONS, FLAG, 
    "turn mod_choke On or Off (default Off)" },
  { "ChokeRate",                   choke_cmd_rate,         NULL, OR_OPTIONS, TAKE1,
    "a rate to send at in bytes" },
  { "ChokeRateEnv",                choke_cmd_rate_env,     NULL, OR_OPTIONS, TAKE1, 
    "environment variable to get the ChokeRate from on each request" },
  { "ChokeBurst",                  choke_cmd_burst,        NULL, OR_OPTIONS, TAKE1,
    "a number of bytes to send at full speed at the beginning of the transfer" },
  { "ChokeBurstEnv",               choke_cmd_burst_env,    NULL, OR_OPTIONS, TAKE1,
    "environment variable to get the ChokeBurst from on each request" },
  { "ChokeSummary",                choke_cmd_summary,      NULL, OR_OPTIONS, FLAG, 
    "print a summary after each send.  On/Off (default Off)" },
  { NULL }
};

/* -------------------------------------------------------------------------- *\
 * The main function that does the actual sending of the file                 *
\* -------------------------------------------------------------------------- */

static int choke_send_file(request_rec *r, choke_dir_config *cfg, FILE *f,
                           long offset, long length, long rate, long burst)
{
  long   sent, to_send;
  long   actual_rate;
  long   fileleft  = length;
  long   send_length;
  int    num_under = 0, num_over = 0;
  double time_start, time_end, time_diff, time_target, time_sleep;
  int    filesent = 0;

#ifdef CHOKE_DEBUG
  ap_log_printf(r->server,
      "choke_send_file(offset = %ld, length = %ld, rate = %ld, burst = %ld)\n",
      offset, length, rate, burst);
#endif

  /* calculate the optimum send size for this rate */
  send_length = rate * MAX_TARGET;
  if(send_length > fileleft * MAX_SEND_PERCENT)
    send_length = fileleft * MAX_SEND_PERCENT;
  if(send_length < MIN_SEND_LENGTH)
    send_length = MIN_SEND_LENGTH;
  if(send_length > MAX_SEND_LENGTH)
    send_length = MAX_SEND_LENGTH;
  
  /* seek to offset so that we send the right part of the file */
  fseek(f, offset, SEEK_SET);

  /* Burst */
  if(burst) {
    /* figure out how much to burst */
    if(fileleft < burst) {
      to_send = fileleft;
    } else {
      to_send = burst;
    }

    /* send the burst */
    sent = ap_send_fd_length(f, r, to_send);

    fileleft  -= sent;
    filesent  += sent;

#ifdef CHOKE_DEBUG
    ap_log_printf(r->server, "[choke] burst %ld/%ld\n", sent, burst);
#endif
  }

  /* This is the main loop that sends a chunk of the file on each
   * iteration */
  while(fileleft > 0) {
    /* record time that we started this individual send */
    time_start = choke_gettime();

    /* Calculate the target amount of time that sending send_length at
     * rate should take */
    time_target = (double)send_length / (double)rate;

    /* figure out the size of this send */
    if(fileleft > send_length) {
      to_send = send_length;
    } else {
      to_send = fileleft;
    }

    /* Send! */
    sent = ap_send_fd_length(f, r, to_send);

    /* break if the connection was closed */
    if(sent < 0 || r->connection->aborted)
      break;

    fileleft -= sent;
    filesent += sent;

    /* record time that we ended */
    time_end = choke_gettime();

    /* calculate the time that it took to send */
    time_diff = time_end - time_start;

    /* calculate the actual rate that we sent at.  This is basically going
     * to be the time it took to inject the data into the buffer */
    actual_rate = sent / time_diff;

    /* calculate the time we need to sleep */
    if(fileleft > 0 && time_diff < time_target) {
      time_sleep = time_target - time_diff;
    } else {
      time_sleep = 0;
    }

#ifdef CHOKE_DEBUG
    ap_log_printf(r->server,
        "[choke] sent %ld/%ld bytes in %f secs, target: %f sleep: %f actual_rate: %ld\n",
        sent, to_send, time_diff, time_target, time_sleep, actual_rate);
#endif

    /* sleep until it's time to send again*/
    if(time_sleep) {
      usleep(time_sleep * 1000000);
    }
  }

  return(filesent);
}

/* -------------------------------------------------------------------------- *\
 * The main content handler                                                   *
\* -------------------------------------------------------------------------- */

static int choke_handler(request_rec *r) {
  int               rc;
  FILE             *f;
  long              filesize, rate, burst;
  int               byterange;
  double            request_start, request_end;
  choke_dir_config *cfg = (choke_dir_config *)ap_get_module_config(r->per_dir_config, &choke_module);

  /* Check if we're set to be on, if not then return DECLINED so that the
   * normal apache handler picks up the request */
  if(cfg->state != CHOKE_ENABLED) {
    return(DECLINED);
  }

  /* find the rate this request will send at */
  if(cfg->rate_env != NULL) {
    const char *rate_env = ap_pstrdup(r->pool, ap_table_get(r->subprocess_env, cfg->rate_env));
    if(rate_env != NULL) {
      rate = choke_parse_bytes(rate_env);
    } else {
      rate = cfg->rate;
    }
  } else {
    rate = cfg->rate;
  }
  
  if(rate <= 0) {
    return(DECLINED);
  }

  /* find the burst */
  if(cfg->burst_env != NULL) {
    const char *burst_env = ap_pstrdup(r->pool, ap_table_get(r->subprocess_env, cfg->burst_env));
    if(burst_env != NULL) {
      burst = choke_parse_bytes(burst_env);
    } else {
      burst = cfg->burst;
    }
  } else {
    burst = cfg->burst;
  }

  /* Check if a burst was set */
  if(burst <= 0) {
    burst = 0;
  }
  
  /* set the X-mod_choke header */
  ap_table_set(r->headers_out, "X-mod-choke", CHOKE_VERSION);

  filesize = r->finfo.st_size;
  
#ifdef CHOKE_DEBUG
  ap_log_printf(r->server,
      "[choke] received request: %s (%ld bytes) state: %d rate: %ld burst: %ld summary: %d\n",
      r->uri, filesize, cfg->state, rate, burst, cfg->summary);
#endif
  
  /* get rid of the request body if any was sent */
  if ((rc = ap_discard_request_body(r)) != OK) {
    return rc;
  }

  /* Check for methods we don't support */
  if (r->method_number == M_INVALID) {
    ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
        "Invalid method in request %s", r->the_request);
    return NOT_IMPLEMENTED;
  }
  if (r->method_number == M_OPTIONS) {
    return DECLINED;
  }
  if (r->method_number == M_PUT) {
      return METHOD_NOT_ALLOWED;
  }

  /* make sure file exists */
  if (r->finfo.st_mode == 0 || (r->path_info && *r->path_info)) {
    char *emsg;

    emsg = "File does not exist: ";
    if (r->path_info == NULL) {
      emsg = ap_pstrcat(r->pool, emsg, r->filename, NULL);
    }
    else {
      emsg = ap_pstrcat(r->pool, emsg, r->filename, r->path_info, NULL);
    }
    ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r, emsg);
    return HTTP_NOT_FOUND;
  }

  /* only proceed if this was a GET request */
  if (r->method_number != M_GET) {
    return METHOD_NOT_ALLOWED;
  }

  /* Open the file */
#if defined(OS2) || defined(WIN32)
  /* Need binary mode for OS/2 or WIN32*/
  f = ap_pfopen(r->pool, r->filename, "rb");
#else
  f = ap_pfopen(r->pool, r->filename, "r");
#endif

  if (f == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
        "file permissions deny server access: %s", r->filename);
    return FORBIDDEN;
  }

  /* Set useful headers */
  ap_update_mtime(r, r->finfo.st_mtime);
  ap_set_last_modified(r);
  ap_set_etag(r);

  if (((rc = ap_meets_conditions(r)) != OK)
      || (rc = ap_set_content_length(r, r->finfo.st_size)))
  {
    return rc;
  }

  /* Send the headers */
  byterange = ap_set_byterange(r);
  ap_send_http_header(r);

  /* Now the main block that does all the fun stuff */
  if (!r->header_only) {
    long filesent = 0;
    /* record the start of the whole request */
    request_start = choke_gettime();

    if(byterange) {
      long offset, length;
      long burstleft = burst;

      while(ap_each_byterange(r, &offset, &length)) {
        long sent;
        
        sent = choke_send_file(r, cfg, f, offset, length, rate, burstleft);
        filesent  += sent;
        burstleft -= sent;
        if(burstleft < 0) {
          burstleft = 0;
        }
        
        if(r->connection->aborted)
          break;
      }
    } else {
      filesent = choke_send_file(r, cfg, f, 0, filesize, rate, burst);
    }

    request_end = choke_gettime();

    /* Print summary if summary printing is on */
    if(cfg->summary == CHOKE_SUMMARY_ENABLED) {
      double request_rate, request_time;
      char   rate_units[5];
  
      /* calculate the rate */
      request_time = request_end - request_start;
      if(request_time > 0) {
        request_rate = filesent / request_time;
      } else {
        request_rate = 999999;
      }
  
      /* convert to more readable units */
      if(request_rate < 1000) {
        strcpy(rate_units, "B/s");
      } else if(request_rate < 1000000) {
        request_rate /= 1000;
        strcpy(rate_units, "KB/s");
      } else {
        request_rate /= 1000000;
        strcpy(rate_units, "MB/s");
      }
  
      /* print */
      ap_log_error(APLOG_MARK, APLOG_NOTICE | APLOG_NOERRNO, r->server,
          "[client %s] mod_choke sent %ld/%ld bytes of %s in %f secs at %.2f %s (rng: %s)",
          ap_get_remote_host(r->connection, r->per_dir_config, REMOTE_NAME),
          filesent, filesize, r->filename, request_time, request_rate,
          rate_units, (byterange ? "Yes" : "No"));
    }
  }

  ap_pfclose(r->pool, f);

  return OK;
}

#ifdef HAVE_SHMGET
static int choke_concurrency_check(request_rec *r) {
  conn_rec            *c            = r->connection;
  choke_score_rec     *rec          = &csb->servers[c->child_num];
  choke_server_config *cfg          = (choke_server_config *)ap_get_module_config(
                                            r->server->module_config, &choke_module);
  const char          *ip           = c->remote_ip;
  const char          *fwd          = ap_table_get(r->headers_in, "X-Forwarded-For");
  const char          *user         = c->user;
  const char          *server_name  = ap_get_server_name(r);
  const char          *request      = r->uri;
  int                  glb_num_ip   = 0;
  int                  glb_num_user = 0;
  int                  num_ip       = 0;
  int                  num_user     = 0;
  int                  i;
  char                 client[64];
  int                  ipset, fwdset, userset, serverset, requestset;

  /* Only handle initial requests */
  if(!ap_is_initial_req(r)) {
    return DECLINED;
  }

#ifdef CHOKE_DEBUG
  ap_log_printf(r->server, "glb_max_ip: [%d] max_ip: [%d] glb_max_user: [%d] max_user: [%d]",
      cfg->glb_max_ip, cfg->max_ip, cfg->glb_max_user, cfg->max_user);
#endif

  /* set the X-mod_choke header */
  if(cfg->glb_max_ip != UNSET   || cfg->max_ip != UNSET ||
     cfg->glb_max_user != UNSET || cfg->max_user != UNSET)
  {
    ap_table_set(r->headers_out, "X-mod-choke", CHOKE_VERSION);
  }
  
  /* Record the info about this connection in the scoreboard */
  if(ip != NULL) {
    strncpy(rec->ip, ip, sizeof(rec->ip));
  } else {
    strcpy(rec->ip, "");
  }

  if(fwd != NULL) {
    strncpy(rec->fwd, fwd, sizeof(rec->fwd));
  } else {
    strcpy(rec->fwd, "");
  }

  if(user != NULL) {
    strncpy(rec->user, user, sizeof(rec->user));
  } else {
    strcpy(rec->user, "");
  }
  
  if(server_name != NULL) {
    strncpy(rec->server_name, server_name, sizeof(rec->server_name));
  } else {
    strcpy(rec->server_name, "");
  }

  if(request != NULL) {
    strncpy(rec->request, request, sizeof(rec->request));
  } else {
    strcpy(rec->request, "");
  }

  /* do the concurrency checks */
  ipset      = (ip          != NULL && strlen(ip))          ? 1 : 0;
  fwdset     = (fwd         != NULL && strlen(fwd))         ? 1 : 0;
  userset    = (user        != NULL && strlen(user))        ? 1 : 0;
  serverset  = (server_name != NULL && strlen(server_name)) ? 1 : 0;
  requestset = (request     != NULL && strlen(request))     ? 1 : 0;

  for(i = 0; i < HARD_SERVER_LIMIT; i++) {
    choke_score_rec *orec = &csb->servers[i];
    short_score ap_rec = ap_scoreboard_image->servers[i];

    if(ap_rec.status != SERVER_DEAD && ap_rec.status != SERVER_STARTING &&
       ap_rec.status != SERVER_READY )
    {
      /* count ips */
      if(cfg->max_ip != UNSET || cfg->glb_max_ip != UNSET)
      {
        if((fwdset && strcmp(fwd, orec->fwd) == 0)
            || (!fwdset && ipset && strcmp(ip, orec->ip) == 0))
        {
          if(serverset && strcmp(server_name, orec->server_name) == 0)
          {
            num_ip++;
            glb_num_ip++;
          } else {
            glb_num_ip++;
          }
        }
      }
    
      /* count users */
      if(cfg->max_user != UNSET)
      {
        if(user != NULL && strlen(user) && strcmp(user, orec->user) == 0)
        {
          if(serverset && strcmp(server_name, orec->server_name) == 0)
          {
            num_user++;
            glb_num_user++;
          } else {
            glb_num_user++;
          }
        }
      }
    }
  }

  /* check if the ip/user is over the limit */

#ifdef CHOKE_DEBUG
  ap_log_printf(r->server,
      "glb_num_ip: [%d] num_ip: [%d] glb_num_user: [%d] num_user: [%d]",
      glb_num_ip, num_ip, glb_num_user, num_user);
#endif

  if(fwdset) {
    snprintf(client, sizeof(client), "%s (%s)", ip, fwd);
  } else {
    snprintf(client, sizeof(client), "%s", ip);
  }

  if(cfg->glb_max_ip != UNSET && glb_num_ip > cfg->glb_max_ip) {
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r->server,
        "client %s exceeded global connection limit (%d)",
        client, cfg->glb_max_ip);
    return(HTTP_SERVICE_UNAVAILABLE);
  }

  if(cfg->max_ip != UNSET && num_ip > cfg->max_ip) {
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r->server,
        "client %s exceeded connection limit (%d)",
        client, cfg->max_ip);
    return(HTTP_SERVICE_UNAVAILABLE);
  }

  if(cfg->glb_max_user != UNSET && glb_num_user > cfg->glb_max_user) {
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r->server,
        "[client %s] user %s exceeded global connection limit (%d)",
        client, user, cfg->glb_max_user);
    return(HTTP_SERVICE_UNAVAILABLE);
  }

  if(cfg->max_user != UNSET && num_user > cfg->max_user) {
    ap_log_error(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r->server,
        "[client %s] user %s exceeded connection limit (%d)",
        client, user, cfg->max_user);
    return(HTTP_SERVICE_UNAVAILABLE);
  }

  return OK;
}
#endif

static int choke_status(request_rec *r) {
  choke_server_config *cfg = (choke_server_config *)
                             ap_get_module_config(r->server->module_config, &choke_module);
  int                  i;

  r->content_type = "text/html";
  ap_send_http_header(r);

  /* output header */
  ap_rprintf(r, "<html><head>"
                "<title>mod_choke " CHOKE_VERSION " %s status</title>"
                "<style type='text/css'>"
                "  .tt             { border: thin solid black; background: #DDDDDD; }"
                "  .data_table     { border: thin solid black; }"
                "</style>"
                "</head>"
                "<body><h2>mod_choke " CHOKE_VERSION " %s status</h2>",
                ap_get_server_name(r), ap_get_server_name(r));

#ifdef HAVE_SHMGET
  ap_rputs("<table border=1 cellpadding=5 cellspacing=0 class='data_table'>"
           "<tr>"
           "<th class='tt'>Srv</th>"
           "<th class='tt'>Status</th>"
           "<th class='tt'>Host</th>"
           "<th class='tt'>Client (X-Forwarded-For)</th>"
           "<th class='tt'>User</th>"
           "<th class='tt'>Request</th>"
           "</tr>", r);

  /* output the table contents */
  for(i = 0; i < HARD_SERVER_LIMIT; i++) {
    choke_score_rec *rec    = &csb->servers[i];
    short_score      ap_rec = ap_scoreboard_image->servers[i];
    char             client[64];
    char             user[50];
    char             server_name[100];
    char             request[100];

    if(ap_rec.status == SERVER_DEAD) {
      continue;
    }
    
    if(rec->fwd != NULL && strlen(rec->fwd)) {
      snprintf(client, sizeof(client), "%s (%s)", rec->ip, rec->fwd);
    } else if(rec->ip != NULL && strlen(rec->ip)) {
      snprintf(client, sizeof(client), "%s", rec->ip);
    } else {
      snprintf(client, sizeof(client), "-");
    }

    if(rec->user != NULL && strlen(rec->user)) {
      snprintf(user, sizeof(user), "%s", rec->user);
    } else {
      snprintf(user, sizeof(user), "-");
    }

    if(rec->server_name != NULL && strlen(rec->server_name)) {
      snprintf(server_name, sizeof(server_name), "%s", rec->server_name);
    } else {
      snprintf(server_name, sizeof(server_name), "-");
    }

    if(rec->request != NULL && strlen(rec->request)) {
      snprintf(request, sizeof(request), "%s", rec->request);
    } else {
      snprintf(request, sizeof(request), "-");
    }

    ap_rprintf(r, "<tr>"
                  "<td>%d</td>"
                  "<td>%c</td>"
                  "<td>%s</td>"
                  "<td>%s</td>"
                  "<td>%s</td>"
                  "<td>%s</td>"
                  "</tr>",
                  i, status_flags[ap_rec.status],
                  server_name, client, user, request);
  }

  ap_rputs("</table><br>", r);

  ap_rputs("Scoreboard Key: <br>\n", r);
  ap_rputs("\"<B><code>_</code></B>\" Waiting for Connection, \n", r);
  ap_rputs("\"<B><code>S</code></B>\" Starting up, \n", r);
  ap_rputs("\"<B><code>R</code></B>\" Reading Request,<BR>\n", r);
  ap_rputs("\"<B><code>W</code></B>\" Sending Reply, \n", r);
  ap_rputs("\"<B><code>K</code></B>\" Keepalive (read), \n", r);
  ap_rputs("\"<B><code>D</code></B>\" DNS Lookup,<BR>\n", r);
  ap_rputs("\"<B><code>L</code></B>\" Logging, \n", r);
  ap_rputs("\"<B><code>G</code></B>\" Gracefully finishing, \n", r);
  ap_rputs("\"<B><code>.</code></B>\" Open slot with no current process<P>\n", r);
#else
  ap_rputs("SHMGET not supported, no scoreboard created.", r);
#endif

  /* output footer */
  ap_rputs("</body>"
           "</html>", r);

  return(OK);
}

/* -------------------------------------------------------------------------- *\
 * Register the module with apache                                            *
\* -------------------------------------------------------------------------- */

static const handler_rec choke_handlers[] = { 
  { "*/*",          choke_handler },
  { "choke-status", choke_status  },
  { NULL }
};

module MODULE_VAR_EXPORT choke_module = {
  STANDARD_MODULE_STUFF, 
  choke_init,                 /* module initializer                  */
  choke_create_dir_config,    /* create per-dir    config structures */
  choke_merge_dir_config,     /* merge  per-dir    config structures */
  choke_create_server_config, /* create per-server config structures */
  choke_merge_server_config,  /* merge  per-server config structures */
  choke_cmds,                 /* table of config file commands       */
  choke_handlers,             /* [#8] MIME-typed-dispatched handlers */
  NULL,                       /* [#1] URI to filename translation    */
  NULL,                       /* [#4] validate user id from request  */
  NULL,                       /* [#5] check if the user is ok _here_ */
  NULL,                       /* [#3] check access by host address   */
  NULL,                       /* [#6] determine MIME type            */
#ifdef HAVE_SHMGET
  choke_concurrency_check,    /* [#7] pre-run fixups                 */
#else
  NULL,                       /* [#7] pre-run fixups                 */
#endif
  NULL,                       /* [#9] log a transaction              */
  NULL,                       /* [#2] header parser                  */
  NULL,                       /* child_init                          */
  NULL,                       /* child_exit                          */
#ifdef HAVE_SHMGET
  choke_concurrency_check,    /* [#0] post read-request              */
#else
  NULL                        /* [#0] post read-request              */
#endif
};
