// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.IO;
using System.Collections.Generic;

using Anculus.Core;
using Anculus.Core.Extended;

using NUnit.Framework;

namespace Anculus.UnitTests
{
	[TestFixture]
	public class ConfigurationXmlTest
	{
		private IConfiguration _emptyConfig;
		private IConfiguration _config;

		public ConfigurationXmlTest ()
		{
			string emptyConfigFile = Path.Combine (Path.GetTempPath (), "empty.xml");
			string configFile = Path.Combine (Path.GetTempPath (), "config.xml");
			
			File.Delete (emptyConfigFile);
			File.Delete (configFile);
			
			_emptyConfig = new XmlConfiguration (emptyConfigFile);
			_config = new XmlConfiguration (configFile);
			
			_config.SetBool ("bool", true);
			_config.SetByte ("byte", byte.MaxValue);
			_config.SetChar ("char", 'd');
			_config.SetDateTime ("datetime", DateTime.MaxValue);
			_config.SetDecimal ("decimal", 100M);
			_config.SetDouble ("double", 100.0);
			_config.SetFloat ("float", 100f);
			_config.SetInt ("int", 100);
			_config.SetLong ("long", 100);
			_config.SetSByte ("sbyte", sbyte.MaxValue);
			_config.SetShort ("short", 100);
			_config.SetString ("string", "abc");
			_config.SetUInt ("uint", 100);
			_config.SetULong ("ulong", 100);
			_config.SetUShort ("ushort", 100);
			
			List<int> lst = new List<int> ();
			lst.Add (1);
			lst.Add (2);
			lst.Add (3);
			
			_config.SetArray<string> ("array_string", new string[] { "a", "b", "c" });
			_config.SetList<int> ("list_int", lst);
			
			Employee e = new Employee (1, "Ben", 25000);
			_config.Set<Employee> ("employee", e);
			
			_config["Section1"].SetInt ("int", 100);
			
			_config.Save ();

			_config = new XmlConfiguration (configFile);
			_config.Load ();
		}
		
		[Test]
		public void UninitializedGetByteTest ()
		{
			byte b = _emptyConfig.GetByte ("test");
			Assert.AreEqual (byte.MinValue, b);
			
			b = _emptyConfig.GetByte ("test", (byte)3);
			Assert.AreEqual ((byte)3, b);
		}
		
		[Test]
		public void UninitializedGetSByteTest ()
		{
			sbyte b = _emptyConfig.GetSByte ("test");
			Assert.AreEqual (sbyte.MinValue, b);
			
			b = _emptyConfig.GetSByte ("test", (sbyte)3);
			Assert.AreEqual ((sbyte)3, b);
		}
		
		[Test]
		public void UninitializedGetBoolTest ()
		{
			bool b = _emptyConfig.GetBool ("test");
			Assert.AreEqual (false, b);
			
			b = _emptyConfig.GetBool ("test", true);
			Assert.AreEqual (true, b);
		}
		
		[Test]
		public void UninitializedGetShortTest ()
		{
			short b = _emptyConfig.GetShort ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetShort ("test", 5);
			Assert.AreEqual ((short)5, b);
		}
		
		[Test]
		public void UninitializedGetIntTest ()
		{
			int b = _emptyConfig.GetInt ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetInt ("test", 5);
			Assert.AreEqual (5, b);
		}
		
		[Test]
		public void UninitializedGetLongTest ()
		{
			long b = _emptyConfig.GetLong ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetLong ("test", 5);
			Assert.AreEqual (5L, b);
		}
		
		[Test]
		public void UninitializedGetUShortTest ()
		{
			ushort b = _emptyConfig.GetUShort ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetUShort ("test", 5);
			Assert.AreEqual ((ushort)5, b);
		}
		
		[Test]
		public void UninitializedGetUIntTest ()
		{
			uint b = _emptyConfig.GetUInt ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetUInt ("test", 5);
			Assert.AreEqual (5U, b);
		}
		
		[Test]
		public void UninitializedGetULongTest ()
		{
			ulong b = _emptyConfig.GetULong ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetULong ("test", 5);
			Assert.AreEqual (5UL, b);
		}
		
		[Test]
		public void UninitializedGetFloatTest ()
		{
			float b = _emptyConfig.GetFloat ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetFloat ("test", 5);
			Assert.AreEqual (5f, b);
		}
		
		[Test]
		public void UninitializedGetDoubleTest ()
		{
			double b = _emptyConfig.GetDouble ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetDouble ("test", 5);
			Assert.AreEqual (5.0, b);
		}
		
		[Test]
		public void UninitializedGetStringTest ()
		{
			string b = _emptyConfig.GetString ("test");
			Assert.AreEqual (null, b);
			
			b = _emptyConfig.GetString ("test", "test");
			Assert.AreEqual ("test", b);
		}
		
		[Test]
		public void UninitializedGetCharTest ()
		{
			char b = _emptyConfig.GetChar ("test");
			Assert.AreEqual (char.MinValue, b);
			
			b = _emptyConfig.GetChar ("test", 'c');
			Assert.AreEqual ('c', b);
		}
		
		[Test]
		public void UninitializedGetDecimalTest ()
		{
			decimal b = _emptyConfig.GetDecimal ("test");
			Assert.AreEqual (0, b);
			
			b = _emptyConfig.GetDecimal ("test", 5);
			Assert.AreEqual (5M, b);
		}
		
		[Test]
		public void UninitializedGetDateTimeTest ()
		{
			DateTime b = _emptyConfig.GetDateTime ("test");
			Assert.AreEqual (DateTime.MinValue, b);
			
			DateTime now = DateTime.Now;
			b = _emptyConfig.GetDateTime ("test", now);
			Assert.AreEqual (now, b);
		}
		
		[Test]
		public void GetByteTest ()
		{
			byte b = _config.GetByte ("byte");
			Assert.AreEqual (byte.MaxValue, b);
		}
		
		[Test]
		public void GetSByteTest ()
		{
			sbyte b = _config.GetSByte ("sbyte");
			Assert.AreEqual (sbyte.MaxValue, b);
		}
		
		[Test]
		public void GetBoolTest ()
		{
			bool b = _config.GetBool ("bool");
			Assert.AreEqual (true, b);
		}
		
		[Test]
		public void GetShortTest ()
		{
			short b = _config.GetShort ("short");
			Assert.AreEqual ((short)100, b);
		}
		
		[Test]
		public void GetIntTest ()
		{
			int b = _config.GetInt ("int");
			Assert.AreEqual (100, b);
		}
		
		[Test]
		public void GetLongTest ()
		{
			long b = _config.GetLong ("long");
			Assert.AreEqual (100L, b);
		}
		
		[Test]
		public void GetUShortTest ()
		{
			ushort b = _config.GetUShort ("ushort");
			Assert.AreEqual ((ushort)100, b);
		}
		
		[Test]
		public void GetUIntTest ()
		{
			uint b = _config.GetUInt ("uint");
			Assert.AreEqual (100U, b);
		}
		
		[Test]
		public void GetULongTest ()
		{
			ulong b = _config.GetULong ("ulong");
			Assert.AreEqual (100UL, b);
		}
		
		[Test]
		public void GetFloatTest ()
		{
			float b = _config.GetFloat ("float");
			Assert.AreEqual (100f, b);
		}
		
		[Test]
		public void GetDoubleTest ()
		{
			double b = _config.GetDouble ("double");
			Assert.AreEqual (100.0, b);
		}
		
		[Test]
		public void GetStringTest ()
		{
			string b = _config.GetString ("string");
			Assert.AreEqual ("abc", b);
		}
		
		[Test]
		public void GetCharTest ()
		{
			char b = _config.GetChar ("char");
			Assert.AreEqual ('d', b);
		}
		
		[Test]
		public void GetDecimalTest ()
		{
			decimal b = _config.GetDecimal ("decimal");
			Assert.AreEqual (100M, b);
		}
		
		[Test]
		public void GetDateTimeTest ()
		{
			DateTime b = _config.GetDateTime ("datetime");
			Assert.AreEqual (DateTime.MaxValue, b);
		}
		
		[Test]
		public void GetStringArrayTest ()
		{
			string[] arr = _config.GetArray<string> ("array_string");
			Assert.AreEqual (3, arr.Length);
			Assert.AreEqual ("a", arr[0]);
			Assert.AreEqual ("b", arr[1]);
			Assert.AreEqual ("c", arr[2]);
		}
		
		[Test]
		public void GetIntegerListTest ()
		{
			IList<int> lst = _config.GetList<int> ("list_int");
			Assert.AreEqual (3, lst.Count);
			Assert.AreEqual (1, lst[0]);
			Assert.AreEqual (2, lst[1]);
			Assert.AreEqual (3, lst[2]);
		}
		
		[Test]
		public void GetGenericObjectTest ()
		{
			Employee e = _config.Get<Employee> ("employee");
			Assert.IsNotNull (e);
			Assert.AreEqual (1, e.Index);
			Assert.AreEqual ("Ben", e.Name);
			Assert.AreEqual (25000D, e.Wage);
		}
		
		[Test]
		public void SectionExistsTest ()
		{
			Assert.IsTrue (_config.ContainsSection ("Section1"));
		}
		
		[Test]
		public void SectionValueTest ()
		{
			Assert.AreEqual (100, _config["Section1"].GetInt ("int"));
		}
	}
}
