// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.Reflection;
using System.Diagnostics;
using System.Collections.Generic;

namespace Anculus.Core
{
	public static class Log
	{
		public static event LogEventHandler MessageLogged;

		private static List<ILogBackend> _backends;
		private static LogLevel _minimalLevel;
		private static object _sync = new object ();
		
		static Log ()
		{
			_backends = new List<ILogBackend> ();
			_minimalLevel = LogLevel.Debug;
		}
		
		public static void AddBackend (ILogBackend backend)
		{
			if (backend == null)
				throw new ArgumentNullException ("backend");

			lock (_sync) {
				if (!_backends.Contains (backend))
					_backends.Add (backend);
			}
		}
		
		public static void RemoveBackend (ILogBackend backend)
		{
			if (backend == null)
				throw new ArgumentNullException ("backend");

			lock (_sync) {
				_backends.Remove (backend);
			}
		}
		
		public static void RemoveBackend (Type type)
		{
			if (type == null)
				throw new ArgumentNullException ("type");
			
			lock (_sync) {
				int index = -1;
				for (int i=0; i < _backends.Count; i++) {
					if (_backends[i].GetType () == type) {
						index = i;
						break;
					}
				}
	
				if (index >= 0)
					_backends.RemoveAt (index);
			}
		}

		public static LogLevel MinimalLogLevel
		{
			get { return _minimalLevel; }
			set { _minimalLevel = value; }
		}

		public static void Debug (string message, params object[] args)
		{
			WriteLog (LogLevel.Debug, message, args, null);
		}

		public static void Info (string message, params object[] args)
		{
			WriteLog (LogLevel.Info, message, args, null);
		}

		public static void Warn (string message, params object[] args)
		{
			WriteLog (LogLevel.Warning, message, args, null);
		}

		public static void Warn (Exception ex, string message, params object[] args)
		{
			WriteLog (LogLevel.Warning, message, args, ex);
		}

		public static void Error (string message, params object[] args)
		{
			WriteLog (LogLevel.Error, message, args, null);
		}

		public static void Error (Exception ex, string message, params object[] args)
		{
			WriteLog (LogLevel.Error, message, args, ex);
		}

		public static void Fatal (string message, params object[] args)
		{
			WriteLog (LogLevel.Fatal, message, args, null);
		}

		public static void Fatal (Exception ex, string message, params object[] args)
		{
			WriteLog (LogLevel.Fatal, message, args, ex);
		}
		
		[System.Diagnostics.DebuggerHidden]
		private static void WriteLog (LogLevel level, string message, object[] args, Exception ex)
		{
			if ((int)level < (int)_minimalLevel)
				return;
			
			if (_backends.Count == 0)
				_backends.Add (new ConsoleLogBackend ());
			
			string msg = null;
			if (args == null || args.Length == 0)
				msg = message == null ? String.Empty : message;
			else
				msg = message == null ? String.Empty : String.Format (message, args);

			string logger = GetCaller (2);
			string timestamp = DateTime.Now.ToString ("yyyy/MM/dd HH:mm:ss");
			
			lock (_sync) {
				foreach (ILogBackend backend in _backends)
					backend.Log (timestamp, level, logger, msg, ex);
			}
			
			if (MessageLogged != null) {
				LogEventArgs evnt = new LogEventArgs (level, logger, msg, ex);
				MessageLogged (null, evnt);
			}
		}
		
		private static string GetCaller (int calllevel)
		{
			StackTrace st = new StackTrace(false);
			StackFrame sf = st.GetFrame(calllevel + 1);
			
			MethodBase method = sf.GetMethod();
			
			return string.Concat (method.DeclaringType.Name, ".", method.Name);
		}
	}
}
